package JavaGroups;

import java.util.*;


/**
   Allows to register commands to be executed at a certain time, or in n milliseconds from now.
   The command will be executed in a separate thread.
 */
public class Timer implements Runnable {

    class CommandThread extends Thread {
	Command  command=null;

	CommandThread(Command c) {command=c;}

	public void run() {command.Execute();}
    }

    class Entry {
	long     when=0;
	Command  command=null;

	Entry(long when, Command command) {
	    this.when=when;
	    this.command=command;
	}
    }

    protected Vector    entries=new Vector();
    protected Thread    timer_thread=null;


    /** Constructor. Starts the timer thread */
    public Timer() {
	Start();
    }

    public void finalize() {
	Stop();
    }
   


    private void AddSorted(Entry e) {
	boolean added=false;
	Entry   tmp;

	for(int i=0; i < entries.size(); i++) {
	    tmp=(Entry)entries.elementAt(i);
	    if(e.when < tmp.when) {
		entries.insertElementAt(e, i);  // add before current item
		added=true;
		return;
	    }
	}
	if(!added)
	    entries.addElement(e);
    }

    /** Execute on a separate thread */
    private void Execute(Command c) {
	new CommandThread(c).start();
    }


    
    /**
       Call <code>command.Execute</code> in <code>time</code> milliseconds. A command can be
       registered multiple times, and will be called multiple times. Each command will be executed
       on a separate thread.
     */
    public void Trigger(Command command, long time) {
	synchronized(entries) {
	    AddSorted(new Entry(System.currentTimeMillis() + time, command));
	    entries.notify();  // notify the timer thread
	}
    }


    public void Start() {
	if(timer_thread == null) {
	    timer_thread=new Thread(this, "TimerThread");
	    timer_thread.start();
	}
    }

    public void Stop() {
	if(timer_thread != null) {
	    timer_thread.stop();
	    timer_thread=null;
	}
    }
    

    /**
       Call <code>command.Execute</code> at time <code>time</code>. A command can be
       registered multiple times, and will be called multiple times. Each command will be executed
       on a separate thread. If the date is in the past, the command will be executed immediately.
     */
    public void Trigger(Command command, Date time) {
	synchronized(entries) {
	    AddSorted(new Entry(time.getTime(), command));
	    entries.notify();  // notify the timer thread
	}
    }


    public void run() {
	Entry  current;
	long   wait_time=0, current_time=0;

	while(true) {
	    synchronized(entries) {
		while(entries.size() < 1) {
		    try {
			entries.wait();  // will be notified when a new entry is added or removed
		    }
		    catch(Exception e) {}
		}
		while(entries.size() > 0) {
		    current=(Entry)entries.elementAt(0);
		    current_time=System.currentTimeMillis();
		    if(current.when <= current_time) {
			Execute(current.command); // in a separate thread
			entries.removeElementAt(0);
		    }
		    else {
			wait_time=current.when - current_time;  // is always positive
			try {
			    entries.wait(wait_time);            // awakened by addition of new entry
			}
			catch(Exception e) {}
		    }
		}
	    }
	}
    }




    public static void main(String[] args) {
	
	class MyCommand implements Command {
	    int no=0;

	    MyCommand(int no) {this.no=no;}

	    public boolean Execute() {
		System.out.println("Command #" + no + " was executed at " + new Date());
		return true;
	    }
	}


	Timer t=new Timer();
	
	t.Trigger(new MyCommand(7), 10000);
	t.Trigger(new MyCommand(1), 1000);
	t.Trigger(new MyCommand(2), 4000);
	t.Trigger(new MyCommand(3), 5000);
	t.Trigger(new MyCommand(4), 10000);
	t.Trigger(new MyCommand(4), 10000);
	t.Trigger(new MyCommand(4), 10000);
	t.Trigger(new MyCommand(4), 10000);
	t.Trigger(new MyCommand(5), 2000);
	t.Trigger(new MyCommand(6), new Date(System.currentTimeMillis() + 1000));

	try {
	    Thread.currentThread().sleep(13000);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
	t.Stop();
    }


}
