package JavaGroups;

import java.util.*;

/**
 * Relates responses with requests. Used for synchronous calls etc. Later, I plan to add
 * support for user-defined policy objects which defines how multiple response are related
 * to a request, e.g. first only, majority, merged response etc. Currently, only the first
 * matching response is returned.
 */

public class MessageCorrelator {
    private Hashtable msgs=new Hashtable();  // msg_id as keys and MessageCorrelator as values


    public void AddRequest(Message m) {
	synchronized(msgs) {
	    msgs.put(new Long(m.GetId()), new Queue());
	}
    }


    /**
     * Adds a response for a corresponding message id to the correlation table. 
     * If corresponding request message id is not found, false is returned, otherwise, true.
     */
    public boolean AddResponse(Message m) throws Exception {
	long       id=m.GetRspId();
	Queue      mq=null;
	if(id <= 0) {
	    System.err.println("MessageCorrelator.AddResponse(): message id " + id + 
			       " is not valid");
	    return false;
	}
	synchronized(msgs) {
	    mq=(Queue)msgs.get(new Long(id));
	    if(mq == null)  // not present at all, or might already have been removed
		return false;
	}
	mq.Add(m);
	return true;
    }


    public Message GetResponse(long req_id, long timeout) throws Exception {
	Message    retval=null;
	Queue      mq=null;

	if(req_id <= 0) {
	    System.err.println("MessageCorrelator.GetResponse(): message id " + req_id + 
			       " not found in correlation table");
	    return null;
	}
	synchronized(msgs) {
	    mq=(Queue)msgs.get(new Long(req_id));
	    if(mq == null) {
		System.err.println("MessageCorrelator.GetResponse(): " + 
				   "no request found with id=" + req_id);
		return null;
	    }
	}

	if(timeout <= 0)
	    retval=(Message)mq.Remove();
	else
	    retval=(Message)mq.Remove(timeout);
	mq.Close();

	    
	synchronized(msgs) {
	    msgs.remove(new Long(req_id));
	}

	return retval;
    }


    
    /**
     * Returns all reponses for a request. A partially filled vector will be returned
     * if a timeout occurs and not all reponses have been collected
     */
    public Vector GetResponses(long req_id, int expected_responses, long timeout) {
	Vector   retval=new Vector();
	Message  msg;
	long     start_time=System.currentTimeMillis();
	long     time_spent=0;
	int      num_responses=0;
	Queue    mq;

	if(req_id <= 0) {
	    System.err.println("MessageCorrelator.GetResponses(): message id " + req_id + 
			       " not found in correlation table");
	    return null;
	}

	synchronized(msgs) {
	    mq=(Queue)msgs.get(new Long(req_id));
	    if(mq == null) {
		System.err.println("MessageCorrelator.GetResponses(): " + 
				   "no request found with id=" + req_id);
		return null;
	    }
	}

	while(num_responses < expected_responses && time_spent < timeout) {
	    try {
		msg=(Message)mq.Remove(timeout - time_spent);
	    }
	    catch(Timeout tex) {
		msg=null;	    }
	    catch(Exception e) {
		break;
	    }
	    if(msg != null) {
		retval.addElement(msg);
		num_responses++;
	    }
	    time_spent=System.currentTimeMillis() - start_time;
	}

	synchronized(msgs) {
	    mq=(Queue)msgs.get(new Long(req_id));
	    if(mq != null) {
		try {mq.Close();}
		catch(Exception e) {
		    System.err.println(e);
		}
	    }
	    msgs.remove(new Long(req_id));
	}
	return retval;
    }


    public void Stop() {
	Queue q;
	synchronized(msgs) {
	    // Release all queues (that releases callers blocked on the queues)
	    for(Enumeration queues=msgs.elements(); queues.hasMoreElements();) {
		q=(Queue)queues.nextElement();
		q.Close();
	    }
	}
    }


    public String toString() {
	StringBuffer ret=new StringBuffer();
	ret.append(msgs);
	return ret.toString();
    }




}
