package JavaGroups;


import java.util.*;

/**
 * Allows to execute methods without the caller having to wait for the method return. Only
 * one-way calls are allowed (return values will be discarded). A <code>MethodCall</code> and
 * a target object are pushed onto a queue from which the LazyEvaluator removes entries on a 
 * first-come first-served basis and evaluates the method calls. <p>Method lookup can be 
 * tailored by supplying a method lookup class at creation time.
 */
public class LazyEvaluator implements Runnable {
    protected Queue         queue=new Queue();
    protected Thread        mythread=null;
    protected MethodLookup  method_lookup=null;


    protected class QueueItem {
	private Object     target;
	private MethodCall call;

	QueueItem(Object target, MethodCall call) {
	    this.target=target;
	    this.call=call;
	}
    }


    public LazyEvaluator() {
	Start();
    }



    public LazyEvaluator(MethodLookup lookup) {
	method_lookup=lookup;
	Start();
    }


    public void Start() {
	if(mythread == null) {
	    mythread=new Thread(this, "LazyEvaluator");
	    mythread.start();
	}
    }

    public void Stop() {
	if(mythread != null) {
	    mythread.stop();
	    mythread=null;
	}
    }

    public void Add(Object target, MethodCall call) {
	try {
	    queue.Add(new QueueItem(target, call));
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }


    public void AddMultiple(Vector targets, MethodCall call) {
	for(int i=0; i < targets.size(); i++)
	    Add(targets.elementAt(i), call);
    }


    public void run() {
	Object      target;
	MethodCall  call;
	QueueItem   item;

	while(true) {
	    try {
		item=(QueueItem)queue.Remove();		

		target=item.target;
		call=item.call;
		if(call == null || target == null) {
		    System.err.println("LazyEvaluator.run(): cannot invoke method as either " +
				       "target object or method call object is null");
		    continue;
		}
		call.Invoke(target, method_lookup);  // not interested in the return value
	    }
	    catch(QueueClosed closed) {
		System.out.println("LazyEvaluator.run(): queue has been closed");
		break;
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	}
    }
    
}
