package JavaGroups.JavaStack.Protocols;

import java.io.*;
import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;





/**
   Implementation of a PT2PT protocol. When a message is sent to a unicast destination 
   (msg.dest == unicast dest instead of null (multicast dest)), if there is not yet a connection
   to that destination, a connection is established (similar to TCP's connection establishment) and the
   message sent. ACKs are used to retransmit lost messages, and all messages are delivered in FIFO
   order. Sliding windows are used at the sending and receiving side (with fixed window size, without 
   window size adaptation).
 */

public class PT2PT extends Protocol {
    boolean     trace=false;
    Hashtable   connections=new Hashtable();

    
    class ConnectionEntry {
	Pt2ptFsm  down_fsm=null;
	Pt2ptFsm  up_fsm=null;

	ConnectionEntry(Protocol p) {
	    down_fsm=new Pt2ptFsm(p);
	    up_fsm=new Pt2ptFsm(p);
	}
	
	void Destroy() {
	    if(down_fsm != null) {
		down_fsm.Destroy();
		down_fsm=null;
	    }
	    if(up_fsm != null) {
		up_fsm.Destroy();
		up_fsm=null;
	    }
	}
    }


    public String  GetName() {return "PT2PT";}


    public void SetProperties(Properties props) {
	String     str;

	this.props=props;
	str=props.getProperty("trace");
	if(str != null)
	    trace=new Boolean(str).booleanValue();
    }


    /** Just remove if you don't need to reset any state */
    public void Reset() {
	ConnectionEntry val;
	synchronized(connections) {
	    for(Enumeration e=connections.keys(); e.hasMoreElements();) {
		val=(ConnectionEntry)connections.get(e.nextElement());
		if(val != null) val.Destroy();
	    }
	}
	connections.clear();
    }





    public void Up(Event evt) {
	Message          msg;
	Object           src;
	ConnectionEntry  conn_entry;
	Pt2ptHeader      hdr;

	if(evt.GetType() == Event.MSG) {
	    msg=(Message)evt.GetArg();
	    hdr=(Pt2ptHeader)msg.RemoveHeader();

	    if(hdr == null) {
		System.err.println("PT2PT.Up(): header is null !");
		PassUp(evt);
		return;
	    }

	    if(hdr.handle == false) {
		PassUp(evt);
		return;
	    }

	    src=msg.GetSrc();

	    conn_entry=(ConnectionEntry)connections.get(src);
	    if(conn_entry == null) {                   // new connection
		conn_entry=new ConnectionEntry(this);
		connections.put(src, conn_entry);
	    }
	    

	    switch(hdr.type) {
	    case Pt2ptHeader.SYN:
	    case Pt2ptHeader.ACK:
	    case Pt2ptHeader.DATA:
		conn_entry.up_fsm.Up(hdr, msg);
		break;
	    case Pt2ptHeader.ACK_SYN:
	    case Pt2ptHeader.DATA_ACK:
		conn_entry.down_fsm.Up(hdr, msg);
		break;
	    default:
		break;
	    }
	    return;
	}

	PassUp(evt); // Pass up to the layer above us
    }






    public void Down(Event evt) {
	Message          msg;
	Object           dst;
	ConnectionEntry  conn_entry;
	Pt2ptHeader      hdr;


	if(evt.GetType() == Event.STOP) {
	    Reset();
	    PassDown(evt);
	    return;
	}

	if(evt.GetType() == Event.MSG) {
	    msg=(Message)evt.GetArg();
	    dst=msg.GetDest();

	    if(dst == null) {   // mcast msg
		PassDown(evt);
		return;
	    }

	    /* Send pt2pt msg */
	    conn_entry=(ConnectionEntry)connections.get(dst);
	    if(conn_entry == null) {                   // new connection
		conn_entry=new ConnectionEntry(this);
		connections.put(dst, conn_entry);
	    }		    
	    conn_entry.down_fsm.Down(msg);	   
	    return;
	}

	PassDown(evt);  // Pass on to the layer below us
    }


}
