package JavaGroups.JavaStack.Protocols;

import java.io.*;
import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;




class PingHeader implements Serializable {
    public static final int GET_MBRS_REQ=1;   // arg = null
    public static final int GET_MBRS_RSP=2;   // arg = PingRsp(local_addr, coord_addr)
    public static final int HELLO=3;          // arg = null

    public int    type=0;
    public Object arg=null;

    public PingHeader(int type, Object arg) {
	this.type=type; this.arg=arg;
    }

    public String toString() {
	return "[PING: type=" + Type2Str(type) + ", arg=" + arg + "]";
    }

    String Type2Str(int t) {
	switch(t) {
	case GET_MBRS_REQ: return "GET_MBRS_REQ";
	case GET_MBRS_RSP: return "GET_MBRS_RSP";
	case HELLO:        return "HELLO";
	default:           return "<unkown type (" + t + ")>";
	}
    }
}




/**
   The PING protocol layer serves 2 tasks: it retrieves the initial membership (used by
   the GMS when started) and it periodically mcasts a HELLO message. When the coordinator
   receives a HELLO message from a process that is currently not member, it will join that
   process to the group.
*/

public class PING extends Protocol implements Runnable {
    boolean         trace=false;
    Vector          members=new Vector(), initial_members=new Vector();
    Address         local_addr=null;
    String          group_addr=null;
    String          groupname=null;
    Thread          pinger=null;
    long            timeout=2000, hello_wait=5000;
    long            num_initial_members=2;
    boolean         gossiping=false;
    String          gossip_host=null;
    int             gossip_port=0;
    GossipClient    gossip=null;



    public String  GetName() {return "PING";}


    public void SetProperties(Properties props) {
	String     str;

	this.props=props;
	str=props.getProperty("trace");
	if(str != null)
	    trace=new Boolean(str).booleanValue();

	str=props.getProperty("timeout");   // max time to wait for initial members
	if(str != null)
	    timeout=new Long(str).longValue();

	str=props.getProperty("num_initial_members");  // wait for at most n members
	if(str != null)
	    num_initial_members=new Integer(str).intValue();

	str=props.getProperty("gossip_host");
	if(str != null)
	    gossip_host=new String(str);

	str=props.getProperty("gossip_port");
	if(str != null)
	    gossip_port=new Integer(str).intValue();

	if(gossip_host != null && gossip_port != 0)
	    gossiping=true;

    }





    public void Up(Event evt) {
	Message      msg, rsp_msg;
	Object       obj;
	PingHeader   hdr, rsp_hdr;
	PingRsp      rsp;
	Object       coord;
	Address      sender;
	boolean      contains;
	Vector       tmp;


	switch(evt.GetType()) {

	case Event.MSG:
	    msg=(Message)evt.GetArg();
	    obj=msg.PeekHeader();
	    	    if(obj == null || !(obj instanceof PingHeader)) {
		PassUp(evt);
		return;
	    }
	    hdr=(PingHeader)msg.RemoveHeader();
	    
	    switch(hdr.type) {

	    case PingHeader.GET_MBRS_REQ:   // return Rsp(local_addr, coord)
		synchronized(members) {
		    coord=members.size() > 0 ? members.firstElement() : local_addr;
		}
		rsp_msg=new Message(msg.GetSrc(), null, null);
		rsp_hdr=new PingHeader(PingHeader.GET_MBRS_RSP, new PingRsp(local_addr, coord));
		rsp_msg.AddHeader(rsp_hdr);
		PassDown(new Event(Event.MSG, rsp_msg));
		return;

	    case PingHeader.GET_MBRS_RSP:   // add response to vector and notify waiting thread
		rsp=(PingRsp)hdr.arg;		
		synchronized(initial_members) {
		    initial_members.addElement(rsp);
		    initial_members.notify();
		}
		return;

	    case PingHeader.HELLO:          // if coord: handle, else: discard
		sender=(Address)msg.GetSrc();
		if(sender != null) {
		    synchronized(members) {
			contains=members.contains(sender);
		    }
		    if(!contains) {
			//System.out.println("PING.Up(): membership " + members + " does not contain " +
			// sender + "; merging it");
			tmp=new Vector();
			tmp.addElement(sender);
			PassUp(new Event(Event.MERGE, tmp));
		    }
		}
		return;

	    default:
		System.err.println("PING.Up(): got PING header with unknown type (" + hdr.type + ")");
		return;
	    }
	    

	case Event.SET_LOCAL_ADDRESS:
	    local_addr=(Address)evt.GetArg();
	    PassUp(evt);
	    break;

	default:
	    PassUp(evt);            // Pass up to the layer above us
	    break;
	}
    }





    public void Down(Event evt) {
	Message      msg;
	PingHeader   hdr;
	long         time_to_wait, start_time;
	Vector       gossip_rsps=new Vector();

	switch(evt.GetType()) {

	case Event.FIND_INITIAL_MBRS:   // sent by GMS layer, pass up a GET_MBRS_OK event

	    if(gossiping) {
		if(gossip != null)
		    gossip_rsps=gossip.Get(group_addr);
		else {
		    System.err.println("PING.Down(): gossip is null !");
		    PassUp(new Event(Event.FIND_INITIAL_MBRS_OK, null));
		    return;
		}

		if(gossip_rsps != null && gossip_rsps.size() > 0) {
		    // Set a temporary membership in the UDP layer, so that the following multicast
		    // will be sent to all of them
		    Event view_event=new Event(Event.VIEW_CHANGE, MakeView(gossip_rsps));
		    PassDown(view_event); // needed e.g. by failure detector or UDP	
		}
		else {
		    PassUp(new Event(Event.FIND_INITIAL_MBRS_OK, null));
		    return;
		}
	    }
	    	    

	    // 1. Mcast GET_MBRS_REQ message

	    hdr=new PingHeader(PingHeader.GET_MBRS_REQ, null);
	    msg=new Message(null, null, null);
	    msg.AddHeader(hdr);
	    PassDown(new Event(Event.MSG, msg));
	    

	    // 2. Wait 'timeout' ms or until 'num_initial_members' have been retrieved
	    synchronized(initial_members) {
		initial_members.removeAllElements();
		start_time=System.currentTimeMillis();
		time_to_wait=timeout;
		
		while(initial_members.size() < num_initial_members && time_to_wait > 0) {
		    try {initial_members.wait(time_to_wait);} catch(Exception e) {}
		    time_to_wait-=System.currentTimeMillis() - start_time;
		}
	    }
	    
	    // 3. Send response
	    PassUp(new Event(Event.FIND_INITIAL_MBRS_OK, initial_members));
	    break;

	case Event.STOP:
	    PassDown(evt);
	    if(gossiping)
		StopGossip();
	    break;

	case Event.VIEW_CHANGE:	    
	    Vector tmp;
	    if((tmp=(Vector)((View)evt.GetArg()).GetMembers()) != null) {
		synchronized(members) {
		    members.removeAllElements();
		    for(int i=0; i < tmp.size(); i++)
			members.addElement(tmp.elementAt(i));
		}
	    }
	    PassDown(evt);
	    break;

	case Event.CONNECT:
	    group_addr=(String)evt.GetArg();
	    PassDown(evt);
	    if(gossiping)
		StartGossip();
	    Start();
	    break;

	case Event.DISCONNECT:
	    if(local_addr != null && evt.GetArg() != null && local_addr.equals(evt.GetArg()))
		Stop();
	    if(gossiping)
		StopGossip();
	    PassDown(evt);
	    break;
	    
	default:
	    PassDown(evt);          // Pass on to the layer below us
	    break;
	}
    }




    /**
       If IP multicast: periodically mcast a HELLO message
       If gossiping: periodically retrieve the membership. Any members not part of our 
       own membership are merged (passing MERGE event up).
     */
    public void run() {
	Message      hello_msg;
	PingHeader   hdr;
	Vector       rsps, members_to_merge=new Vector();
	Object       mbr;


	try {Thread.currentThread().sleep(3000);}
	catch(Exception e) {}
	    

	while(true) {
	    Util.Sleep(hello_wait);

	    if(!gossiping) {                              // plain IP MCAST
		hello_msg=new Message(null, null, null);
		hdr=new PingHeader(PingHeader.HELLO, null);
		hello_msg.AddHeader(hdr);
		PassDown(new Event(Event.MSG, hello_msg));
	    }
	    else {                                       // gossiping; contact Gossip server
		if(gossip != null)
		    rsps=gossip.Get(group_addr);
		else {
		    System.err.println("PING.run(): gossip server could not be contacted !");
		    continue;
		}

		synchronized(members) {
		    members_to_merge.removeAllElements();
		    
		    for(int i=0; i < rsps.size(); i++) {
			mbr=rsps.elementAt(i);
			if(!members.contains(mbr)) {
			    System.out.println("PING.run(): membership " + members + 
					       " does not contain " + mbr + "; merging it");
			    members_to_merge.addElement(mbr);
			}
		    }

		    if(members_to_merge.size() > 0)
			PassUp(new Event(Event.MERGE, members_to_merge));
		}
		
		
	    }
	}
    }






    /* -------------------------- Private methods ---------------------------- */

    void Start() {
	if(pinger == null) {
	    pinger=new Thread(this, "PING-PingerThread");
	    pinger.start();
	}
    }

    
    void Stop() {
	if(pinger != null) {
	    pinger.stop();
	    pinger=null;
	}
    }


    private void StartGossip() {
	if(gossip == null) {
	    gossip=new GossipClient((String)group_addr, gossip_host, gossip_port);
	    if(local_addr != null)
		gossip.Register(local_addr);
	    else
		System.err.println("PING.StartGossip(): starting gossip client, but local " +
				   "address is null !");
	    gossip.Start();
	}
    }



    private void StopGossip() {
	if(gossip != null) {
	    gossip.Stop();
	    gossip=null;
	}
    }



    private View MakeView(Vector mbrs) {
	Object  coord=null;
	long    id=0;
	ViewId  view_id=new ViewId(local_addr);

	coord=view_id.GetCoordAddress();
	id=view_id.GetId();

	return new View(coord, id, mbrs);
    }




}
