package JavaGroups.JavaStack.Protocols;


import java.io.Serializable;
import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;



class GmsHeader implements Serializable {
    public ViewId   view_id=null;
    public boolean  handle=false;

    public GmsHeader(ViewId v, boolean h) {
	view_id=v;
	handle=h;
    }

    public String toString() {
	return "[GMS: view_id=" + view_id + ", handle=" + handle + "]";
    }
}



/**
   <b>G</b>roup <b>M</b>embership <b>S</b>ervice. The GMS finds the initial members, and then
   manages CONNECT, DISCONNECT, VIEW_CHANGE and SUSPECT events, i.e. it manages the group membership,
   joining new members, and broadcasting membership changes to all members. Only one of all GMS layers
   is the <em>coordinator</em> and therefore in charge of broadcasting membership changes. When the
   coordinator itself crashes, the remaining members elect a new coordinator to take over. The
   coordinator is always the <em>first</em> member of the membership list (which is an ordered
   sequence of member addresses, and is the same in all members).
   <p>
   <b>Requires PING layer to retrieve initial membership (via IP MCAST or GOSSIP)</b>
 */
public class GMS extends Protocol implements Transport {
    public  final int       type=Conf.GMS_MSG;
    private Properties      props=null;
    private GmsImpl         gms_impl=null;
    private MethodInvoker   method_invoker=new MethodInvoker(this, false);
    private Queue           rsp_queue=new Queue();
    private Address         local_addr=null;
    private String          group_addr=null;
    private Vector          suspected_members=new Vector();
    private Vector          initial_mbrs=new Vector();
    private long            initial_mbrs_timeout=5000;

    public ProtocolStack    GetProtocolStack()    {return stack;}
    public Address          GetLocalAddress()     {return local_addr;}
    public String           GetGroupAddress()     {return group_addr;}
    public String           GetName()             {return "GMS";}





    public GMS() {
	method_invoker.SetMethodLookup(new MethodLookupClos());
	gms_impl=new GmsImpl(this);
	method_invoker.AddTargetObject(gms_impl);
    }





    /*-------------------------- Interface Transport -------------------------------*/

    /** 
	Used e.g. by MethodInvoker to return a response. Tag the message with our type, 
	this result in our the message being caught by the corresponding GMS layer in 
	the other protocol stack(s).
	The assumption here is that all messages sent using the <code>Send</code> method 
	will always only be sent to other GMSs ! If this is not the case, then the 
	<code>Down</code> method should be used, with the 
	<code>type</code> field set accordingly 
    */
    public void Send(Message msg) throws Exception {
	Address    dest=(Address)msg.GetDest();
	GmsHeader  hdr=new GmsHeader(null, true);
	msg.AddHeader(hdr);	
	Down(new Event(Event.MSG, msg));
    }



    
    /** Remove a message from the rsp_queue */
    public Object Receive(long timeout) throws Exception {
	try {
	    return rsp_queue.Remove(timeout);
	}
	catch(Timeout tex) {
	    throw tex;
	}
	catch(QueueClosed closed) {
	    return null;
	}
	catch(Exception e) {
	    System.err.println("GMS.Receive(): " + e);
	}
	return null;
    }

    /*----------------------------------------------------------------------------------*/







    /*------------------------------- Interface Protocol -------------------------------*/
    /**
     * In case of a request, forward the message to the method invoker. In case of a response,
     * put it on the response queue, to be retrieved by later Receive() calls.
     */
    public void Up(Event evt) {
	int        msg_type;
	Message    msg;
	GmsHeader  hdr;

	if(evt.GetType() != Event.MSG) {
	    HandleUpEvent(evt);
	    return;
	}
	
	msg=(Message)evt.GetArg();

	hdr=(GmsHeader)msg.RemoveHeader();
	// Do something with the view_id


	try {	    
	    if(hdr.handle == true) {
		if(msg.IsResponse())
		    rsp_queue.Add(msg);
		else
		    method_invoker.Receive(msg);
		return;
	    }
	    PassUp(evt);
	}
	catch(Exception e) {
	    System.err.println(e);
	}	
    }


	
    private void HandleUpEvent(Event evt) {
	Vector tmp;

	switch(evt.GetType()) {

	case Event.SUSPECT:

	    // System.out.println("GMS.Up(): SUSPECT(" + evt.GetArg() +")");

	    if(local_addr.equals(evt.GetArg())) {
		// System.out.println("I'm suspected, ignoring !");
		PassUp(evt);
		return;
	    }




// 	    Address coord=gms_impl.DetermineCoordinator();
// 	    if(coord != null && coord.equals(evt.GetArg())) {  // coord itself is suspected !

// 		System.out.println("Coord itself is suspected ! Casting merge");

// 		Vector suspects=new Vector();
// 		suspects.addElement(coord);
// 		try {
// 		    gms_impl.CastMerge(null, suspects);
// 		}
// 		catch(Exception e) {}
// 		PassUp(evt);
// 		return;
// 	    }



	    Address coord=gms_impl.DetermineCoordinator();
	    if(coord != null && coord.equals(evt.GetArg())) {  // coord itself is suspected !

		System.out.println("Coord itself is suspected ! Calling Merge()");

		Vector suspects=new Vector();
		suspects.addElement(coord);
		try {
		    gms_impl.Merge(null, suspects, gms_impl.GetLtime());
		}
		catch(Exception e) {}
		PassUp(evt);
		return;
	    }






	    if(gms_impl.IsCoordinator()) {
		try {
		    gms_impl.CastSuspect(evt.GetArg());
		}
		catch(Exception e) {}
	    }



	    PassUp(evt);
	    break;
	    
	case Event.DISCONNECT:                           // will be received before FLUSH
	    gms_impl.StartLeave((Address)evt.GetArg());
	    PassUp(new Event(Event.DISCONNECT_OK));
	    break;                                       // don't pass up

	case Event.SET_LOCAL_ADDRESS:
	    local_addr=(Address)evt.GetArg();
	    if(gms_impl != null)
		gms_impl.SetLocalAddress(local_addr);
	    else
		System.err.println("GMS.Up(): local_addr could not be set in gms_impl !");
	    PassUp(evt);
	    break;

	case Event.FIND_INITIAL_MBRS_OK:
	    tmp=(Vector)evt.GetArg();

	    synchronized(initial_mbrs) {
		if(tmp != null && tmp.size() > 0)
		    for(int i=0; i < tmp.size(); i++)
			initial_mbrs.addElement(tmp.elementAt(i));
		initial_mbrs.notify();
	    }
	    break;

	case Event.MERGE:
	    if(gms_impl.IsCoordinator()) {
		tmp=(Vector)evt.GetArg();
		if(tmp != null && tmp.size() > 0) {
		    try {
			gms_impl.CastMerge(tmp, null);
		    }
		    catch(Exception e) {}
		}
	    }
	    break;

	default:
	    PassUp(evt);
	}	
    }



    public void Down(Event evt) {	
	Message  msg;
	Object   hdr;

	if(evt.GetType() != Event.MSG) {
	    HandleDownEvent(evt);
	    return;
	}

	/* If a GmsHeader has already been added (by 'Send') -> update it with the view id.
	   Otherwise, add a new GmsHeader with the view id */

	msg=(Message)evt.GetArg();
	hdr=msg.PeekHeader();
	if(hdr != null && hdr instanceof GmsHeader)
	    ((GmsHeader)hdr).view_id=gms_impl.GetViewId();
	else
	    msg.AddHeader(new GmsHeader(gms_impl.GetViewId(), false));
	PassDown(evt);
    }


	
    private void HandleDownEvent(Event evt) {
	switch(evt.GetType()) {	    

	case Event.CONNECT:
	    try {
		group_addr=(String)evt.GetArg();
	    }
	    catch(ClassCastException cce) {
		System.err.println("GMS.HandleDownEvent(CONNECT): group address must " +
				   "be a string (group name) for GMS to make sense");
		return;
	    }
	    PassDown(evt);
	    
	    /* Retrieve initial_members (Vector of PingRsps) by sending down event
	       FIND_INITIAL_MBRS and waiting for FIND_INITIAL_MBRS_OK or timeout.
	       The event is handled by the PING layer (which has to be present).
	    */
	    synchronized(initial_mbrs) {
		initial_mbrs.removeAllElements();
		PassDown(new Event(Event.FIND_INITIAL_MBRS));

		try {
		    initial_mbrs.wait(initial_mbrs_timeout);
		}
		catch(Exception e) {}

		// System.out.println("GMS: initial_mbrs are " + initial_mbrs);
	    }
	    gms_impl.StartJoin(initial_mbrs);
	    PassUp(new Event(Event.CONNECT_OK));
	    break;

	case Event.SUSPECT:                        // todo:  treat this differently in the future !
	    if(!local_addr.equals(evt.GetArg())) {
		gms_impl.StartLeave((Address)evt.GetArg());
		break;
	    }

	case Event.DISCONNECT:                            // will be received before FLUSH
	    gms_impl.StartLeave((Address)evt.GetArg());
	    PassDown(evt);
	    PassUp(new Event(Event.DISCONNECT_OK));
	    break;

	case Event.FLUSH:   // will be received after DISCONNECT
	    PassDown(evt);
	    break;

	default:
	    PassDown(evt);
	}
	
    }




 /** Setup the Protocol instance acording to the configuration string */
    public void SetProperties(Properties props) {
	String     str;
	this.props=props;

 	str=props.getProperty("initial_mbrs_timeout");   // time to wait for initial member retrieval
 	if(str != null)
 	    initial_mbrs_timeout=new Long(str).longValue();
    }



    public String toString() {return "Protocol GMS";}

    /*----------------------------------------------------------------------------------*/


}
