package JavaGroups.JavaStack.Protocols;

import java.io.*;
import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;



class FragHeader implements Serializable {
    public long    msg_id=0;
    public int     frag_id=0;
    public int     num_frags=0;

    public FragHeader(long msg_id, int frag_id, int num_frags) {
	this.msg_id=msg_id;
	this.frag_id=frag_id;
	this.num_frags=num_frags;
    }

    public String toString() {
	return "[FRAG: msg_id=" + msg_id + ", frag_id=" + frag_id + ", num_frags=" + num_frags + "]";
    }
}

/**
   Keeps track of the fragments that are received. Reassembles fragements into entire
   messages when all fragments have been received.
 */
class FragmentationTable {

    class Entry {
	int       tot_frags=0;
	Object[]  fragments=null;  // Object is of type 'byte[]'
	int       number_of_frags=0;

	Entry(int tot_frags) {
	    this.tot_frags=tot_frags;
	    fragments=new Object[tot_frags];
	    for(int i=0; i < tot_frags; i++) fragments[i]=null;
	}

	public void Set(int frag_id, byte[] frag) {
	    fragments[frag_id]=frag;
	    number_of_frags++;
	}

	public boolean IsComplete() {
	    if(number_of_frags < tot_frags)
		return false;
	    for(int i=0; i < fragments.length; i++) {
		if(fragments[i] == null)
		    return false;
	    }
	    return true;
	}

	public byte[] AssembleBuffer() {
	    Vector tmp=new Vector();
	    for(int i=0; i < fragments.length; i++)
		tmp.addElement(fragments[i]);
	    return Util.DefragmentBuffer(tmp);
	}

	public String toString() {
	    StringBuffer ret=new StringBuffer();
	    ret.append("[tot_frags=" + tot_frags + ", number_of_frags=" + number_of_frags + "]");
	    return ret.toString();
	}

    }


    private Hashtable h=new Hashtable();

    /**
       Creates a new entry if not yet present. Adds the fragment. If all fragements for a given
       message haven been received, an entire message is reassembled and returned. Otherwise
       null is returned.
     */
    public synchronized byte[] Add(Object src, long msg_id, int frag_id, int tot_frags, 
				    byte[] fragment) {

	//System.out.println("Add(): src=" + src + ", msg_id=" + msg_id + ", frag_id=" + frag_id +
	//	   ", tot_frags=" + tot_frags + ", fragment=" + fragment.length + "bytes");

	byte[]     retval=null;
	Hashtable  val=(Hashtable)h.get(src);
	Entry      e;

	if(val == null) {         // Create new entry if not yet present
	    val=new Hashtable();
	    h.put(src, val);
	    e=new Entry(tot_frags);
	    val.put(new Long(msg_id), e);
	}
	else {
	    e=(Entry)val.get(new Long(msg_id));
	    if(e == null) {
		e=new Entry(tot_frags);
		val.put(new Long(msg_id), e);
	    }
	}
	e.Set(frag_id, fragment);
	if(e.IsComplete()) {
	    retval=e.AssembleBuffer();
	    val.remove(new Long(msg_id));
	}

	return retval;
    }

    public void Reset() {}
}



/**
   Fragmentation layer. Fragments messages larger than FRAG_SIZE into smaller packets.
   Reassembles fragmented packets into bigger ones. The fragmentation number is prepended
   to the messages as a header (and removed at the receiving side).<p>
   Requirement: lossless delivery (e.g. NAK, ACK). No requirement on ordering.

   Typical stack:
   <pre>
   FIFO
   FRAG
   NAK
   UDP
   </pre>

 */

public class FRAG extends Protocol {
    private int                 frag_size=60000;
    private boolean             trace=false;
    private FragmentationTable  frag_table=new FragmentationTable();

    public String  GetName() {return "FRAG";}



    /** Setup the Protocol instance acording to the configuration string */
    public void SetProperties(Properties props) {
	String     str;
	//this.props=props;

	super.SetProperties(props);


	str=props.getProperty("frag_size");
	if(str != null)
	    frag_size=new Integer(str).intValue();

	

	str=props.getProperty("trace");
	if(str != null)
	    trace=new Boolean(str).booleanValue();
    }



    /** Just remove if you don't need to reset any state */
    public void Reset() {frag_table.Reset();}



    /**
       1. Remove header
     */
    public void Up(Event evt) {
	byte[] m;
	if(evt.GetType() == Event.MSG) {
	    Message    msg=(Message)evt.GetArg();
	    FragHeader hdr=(FragHeader)msg.RemoveHeader();

	    if(hdr.num_frags > 0) {
		// 1. Get all the fragment buffers
		// 2. When all are received -> Assemble them into one big buffer
		// 3. Read headers and byte buffer from big buffer
		// 4. Set headers and buffer in msg
		// 5. Pass msg up the stack

		m=frag_table.Add(msg.GetSrc(), hdr.msg_id, hdr.frag_id, hdr.num_frags, 
				 msg.GetBuffer());

		if(m != null) {
		    msg.SetId(hdr.msg_id);


		    // 1. Read headers and set them in msg
		    // 2. Read buffer and set it in msg

		    ByteArrayInputStream in_stream=null;
		    ObjectInputStream    in=null;
		    Stack                headers;
		    byte[]               tmpbuf=null;
		    int                  len;

		    try {
			in_stream=new ByteArrayInputStream(m);
			in=new ObjectInputStream(in_stream);			
			headers=(Stack)in.readObject();
			tmpbuf=(byte[])in.readObject();
		    }
		    catch(Exception e) {
			System.err.println("FRAG.Up(): " + e);
			return;
		    }

		    msg.SetHeaders(headers);
		    msg.SetBuffer(tmpbuf);
		    PassUp(new Event(Event.MSG, msg));
		}
		return;
	    }
	}
		

	PassUp(evt); // Pass up to the layer above us
    }



    /**
       Fragment a packet if larger than FRAG_SIZE
     */
    public void Down(Event evt) {

	if(evt.GetType() == Event.MSG) {
	    Message msg=(Message)evt.GetArg();
	    long    size=msg.Size();
	    
	    if(size > frag_size) {
		Fragment(msg);  // Fragment and pass down
		return;
	    }
	    else
		msg.AddHeader(new FragHeader(msg.GetId(), 0, 0)); // Add dummy header
	}
	
	PassDown(evt);  // Pass on to the layer below us
    }


    private void Fragment(Message msg) {	
	ByteArrayOutputStream out_stream;
	ObjectOutputStream    out;
	byte[]                buffer;
	Vector                fragments;
	Event                 evt;
	FragHeader            hdr;
	Message               frag_msg;
	Object                dest=msg.GetDest(), src=msg.GetSrc();
	long                  msg_id=msg.GetId(), rsp_id=msg.GetRspId();
	boolean               oneway=msg.IsOneway();


	try {

	    // Write headers and buffer into a byte buffer

	    out_stream=new ByteArrayOutputStream();
	    out=new ObjectOutputStream(out_stream);
	    out.writeObject(msg.GetHeaders());
	    out.writeObject(msg.GetBuffer());

	    buffer=out_stream.toByteArray();
	    fragments=Util.FragmentBuffer(buffer, frag_size);
	    
	    if(trace)
		System.out.println("FRAG.Fragment(): fragmenting packet (size=" + buffer.length + 
				   ") into " + fragments.size() + " fragment(s)");

	    
	    /* Send all fragments as separate messages (with separate msg IDs). The first
	       message has to have the same msg ID as the entire message though !
	       Example:
	       Message '2344 | dst | src | buf'  would be fragmented into
	       '2344 | dst | src | buf1', '2345 | dst | src | buf1' and '2346 | dst | src | buf1'
	    */

	    frag_msg=msg;
	    frag_msg.GetHeaders().removeAllElements();	    
	    frag_msg.SetRspId(rsp_id);
	    frag_msg.SetBuffer((byte[])fragments.elementAt(0));
	    if(oneway) frag_msg.SetOneway();	    	    
	    hdr=new FragHeader(msg_id, 0, fragments.size());
	    frag_msg.AddHeader(hdr);
	    evt=new Event(Event.MSG, frag_msg);
	    PassDown(evt);

	    for(int i=1; i < fragments.size(); i++) {
		frag_msg=new Message(dest, src, (byte[])fragments.elementAt(i));
		frag_msg.SetRspId(rsp_id);
		if(oneway) frag_msg.SetOneway();
		hdr=new FragHeader(msg_id, i, fragments.size());
		frag_msg.AddHeader(hdr);
		evt=new Event(Event.MSG, frag_msg);
		PassDown(evt);
	    }	   
	}
	catch(Exception e) {
	    System.err.println("FRAG.Fragment(): " + e);
	}

    }

}
