package JavaGroups.JavaStack;

import java.util.Properties;
import JavaGroups.*;


class UpHandler extends Thread {
    private Queue    mq=null;
    private Protocol handler=null;

    public UpHandler(Queue mq, Protocol handler) {
	this.mq=mq; this.handler=handler;
	setName("UpHandler");
    }


    
    /** Removes events from mq and calls handler.Up(evt) */
    public void run() {
	Event evt;
	while(true) {
	    try {
		evt=(Event)mq.Remove();
		if(evt == null) {
		    System.err.println("UpHandler.run(): removed null event !");
		    continue;
		}
		handler.Up(evt);
	    }
	    catch(QueueClosed queue_closed) {
		break;
	    }
	    catch(Exception e) {
		System.err.println(e);		
	    }	
	}
    }


}



class DownHandler extends Thread {
    private Queue    mq=null;
    private Protocol handler=null;

    
    public DownHandler(Queue mq, Protocol handler) {
	this.mq=mq; this.handler=handler;
	setName("DownHandler");
    }



    /** Removes events from mq and calls handler.Down(evt) */
    public void run() {
	Event evt;
	while(true) {
	    try {
		evt=(Event)mq.Remove();
		if(evt == null) {
		    System.err.println("DownHandler.run(): removed null event !");
		    continue;
		}
		handler.Down(evt);
	    }
	    catch(QueueClosed queue_closed) {
		System.err.println("DownHandler.run(): " + queue_closed);
		break;
	    }
	    catch(Exception e) {
		System.err.println(e);		
	    }	
	}
    }
    
}



/**
 * The Protocol class provides a set of common services for protocol layers. Each layer has to
 * be a subclass of Protocol and override a number of methods (typically just <code>Up</code>,
 * <code>Down</code> and <code>GetName</code>. Layers are stacked in a certain order to form
 * a protocol stack. <a href=JavaGroups.JavaStack.Event.html>Events</a> are passed from lower 
 * layers to upper ones and vice versa. E.g. a Message received by the UDP layer at the bottom 
 * will be passed to its higher layer as an Event. That layer will in turn pass the Event to 
 * its layer and so on, until a layer handles the Message and sends a response or discards it, 
 * the former resulting in another Event being passed down the stack.<p>
 * Each layer has 2 FIFO queues, one for up Events and one for down Events. When an Event is
 * received by a layer (calling the internal upcall <code>ReceiveUpEvent</code>), it is placed
 * in the up-queue where it will be retrieved by the up-handler thread which will invoke method
 * <code>Up</code> of the layer. The same applies for Events traveling down the stack. Handling
 * of the up-handler and down-handler threads and the 2 FIFO queues is donw by the Protocol
 * class, subclasses will almost never have to override this behavior.<p>
 * The important thing to bear in mind is that Events have to passed on between layers in FIFO
 * order which is guaranteed by the Protocol implementation and must be guranteed by subclasses
 * implementing their on Event queuing.<p>
 * <b>Note that each class implementing interface Protocol MUST provide an empty, public
 * constructor !</b>
 */
public abstract class Protocol {
    protected Properties    props=null;
    protected Protocol      up_prot=null, down_prot=null;
    protected ProtocolStack stack=null;
    protected Queue         up_queue=new Queue(), down_queue=new Queue();
    protected UpHandler     up_handler=null;
    protected DownHandler   down_handler=null;
    

    
    /**
     * Configures the protocol initially. A configuration string consists of name=value 
     * items, separated by a ';' (semicolon), e.g.:<pre>
     * "loopback=false;unicast_inport=4444"
     * </pre>
     */
    public void             SetProperties(Properties props) {this.props=props;}
    public Properties       GetProperties()                 {return props;}

    public abstract String  GetName();

    public Protocol         GetUpProtocol()   {return up_prot;}
    public Protocol         GetDownProtocol() {return down_prot;}

    public void             SetUpProtocol(Protocol up_prot)     {this.up_prot=up_prot;}
    public void             SetDownProtocol(Protocol down_prot) {this.down_prot=down_prot;}

    public void             SetProtocolStack(ProtocolStack stack) {this.stack=stack;}


    /** Used internally. If overridden, call parent's method first */
    public void StartInternal() {       
	if(down_handler == null) {
	    down_handler=new DownHandler(down_queue, this);
	    down_handler.start();
	}
	if(up_handler == null) {
	    up_handler=new UpHandler(up_queue, this);
	    up_handler.start();
	}
    }

    /** Used internally. If overridden, call parent's method first */
    public void StopInternal() {
	if(up_handler != null) {
	    up_handler.stop();
	    up_queue.Close();
	    up_handler=null;
	}
	if(down_handler != null) {
	    down_handler.stop();
	    down_queue.Close();
	    down_handler=null;
	}
	up_queue.Reset(); down_queue.Reset();
    }


    /** Used when new member joins (new stack needs to start from a defined point),
     *  typically just resetting its state. For example, a sliding window protocol used for
     *  negative acknowledgement would reset its lower mark to 0 again. */
    public void Reset() {}  // used when new member joins -> all stacks are reset


    /**
     * Internal method, should not be called by clients. Used by ProtocolStack. I would have
     * used the 'friends' modifier, but this is available only in C++ ...
     */
    protected void ReceiveUpEvent(Event evt) {
	try {
	    up_queue.Add(evt);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }

    /**
     * Internal method, should not be called by clients. Used by ProtocolStack. I would have
     * used the 'friends' modifier, but this is available only in C++ ...
     */
    protected void ReceiveDownEvent(Event evt) {
	try {
	    down_queue.Add(evt);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }

    /**
     * Causes the event to be forwarded to the next layer up in the hierarchy. Typically called
     * by the implementation of <code>Up</code> (when done).
     */
    public void PassUp(Event evt) {
	if(up_prot != null) {
	    up_prot.ReceiveUpEvent(evt);
	}
	else
	    System.err.println("Protocol["+GetName()+"].PassUp(): no upper layer available !");
    }

    /**
     * Causes the event to be forwarded to the next layer down in the hierarchy.Typically called
     * by the implementation of <code>Down</code> (when done).
     */
    public void PassDown(Event evt) {
	if(down_prot != null)
	    down_prot.ReceiveDownEvent(evt);
	else
	    System.err.println("Protocol["+GetName()+"].PassDown(): no lower layer available !");
    }


    /**
     * An event was received from the layer below. Usually the current layer will want to examine
     * the event type and - depending on its type - perform some computation 
     * (e.g. removing headers from a MSG event type, or updating the internal membership list 
     * when receiving a VIEW_CHANGE event). 
     * Finally the event is either a) discarded, or b) an event is sent down
     * the stack using <code>PassDown</code> or c) the event (or another event) is sent up
     * the stack using <code>PassUp</code>.
     */
    public void Up(Event evt) {
	PassUp(evt);
    }

    /**
     * An event is to be sent down the stack. The layer may want to examine its type and perform
     * some action on it, depending on the event's type. If the event is a message MSG, then
     * the layer may need to add a header to it (or do nothing at all) before sending it down
     * the stack using <code>PassDown</code>. In case of a GET_ADDRESS event (which tries to
     * retrieve the stack's address from one of the bottom layers), the layer may need to send
     * a new response event back up the stack using <code>PassUp</code>.
     */
    public void Down(Event evt) {
	PassDown(evt);
    }


}
