package JavaGroups.JavaStack;

import java.io.*;
import java.net.*;
import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.Protocols.TunnelHeader;






/** 
 * Router for TCP based group comunication (using layer TCP instead of UDP). Instead of the TCP
 * layer sending packets point-to-point to each other member, it sends the packet to the router
 * which - depending on the target address - multicasts or unicasts it to the group / or single
 * member.<p>
 * This class is especially interesting for applets which cannot directly make connections
 * (neither UDP nor TCP) to a host different from the one they were loaded from. Therefore,
 * an applet would create a normal channel plus protocol stack, but the bottom layer would have
 * to be the TCP layer which sends all packets point-to-point (over a TCP connection) to the 
 * router, which in turn forwards them to their end location(s) (also over TCP). A centralized
 * router would therefore have to be running on the host the applet was loaded from.<p>
 * An alternative for running JavaGroups in an applet (IP multicast is not allows in applets as of
 * 1.2), is to use point-to-point UDP communication via the gossip server. However, then the appplet
 * has to be signed which involves additional administrative effort on the part of the user.
 */
public class JRouter {


    class AddressEntry {
	Address             addr=null;
	Socket              sock=null;
	ObjectOutputStream  out=null;
	
	public AddressEntry(Address addr, Socket sock) {
	    this.addr=addr; this.sock=sock;
	    try {
		this.out=new ObjectOutputStream(sock.getOutputStream());
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	}
	
	public boolean equals(Object other) {
	    return addr.equals(((AddressEntry)other).addr);
	}

	public String toString() {
	    return "addr=" + addr + ", sock=" + sock;
	}
    }


    
    class SocketThread extends Thread {
	private Socket sock=null;


	public SocketThread(Socket sock) {this.sock=sock;}

	private void CloseSocket() {
	    try {
		if(sock != null)
		    sock.close();
	    }
	    catch(Exception e) {}
	}

	public void run() {
	    Object               obj;
	    InputStream          i=null;
	    ObjectInputStream    in=null;
	    RegistrationRequest  req;

	    try {
		i=sock.getInputStream();
		in=new ObjectInputStream(i);
		ProcessRequest(sock, in);  // process initial registration
	    }
	    catch(EOFException eof_ex) {
		System.err.println("JRouter.SocketThread.run(): client closed connection, " +
				   "terminating socket thread");
		CloseSocket();
		return;
	    }
	    catch(Exception e) {
		System.err.println("JRouter.SocketThread.run(): " + e);
		CloseSocket();
		return;
	    }
	   
	    	    
	    while(true) {
		try {
		    obj=in.readObject();
		    if(!(obj instanceof Message)) {
			if(obj instanceof RegistrationRequest) {
			    req=(RegistrationRequest)obj;
			    if(req.unregister == true)
				RemoveEntry(req.groupname, req.addr);
			}
			else {
			    System.err.println("JRouter.SocketThread.run(): received request of " +
					       "type " + obj.getClass().getName() + " cannot " + 
					       "process; must be of type Message !");
			}
			continue;
		    }		
		    queue.Add((Message)obj);		
		}
		catch(IOException io_ex) {
		    System.err.println("JRouter.SocketThread.run(): client socket closed " +
				       "connection; removing entry from registration table");
		    RemoveEntry(sock);
		    return; // terminates thread
		}
		catch(QueueClosed closed) {
		    System.err.println("JRouter.SocketThread.run() queue is closed");
		    break;
		}
		catch(Exception e) {
		    System.err.println("JRouter.SocketThread.run(): " + e);
		    break;
		}
	    }
	    CloseSocket();
	}       
	
    }




    class RouterThread extends Thread {
	Message msg;

	public void run() {
	    while(true) {
		try {
		    msg=(Message)queue.Remove();
		    Route(msg);
		}
		catch(QueueClosed closed) {
		    break;
		}
		catch(Exception e) {
		    System.err.println(e);
		}
	    }

	}

    }


    


    private Hashtable     groups=new Hashtable();  // groupname - vector of AddressEntry's
    private Queue         queue=new Queue();
    private int           port=12002;
    private ServerSocket  srv_sock=null;
    private RouterThread  router_thread=null;


    
    private void AddEntry(String groupname, AddressEntry e) {
	Vector        val;
	
	Util.Print("AddEntry(" + groupname + ", " + e + ")");

	synchronized(groups) {
	    val=(Vector)groups.get(groupname);
	    
	    if(val == null) {
		val=new Vector();
		val.addElement(e);
		groups.put(groupname, val);
	    }
	    else
		if(!val.contains(e))
		    val.addElement(e);
	}
    }


    
    private void RemoveEntry(Socket sock) {
	Vector        val;
	AddressEntry  entry;
	
	synchronized(groups) {
	    for(Enumeration e=groups.keys(); e.hasMoreElements();) {
		val=(Vector)groups.get((String)e.nextElement());
		for(int i=0; i < val.size(); i++) {
		    entry=(AddressEntry)val.elementAt(i);
		    if(entry.sock == sock) {
			try {
			    entry.sock.close();
			}
			catch(Exception ex) {
			    System.err.println(ex);
			}
			Util.Print("Removing entry " + entry);
			val.removeElement(entry);
			return;
		    }
		}
	    }
	}
    }


    private void RemoveEntry(OutputStream out) {
	Vector        val;
	AddressEntry  entry;
	
	synchronized(groups) {
	    for(Enumeration e=groups.keys(); e.hasMoreElements();) {
		val=(Vector)groups.get((String)e.nextElement());
		for(int i=0; i < val.size(); i++) {
		    entry=(AddressEntry)val.elementAt(i);
		    if(entry.out == out) {
			try {
			    if(entry.sock != null)
				entry.sock.close();
			}
			catch(Exception ex) {
			    System.err.println(ex);
			}
			Util.Print("Removing entry " + entry);
			val.removeElement(entry);
			return;
		    }
		}
	    }
	}
    }


    private void RemoveEntry(String groupname, Address addr) {
	Vector        val;
	AddressEntry  entry;
	


	synchronized(groups) {
	    val=(Vector)groups.get(groupname);
	    if(val == null || val.size() == 0)
		return;
	    for(int i=0; i < val.size(); i++) {
		entry=(AddressEntry)val.elementAt(i);
		if(entry.addr.equals(addr)) {
		    try {
			if(entry.sock != null)
			    entry.sock.close();
		    }
		    catch(Exception ex) {
			System.err.println(ex);
		    }
		    Util.Print("Removing entry " + entry);
		    val.removeElement(entry);
		    return;
		}
	    }
	}
    }


    private ObjectOutputStream FindSocket(Address addr) {
	Vector        val;
	AddressEntry  entry;
	
	synchronized(groups) {
	    for(Enumeration e=groups.keys(); e.hasMoreElements();) {
		val=(Vector)groups.get((String)e.nextElement());
		for(int i=0; i < val.size(); i++) {
		    entry=(AddressEntry)val.elementAt(i);
		    if(addr.equals(entry.addr))
			return entry.out;
		}
	    }
	    return null;
	}
    }

    
    public void ProcessRequest(Socket sock, ObjectInputStream in) throws Exception {
	Object        req;
	AddressEntry  entry;
	
	req=in.readObject();
	if(req instanceof RegistrationRequest) {

	    Address peer_addr=new Address(sock.getInetAddress(), sock.getPort());
	    Util.Print("\nPeer address is " + peer_addr);

	    ((RegistrationRequest)req).addr=peer_addr;
	    Util.Print("Received reg req: " + req);

	    entry=new AddressEntry(((RegistrationRequest)req).addr, sock);


	    AddEntry(((RegistrationRequest)req).groupname, entry);

	    Util.Print("Sending inet_addr back to client: " + peer_addr);
	    entry.out.writeObject(peer_addr);
	}
	else
	    System.err.println("JRouter.SocketThread.ProcessRequest(): request " + 
			       req.getClass().getName() + " not expected !");
    }
    

    

    private void SendToAllMembersInGroup(String groupname, Message msg) {
	Vector val;

	synchronized(groups) {
	    val=(Vector)groups.get(groupname);
	    if(val == null || val.size() == 0)
		return;
	    for(int i=0; i < val.size(); i++)
		SendToMember(((AddressEntry)val.elementAt(i)).out, msg);
	}
    }


    private void SendToMember(ObjectOutputStream out, Message msg) {
	try {
	    if(out != null)
		out.writeObject(msg);
	}
	catch(Exception e) {
	    System.err.println("JRouter.SendToMember(): " + e);
	    RemoveEntry(out);
	}
    }


    private void Route(Message msg) {
	Address         dest=(Address)msg.GetDest(), src=(Address)msg.GetSrc();
	TunnelHeader    hdr=(TunnelHeader)msg.PeekHeader();
	String          dest_group=hdr != null ? hdr.channel_name : null;
	
	if(dest == null) { // send to all members in group 'dest.GetChannelName()'	
	    if(dest_group == null) {
		System.err.println("JRouter.Route(): both dest address and group are null");
		return;
	    }
	    else
		SendToAllMembersInGroup(dest_group, msg);
	}                      
	else {                  // send to destination address
	    ObjectOutputStream out=FindSocket(dest);
	    if(out != null)
		SendToMember(out, msg);
	    else
		System.out.println("FAILED, outstream is null !");
	}
    }






    public JRouter(int port) throws Exception {
	this.port=port;
	srv_sock=new ServerSocket(port, 50);  // backlog of 50 connections
    }



    public void Start() {
 	if(router_thread == null) {
 	    router_thread=new RouterThread();
 	    router_thread.start();
 	}

	System.out.println("\nJRouter started at " + new Date());
	System.out.println("Listening on port " + port + "\n");

	while(true) {
	    try {
		Socket sock=srv_sock.accept();
		new SocketThread(sock).start();
	    }
	    catch(Exception e) {
		System.err.println(e);
		continue;
	    }
	}
    }

    public void Stop() {
 	if(router_thread != null) {
 	    router_thread.stop();
 	    router_thread=null;
 	}
    }



    





    public static void main(String[] args) {
	String           arg, next_arg;
	int              port=12002;
	JRouter          router=null; 

	for(int i=0; i < args.length; i++) {
	    arg=args[i];
	    if(arg.equals("-help")) {
		System.out.println("JRouter [-port <port> ]");
		return;
	    }
	    else if(arg.equals("-port")) {
		port=new Integer(args[++i]).intValue();
		continue;
	    }
	}
	try {
	    router=new JRouter(port);
	    router.Start();
	    System.out.println("JRouter was created at " + new Date());
	    System.out.println("Listening on port " + port);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }

    
}
