package JavaGroups.JavaStack;

import java.util.*;

/**
 * The task if this class is to setup and configure the protocol stack. A string describing
 * the desired setup, which is both the layering and the configuration of each layer, is
 * given to the configurator which creates and configures the protocol stack and returns
 * a reference to the top layer (Protocol).<p>
 * Future functionality will include the capability to dynamically modify the layering
 * of the protocol stack and the properties of each layer.
 */
public class Configurator {
    
    public class ProtocolConfiguration {
	private String        protocol_name=null;
	private String        properties_str=null;
	private Properties    properties=new Properties();
	private final String  protocol_prefix="JavaGroups.JavaStack.Protocols";



	private void SetContents(String config_str) {
	    int index=config_str.indexOf('(');  // e.g. "UDP(in_port=3333)"
	    int end_index=config_str.lastIndexOf(')');

	    if(index == -1) {
		protocol_name=config_str;
	    }
	    else {
		if(end_index == -1) {
		    System.err.println("Closing ')' not found in " + config_str);
		    System.err.println("Properties cannot be set !");
		}
		else {
		    properties_str=config_str.substring(index+1, end_index);
		    protocol_name=config_str.substring(0, index);
		}
	    }

	    /* "in_port=5555;out_port=6666" */
	    if(properties_str != null) {
		Vector  components=ParseComponentStrings(properties_str, ';');
		if(components.size() > 0) {
		    for(int i=0; i < components.size(); i++) {
			String  name, value, comp=(String)components.elementAt(i);
			index=comp.indexOf('=');
			if(index == -1) {
			    System.err.println("'=' not found in " + comp);
			    continue;
			}
			name=comp.substring(0, index);
			value=comp.substring(index+1, comp.length());
			properties.put(name, value);			
		    }
		}
	    }
	}



	public ProtocolConfiguration(String config_str) {
	    SetContents(config_str);
	}

	private String     GetProtocolName() {return protocol_name;}
	private Properties GetProperties()   {return properties;}


	private Protocol CreateLayer(ProtocolStack prot_stack) {
	    Protocol retval=null;
	    if(protocol_name == null)
		return null;
	    try {
		protocol_name=protocol_prefix + "." + protocol_name;
		retval=(Protocol)Class.forName(protocol_name).newInstance();
		if(retval == null)
		    System.err.println("Creation of protocol " + protocol_name  +"failed !");
		else
		    retval.SetProtocolStack(prot_stack);
	    }
	    catch(InstantiationException inst_ex) {
		System.err.println(inst_ex);
		System.err.println("An instance of " + protocol_name +
				   " could not be created. Please check that it implements" +
				   " interface Protocol and that is has a public empty" +
				   " constructor !");
	    }
	    catch(Exception e) {
		System.err.println(e);
		return null;
	    }
	    if(properties != null)
		retval.SetProperties(properties);
	    return retval;
	}



	public String toString() {
	    StringBuffer retval=new StringBuffer();
	    retval.append("Protocol: ");
	    if(protocol_name == null)
		retval.append("<unknown>");
	    else
		retval.append(protocol_name);
	    if(properties != null)
		retval.append("(" + properties + ")");
	    return retval.toString();
	}
    }





    /**
     * The configuration string has a number of entries, separated by a ':' (semi).
     * Each entry consists of the name of the protocol, followed by an optional configuration
     * of that protocol. The configuration is enclosed in parentheses, and contains entries
     * which are name/value pairs connected with an assignment sign (=) and separated by
     * a semicolon.
     * <pre>UDP(in_port=5555;out_port=4445):FRAG(frag_size=1024)</pre><p>
     * The <em>first</em> entry defines the <em>bottommost</em> layer, the string is parsed
     * left to right and the protocol stack constructed bottom up. Example: the string
     * "UDP(in_port=5555):FRAG(frag_size=32000):DEBUG" results is the following stack:<pre>
     *   
     *   -----------------------
     *  | DEBUG                 |
     *  |-----------------------|
     *  | FRAG frag_size=32000  |
     *  |-----------------------|
     *  | UDP in_port=32000     |
     *   -----------------------
     * </pre>
     */
    public Protocol SetupProtocolStack(String configuration, ProtocolStack st) throws Exception {
	Protocol                protocol_stack=null;
	Protocol                layer;
	Vector                  protocol_configs;
	ProtocolConfiguration   protocol_config;
	Vector                  protocols;

	protocol_configs=ParseConfigurations(configuration);
	protocols=CreateProtocols(protocol_configs, st);
	protocol_stack=ConnectProtocols(protocols);
	return protocol_stack;
    }

	
	
    /**
     * Get a string of the form "P1(config_str1):P2:P3(config_str3)" and return
     * ProtocolConfigurations for it. That means, parse "P1(config_str1)", "P2" and 
     * "P3(config_str3)"
     * @param config_str Configuration string
     * @return Vector of ProtocolConfigurations
     */
    private Vector ParseComponentStrings(String config_str, int delimiter) {
	Vector         retval=new Vector();
	StringBuffer   buf=new StringBuffer();
	int            index=0;
	int            ch;
	boolean        esc=false;

	for(int i=0; i < config_str.length(); i++) {
	    ch=config_str.charAt(i);

	    if(ch == '\\') { // esc character, copy the next char literally into string buffer
		i++;
		buf.append(config_str.charAt(i));
		continue;
	    }
	    if(ch == delimiter) {
		retval.addElement(buf.toString());		
		buf=new StringBuffer();
		continue;		
	    }
	    buf.append(config_str.charAt(i)); // catches rest
	}
	if(buf.length() > 0)
	    retval.addElement(buf.toString());
	return retval;
    }



    /**
     * Return a number of ProtocolConfigurations in a vector
     * @param configuration protocol-stack configuration string
     * @return Vector of ProtocolConfigurations
     */
    private Vector ParseConfigurations(String configuration) {
	Vector                 retval=new Vector();
	Vector                 component_strings=ParseComponentStrings(configuration, ':');
	String                 component_string;
	ProtocolConfiguration  protocol_config;

	if(component_strings == null)
	    return null;
	for(int i=0; i < component_strings.size(); i++) {
	    component_string=(String)component_strings.elementAt(i);
	    protocol_config=new ProtocolConfiguration(component_string);
	    retval.addElement(protocol_config);
	}
	return retval;
    }






    /**
     * Takes vector of ProtocolConfigurations, iterates through it, creates Protocol for
     * each ProtocolConfiguration and returns all Protocols in a vector.
     * @param configurations Vector of ProtocolConfigurations
     * @return Vector of Protocols
     */

    private Vector CreateProtocols(Vector protocol_configs, ProtocolStack stack) {
	Vector                 retval=new Vector();
	ProtocolConfiguration  protocol_config;
	Protocol               layer;

	for(int i=0; i < protocol_configs.size(); i++) {
	    protocol_config=(ProtocolConfiguration)protocol_configs.elementAt(i);
	    layer=protocol_config.CreateLayer(stack);
	    retval.addElement(layer);
	}
	return retval;
    }

    /**
     * Creates a protocol stack by iterating through the protocol list and connecting 
     * adjacent layers. The list starts with the topmost layer and has the bottommost
     * layer at the tail. When all layers are connected the algorithms traverses the list
     * once more to call StartInternal() and StartWork() on each layer.
     * @param protocol_list List of Protocol elements (from top to bottom)
     * @return Protocol stack
     */
    private Protocol ConnectProtocols(Vector protocol_list) {
	Protocol current_layer=null, next_layer=null;

	// System.out.println("Connecting " + protocol_list.size() + " protocols: ");
	for(int i=0; i < protocol_list.size(); i++) {
	    current_layer=(Protocol)protocol_list.elementAt(i);
	    try {
		next_layer=(Protocol)protocol_list.elementAt(i+1);
	    }
	    catch(ArrayIndexOutOfBoundsException out) {
		break;
	    }
	    current_layer.SetUpProtocol(next_layer);
	    next_layer.SetDownProtocol(current_layer);	    
	}
	return current_layer;
    }

    
    public void StartProtocolStack(Protocol bottom_prot) {
	while(bottom_prot != null) {
	    bottom_prot.StartInternal();
	    bottom_prot=bottom_prot.GetUpProtocol();
	}
    }




    public void StopProtocolStack(Protocol start_prot) {	
	while(start_prot != null) {
	    start_prot.StopInternal();
	    start_prot=start_prot.GetDownProtocol();
	}
    }






    public Protocol FindProtocol(Protocol prot_stack, String name) {
	String    s;
	Protocol  curr_prot=prot_stack;

	while(true) {
	    s=curr_prot.GetName();
	    if(s == null)
		continue;
	    if(s.equals(name))
		return curr_prot;
	    curr_prot=curr_prot.GetDownProtocol();
	    if(curr_prot == null)
		break;
	}
	return null;
    }


    Protocol GetBottommostProtocol(Protocol prot_stack) throws Exception {
	Protocol  tmp=null, curr_prot=prot_stack;

	while(true) {
	    if((tmp=curr_prot.GetDownProtocol()) == null)
		break;
	    curr_prot=tmp;
	}
	if(curr_prot == null)
	    throw new Exception("Configurator.GetBottommostProtocol(): bottommost protocol " +
				"is null !");
	return curr_prot;
    }


    public static void main(String args[]) {
	if(args.length != 1) {
	    System.err.println("Configurator <string>");
	    System.exit(0);
	}
	String         config_str=args[0];
	Configurator   conf=new Configurator();
	Vector         protocol_configs;
	Vector         protocols;
	Protocol       protocol_stack;


	protocol_configs=conf.ParseConfigurations(config_str);
	protocols=conf.CreateProtocols(protocol_configs, null);
	protocol_stack=conf.ConnectProtocols(protocols);

	try {
	    Thread.currentThread().sleep(3000);
	    conf.StopProtocolStack(protocol_stack);
	    // conf.StopProtocolStackInternal(protocol_stack);
	}
	catch(Exception e) {
	    System.err.println(e);
	}

	System.err.println(protocols);
    }
    
   
}



