package JavaGroups;

import java.io.*;
import java.util.*;
import iBus.View;
import iBus.*;

/**
 * Implementation of a channel using iBus. <a href=http://www.softwired.ch/ibus/>iBus</a>
 * is a pure Java-based publish/subscribe bus. An entity subscribed to a certain address
 * receives all messages published by any publisher under the same address (or topic).
 */
public class IbusChannel extends Channel implements iBus.Receiver, iBus.Membership {
    private iBusURL             group_addr=null;
    private iBusURL             my_addr=null;
    private JavaGroups.View     my_view=null;
    private String              properties="Reliable"; // replace with Vsync when available !
    private iBus.Stack          stack=null;
    private Queue               mq=new Queue();
    private Vector              members=new Vector();
    private String              channel_name=null;
    private boolean             closed=false;
    private boolean             receive_views=true;
    private boolean             receive_suspects=true;
    private boolean             receive_blocks=false;
    private boolean             receive_local_msgs=true;
    


    
    private void CheckConnection() throws ChannelNotConnected {
	if(stack == null)
	    throw new ChannelNotConnected();
    }

    private void CheckClosed() throws ChannelClosed {
	if(closed)
	    throw new ChannelClosed();
    }


    public IbusChannel() throws ChannelException {
	this(null);
    }

    public IbusChannel(Object props) throws ChannelException {
	if(props != null) 
	    if(props instanceof String)
		properties=(String)props;
	    else
		throw new ChannelException("IbusChannel(): properties must be of type String !");
	closed=false;
    }


    public void Connect(Object address) throws ChannelClosed {
	CheckClosed();

	try {
	    channel_name=(String)address;	    
	    group_addr=new iBusURL("ibus://226.0.0.1/tmp/" + channel_name);
	    stack=new iBus.Stack(properties);
	    my_addr=stack.getStackURL();
	    stack.registerMonitor(group_addr, this);
	    stack.subscribe(group_addr, this);
	    stack.registerTalker(group_addr);
	}
	catch(ClassCastException cce) {
	    System.err.println(cce.getMessage() + " (IbusChannel.Connect(): " + 
				"group address is not a String)");
	}
	catch(Exception e) {
	    System.err.println("IbusChannel.Connect(): " + e);
	}
    }




    public void Disconnect() {
	try {
	    if(stack != null) {
		stack.unregisterTalker(group_addr);
		stack.unsubscribe(group_addr, this);
		stack.unregisterMonitor(group_addr, this);
	    }
	}
	catch(Exception e) {
	    System.err.println("IbusChannel.Disconnect(): " + e);
	}
    }



    public void Close() {
	if(stack != null) {
	    stack.readyToExit();	    
	    stack=null;
	}
	closed=true;
    }




    public void Send(Message msg) throws ChannelNotConnected, ChannelClosed {
	byte buf[];

	CheckClosed();
	CheckConnection();

	if(msg.GetDest() == null)
	    msg.SetDest(group_addr);
	msg.SetSrc(stack.getStackURL());

	Posting p=new Posting();
	p.setLength(1);
	p.setObject(0, msg);
	try {
	    stack.push(group_addr, p);
	}
	catch(Exception e) {
	    System.err.println("IbusChannel.Send(): " + e);
	}
    }





    public Object Receive(long timeout) throws ChannelNotConnected, ChannelClosed, Timeout {
	Message    retval=null;
	Event      evt;

	CheckClosed();
	CheckConnection();
	    
	if(mq == null)
	    throw new ChannelNotConnected();
	    
	try {
	    evt=(timeout <= 0)? (Event)mq.Remove() : (Event)mq.Remove(timeout);
	    if(evt == null)
		return null; // correct ?

	    switch(evt.GetType()) {
	    case Event.MSG:
		return evt.GetArg();
	    case Event.VIEW_CHANGE:
		my_view=(JavaGroups.View)evt.GetArg();
		return my_view;
	    case Event.SUSPECT:
		return new SuspectEvent(evt.GetArg());
	    case Event.BLOCK:
		return new Block();
	    default:
		System.err.println("IbusChannel.Receive(): event is neither message " + 
				   "nor view nor block !");
		return null;
	    }
	}
	catch(Timeout tex) {
	    throw tex;
	}
	catch(QueueClosed queue_closed) {
	    System.err.println("IbusChannel.Receive(): " + queue_closed);
	    throw new ChannelNotConnected();
	}
	catch(Exception e) {
	    System.err.println(e);
	    return null;
	}
    }



    public Object Peek(long timeout) throws ChannelNotConnected, ChannelClosed, Timeout {
	Message    retval=null;
	Event      evt;

	CheckClosed();
	CheckConnection();
	    
	if(mq == null)
	    throw new ChannelNotConnected();
	    
	try {
	    evt=(timeout <= 0)? (Event)mq.Peek() : (Event)mq.Peek(timeout);
	    if(evt == null)
		return null; // correct ?

	    switch(evt.GetType()) {
	    case Event.MSG:
		return evt.GetArg();
	    case Event.VIEW_CHANGE:
		return evt.GetArg();
	    case Event.SUSPECT:
		return new SuspectEvent(evt.GetArg());
	    case Event.BLOCK:
		return new Block();
	    default:
		System.err.println("IbusChannel.Peek(): event is neither message " + 
				   "nor view nor block !");
		return null;
	    }
	}
	catch(Timeout tex) {
	    throw tex;
	}
	catch(QueueClosed queue_closed) {
	    System.err.println("IbusChannel.Peek(): " + queue_closed);
	    throw new ChannelNotConnected();
	}
	catch(Exception e) {
	    System.err.println(e);
	    return null;
	}
    }




    public JavaGroups.View   GetView()         {return my_view;}

    
    public Object            GetLocalAddress() {return my_addr;}


    public Object            GetGroupAddress()  {return group_addr;}



    public void    SetOpt(int option, Object value) {
	switch(option) {
	case VIEW:
	    if(value instanceof Boolean)
		receive_views=((Boolean)value).booleanValue();
	    else
		System.err.println("IbusChannel.SetOpt(" + option + ", " + value + 
				   "): value has to be Boolean");
	    break;
	case SUSPECT:
	    if(value instanceof Boolean)
		receive_suspects=((Boolean)value).booleanValue();
	    else
		System.err.println("IbusChannel.SetOpt(" + option + ", " + value + 
				   "): value has to be Boolean");
	    System.out.println("IbusChannel.SetOpt(): SUSPECT is not supported !");
	    break;
	case BLOCK:
	    if(value instanceof Boolean)
		receive_blocks=((Boolean)value).booleanValue();
	    else
		System.err.println("IbusChannel.SetOpt(" + option + ", " + value + 
				   "): value has to be Boolean");
	    if(receive_blocks)
		receive_views=true;
	    break;
	case LOCAL:
	    if(value instanceof Boolean)
		receive_local_msgs=((Boolean)value).booleanValue();
	    else
		System.err.println("IbusChannel.SetOpt(" + option + ", " + value + 
				   "): value has to be Boolean");
	    break;
	default:
	    System.err.println("IbusChannel.SetOpt(" + option + ", " + value + 
			       "): option not known");
	    break;
	}
    }



    public Object  GetOpt(int option) {
	switch(option) {
	case VIEW:
	    return new Boolean(receive_views);
	case SUSPECT:
	    return new Boolean(receive_suspects);
	case BLOCK:
	    return new Boolean(receive_blocks);
	case LOCAL:
	    return new Boolean(receive_local_msgs);
	default:
	    System.err.println("IbusChannel.GetOpt(" + option + "): option not known");
	    return null;
	}
    }



    public void    BlockOk() {
	// pointless; iBus does not know BLOCK events
    }









    /*-------------------- iBus callbacks ---------------------------- */

    public Posting dispatchPull(iBusURL channel, Posting request) {
	System.err.println("IbusChannel.dispatchPull(): not implemented !");
	return null;
    }



    public void dispatchPush(iBusURL source, Posting p) {
	Message    msg=(Message)p.getObject(0);

	if(!receive_local_msgs) {  // discard local messages (sent by myself to me)
	    if(my_addr != null && p.getSender() != null)
		if(my_addr.equals(p.getSender()))
		    return;
	}
	
	try {
	    if(msg.GetSrc() == null)
		msg.SetSrc(p.getSender());

	    if(!msg.GetDest().equals(stack.getStackURL()) && !msg.GetDest().equals(group_addr))
		return;

	    mq.Add(new Event(Event.MSG, msg));
	}
	catch(Exception e) {
	    System.err.println("IbusChannel.dispatchPush(): " + e);
	}
    }


    
    public void error (iBusURL url, String details) {
	System.err.println ("IbusChannel.error(): " + details);
	// System.exit (1); // ??
    }


    public Serializable getState(iBusURL channel) {return null;}

    public void setState(iBusURL channel, Serializable state) {}


    public synchronized void viewChange(View newView) {
	Object            coord=newView.getStack(); // originator of the view
	long              id=newView.getViewID();
	int               len=newView.getNumMembers();
	Vector            members=new Vector();
	JavaGroups.View   v;       

	if(!receive_views)
	    return;
	
	members.removeAllElements();
	ChannelMember[] mems=newView.getMembers();
	for(int i=0; i < len; i++)
	    members.addElement(mems[i].getURL());		

	v=new JavaGroups.View(coord, id, members);

	try {
	    mq.Add(new Event(Event.VIEW_CHANGE, v));
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }

}
