package JavaGroups;

import java.util.*;
import JavaGroups.Ensemble.*;

/**
 * EnsChannel is an implementation of <em>Channel</em> based on 
 * <a href=http://simon.cs.cornell.edu/Info/Projects/Ensemble/index.html>Ensemble</a>. It 
 * maps a process group to a channel. Requirements are the presence of an executable called
 * <em>outboard</em> (which is Ensemble) and <em>gossip</em> running.
 */
public class EnsChannel extends Channel implements Hot_Callbacks {
    private Hot_Ensemble        ensemble=null;
    private Thread              ens_thread=null;
    private Hot_GroupContext    channel_id=null;
    private String              channel_name=null;
    private Hot_JoinOps         options=new Hot_JoinOps();
    private Queue               mq=new Queue();
    private Object              my_addr=null;
    private View                my_view=null;
    private final String        default_properties="Gmp:Sync:Heal:Frag:Suspect:Flow:Total";

    private final int           CAST_MSG  = 0;
    private final int           SEND_MSG  = 1;
    private final int           VIEW_MSG  = 2;
    private final int           EXIT_MSG  = 3;
    private final int           BLOCK_MSG = 4;
    
    private boolean             receive_views=true;
    private boolean             receive_suspects=true;
    private boolean             receive_blocks=false;
    private boolean             receive_local_msgs=true;


    private void CheckConnection() throws ChannelNotConnected {
	if(channel_id == null)
	    throw new ChannelNotConnected();
    }


    private void CheckClosed() throws ChannelClosed {
	if(ensemble == null)
	    throw new ChannelClosed();
    }


    public EnsChannel() throws ChannelException {
	this(null);
    }


    /**
       Creates a new EnsChannel, which spawns an outboard process and connects to it.
       @param name Channel name. All channels with the same name form a group.
       @param properties Ensemble properties (cf. Ensemble reference manual). 
                         A value of <code>null</code> uses the default properties.
     */
    public EnsChannel(Object props) throws ChannelException {
	String properties=null;

	try {
	    properties=(String)props;
	}
	catch(ClassCastException cce) {
	    throw new ChannelException("EnsChannel(): properties argument must be of type String !");
	}

	options.heartbeat_rate=5000;
	options.transports="UDP";
	options.properties=properties == null? default_properties : properties;	
	options.params="suspect_max_idle=3:int;suspect_sweep=3.000:time";
	options.conf=this;
	options.use_properties=true;
	ensemble=new Hot_Ensemble();
	System.out.println("Created Hot_Ensemble");
	ens_thread=new Thread(ensemble, "EnsembleThread");
	ens_thread.start();
	System.out.println("EnsChannel(): done");
    }

    /**
       Creates a new EnsChannel. Instead of spawning a new outboard process, it connects to an
       already running outboard process (on the same machine) using <code>outboard_port</code>.
       This allows multiple EnsChannels to share a copy of outboard. If the port is 0, outboard
       <em>will</em> be spawned. Parameter <code>transport_props</code> defines the type 
       of transport to be used (UDP, ATM, IP MCAST etc).
       @param properties Ensemble properties (cf. Ensemble reference manual). 
                         A value of <code>null</code> uses the default properties.
       @param transport_props Transport parameters. <code>Null</code> means use default (UDP).
                              Example: <code>"UDP:DEERING"</code> uses IP multicast (gossip is not
			      needed in this case).
       @param outboard_port Port on which the local outboard process is listening. The outboard
                            process has to be started before. Value of 0 means spawn outboard
			    nevertheless.
    */
    public EnsChannel(Object props, String transport_props, int outboard_port) 
	throws ChannelException {

	String properties=null;

	try {
	    properties=(String)props;
	}
	catch(ClassCastException cce) {
	    throw new ChannelException("EnsChannel(): properties argument must be of type String !");
	}

	// channel_name=name;
	options.heartbeat_rate=5000;
	options.transports="UDP";
	if(transport_props != null)
	    options.transports=transport_props;
	// options.group_name=channel_name;
	options.properties=properties == null? default_properties : properties;	
	options.params="suspect_max_idle=3:int;suspect_sweep=3.000:time";
	options.conf=this;
	options.use_properties=true;
	ensemble=outboard_port == 0 ? new Hot_Ensemble() : new Hot_Ensemble(outboard_port);
	ens_thread=new Thread(ensemble, "EnsembleThread");
	ens_thread.start();
	System.out.println("EnsChannel(): done");
    }
    

    public void Connect(Object address) throws ChannelClosed {
	Hot_Error         rc;
	Hot_GroupContext  tmp[]=new Hot_GroupContext[1];

	CheckClosed();

	try {
	    channel_name=(String)address;
	    options.group_name=channel_name;
	}
	catch(ClassCastException cce) {
	    System.err.println("EnsChannel.Connect(): group address must be of type String !");
	    return;
	}
	
	if(channel_id != null) {
	    System.err.println("JChannel.Join(" + channel_name + "): already connected !");
	    return;
	}

	if(ensemble == null || ens_thread == null) {
	    System.err.println("Ensemble has not been started");
	    return;
	}


	rc=ensemble.Join(options, tmp);
	if(rc != null) {
	    System.err.println("EnsChannel.Connect(): " + rc.toString());
	    return;
	}
	channel_id=tmp[0];
    }



    public void Disconnect() {
	Hot_Error rc;
	if(channel_id == null) {
	    System.err.println("Cannot disconnect as channel id is null");
	    return;
	}
	rc=ensemble.Leave(channel_id);	
	if(rc != null)
	    System.err.println("EnsChannel.Disconnect(): " + rc);
	channel_id=null;

    }



    public void Close() {
	if(ensemble != null) {
	    if(ens_thread != null) {
		ens_thread.stop();
		ens_thread=null;
	    }
	    try {
		Thread.currentThread().sleep(500);
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	    ensemble.destroyOutboard();
	    ensemble=null;
	    try {
		mq.Close();
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	    mq.Reset();
	}
    }


    public void finalize() {
	if(channel_id != null)
	    Disconnect();
	Close();
    }



    private void Cast(byte[] msg) throws ChannelNotConnected, ChannelClosed {
	Hot_ObjectMessage  m;
	Hot_Error          rc;
	int                tmp[]=new int[1];

	CheckConnection();
	CheckClosed();

	m=new Hot_ObjectMessage(new Message(null, null, msg));

	rc=ensemble.Cast(channel_id, m, tmp);
	if(rc != null)
	    System.err.println("EnsChannel.Cast(): " + rc);
    }


    private void Send(Object dest_addr, byte[] msg) throws ChannelNotConnected, ChannelClosed {
	Hot_Endpoint        dest;
	Hot_Error           rc;
	Hot_ObjectMessage   m;
	int                 tmp[]=new int[1];

	CheckConnection();
	CheckClosed();

	if(dest_addr == null) {
	    System.err.println("EnsChannel.Send(): destination is null");
	    return;
	}
	dest=(Hot_Endpoint)dest_addr;
	if(dest == null) {
	    System.err.println("EnsChannel.Send(): destination object is not of type " +
			       "Hot_Endpoint, but of type " + dest_addr.getClass().getName());
	    return;
	}

	m=new Hot_ObjectMessage(new Message(dest, null, msg));

	rc=ensemble.Send(channel_id, dest, m, tmp);
	if(rc != null)
	    System.err.println("EnsChannel.Send(): " + rc);
    }





    public void Send(Message msg) throws ChannelNotConnected, ChannelClosed  {
	Object             dest=msg.GetDest();
	Hot_ObjectMessage  m=new Hot_ObjectMessage(msg);
	Hot_Error          rc=null;
	int                tmp[]=new int[1];

	CheckConnection();
	CheckClosed();
	
	if(dest == null || dest instanceof String || dest instanceof Vector)
	    rc=ensemble.Cast(channel_id, m, tmp);
	else if(dest instanceof Hot_Endpoint)
	    rc=ensemble.Send(channel_id, (Hot_Endpoint)dest, m, tmp);
	else {
	    System.err.println("EnsChannel.Send(): dest address is " + 
			       "wrong (" + dest + ")");
	    return;
	}

	if(rc != null)
	    System.err.println("EnsChannel.Send(): rc is " + rc);
    }


	


    public Object Receive(long timeout) throws ChannelNotConnected, ChannelClosed,  Timeout {
	Message    retval=null;
	Event      evt;

	CheckConnection();
	CheckClosed();
	    
	if(mq == null)
	    throw new ChannelNotConnected();
	    
	try {
	    evt=(timeout <= 0)? (Event)mq.Remove() : (Event)mq.Remove(timeout);
	    if(evt == null)
		return null; // correct ?

	    switch(evt.GetType()) {
	    case Event.MSG:
		return evt.GetArg();
	    case Event.VIEW_CHANGE:
		my_view=(View)evt.GetArg();
		return my_view;
	    case Event.SUSPECT:
		return new SuspectEvent(evt.GetArg());
	    case Event.BLOCK:
		return new BlockEvent();
	    default:
		System.err.println("EnsChannel.Receive(): event is neither message " + 
				   "nor view nor block !");
		return null;
	    }
	}
	catch(Timeout tex) {
	    throw tex;
	}
	catch(QueueClosed queue_closed) {
	    System.err.println("EnsChannel.Receive(): " + queue_closed);
	    throw new ChannelNotConnected();
	}
	catch(Exception e) {
	    System.err.println(e);
	    return null;
	}
    }




    public Object Peek(long timeout) throws ChannelNotConnected, ChannelClosed,  Timeout {
	Message    retval=null;
	Event      evt;

	CheckConnection();
	CheckClosed();
	    
	if(mq == null)
	    throw new ChannelNotConnected();
	    
	try {
	    evt=(timeout <= 0)? (Event)mq.Peek() : (Event)mq.Peek(timeout);
	    if(evt == null)
		return null; // correct ?

	    switch(evt.GetType()) {
	    case Event.MSG:
		return evt.GetArg();
	    case Event.VIEW_CHANGE:
		return evt.GetArg();
	    case Event.SUSPECT:
		return new SuspectEvent(evt.GetArg());
	    case Event.BLOCK:
		return new BlockEvent();
	    default:
		System.err.println("EnsChannel.Peek(): event is neither message " + 
				   "nor view nor block !");
		return null;
	    }
	}
	catch(Timeout tex) {
	    throw tex;
	}
	catch(QueueClosed queue_closed) {
	    System.err.println("EnsChannel.Peek(): " + queue_closed);
	    throw new ChannelNotConnected();
	}
	catch(Exception e) {
	    System.err.println(e);
	    return null;
	}
    }



    public View    GetView()         {return my_view;}

    
    public Object  GetLocalAddress() {return my_addr;}


    public Object  GetGroupAddress()  {return channel_name;}



    public void    SetOpt(int option, Object value) {
	switch(option) {
	case VIEW:
	    if(value instanceof Boolean)
		receive_views=((Boolean)value).booleanValue();
	    else
		System.err.println("EnsChannel.SetOpt(" + option + ", " + value + "): value has " +
				   "to be Boolean");
	    break;
	case SUSPECT:
	    if(value instanceof Boolean)
		receive_suspects=((Boolean)value).booleanValue();
	    else
		System.err.println("EnsChannel.SetOpt(" + option + ", " + value + "): value has " +
				   "to be Boolean");
	    System.out.println("EnsChannel.SetOpt(): SUSPECT is not supported !");
	    break;
	case BLOCK:
	    if(value instanceof Boolean)
		receive_blocks=((Boolean)value).booleanValue();
	    else
		System.err.println("EnsChannel.SetOpt(" + option + ", " + value + "): value has " +
				   "to be Boolean");
	    if(receive_blocks)
		receive_views=true;
	    break;

	case LOCAL:
	    if(value instanceof Boolean)
		receive_local_msgs=((Boolean)value).booleanValue();
	    else
		System.err.println("EnsChannel.SetOpt(" + option + ", " + value + "): value has " +
				   "to be Boolean");
	    break;
	default:
	    System.err.println("EnsChannel.SetOpt(" + option + ", " + value + 
			       "): option not known");
	    break;
	}
    }



    public Object  GetOpt(int option) {
	switch(option) {
	case VIEW:
	    return new Boolean(receive_views);
	case SUSPECT:
	    return new Boolean(receive_suspects);
	case BLOCK:
	    return new Boolean(receive_blocks);
	case LOCAL:
	    return new Boolean(receive_local_msgs);
	default:
	    System.err.println("EnsChannel.GetOpt(" + option + "): option not known");
	    return null;
	}
    }



    public void    BlockOk() {
	
    }









    /* --------------------- Ensemble Callbacks ------------------------- */

    public void ReceiveCast(Hot_GroupContext gctx, Object env, 
			    Hot_Endpoint origin, Hot_Message msg) {

	Event               evt;
	Hot_ObjectMessage   tmp=new Hot_ObjectMessage(msg);
	Message             m=(Message)tmp.getObject();

	if(m == null) {
	    System.out.println("EnsChannel.ReceiveCast(): received message that is " +
			       "not of type Message. Discarding.");
	    return;
	}

	if(m.GetSrc() == null)
	    m.SetSrc(origin);

	if(!receive_local_msgs) {  // discard local messages (sent by myself to me)
	    if(my_addr != null && m.GetSrc() != null)
		if(my_addr.equals(m.GetSrc()))
		    return;
	}
	try {
	    mq.Add(new Event(Event.MSG, m));
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }



    public void ReceiveSend(Hot_GroupContext gctx, Object env, 
			    Hot_Endpoint origin, Hot_Message msg) {
	ReceiveCast(gctx, env, origin, msg);
    }



    public void AcceptedView(Hot_GroupContext gctx, Object env, Hot_ViewState viewState) {
	View          v;
	Hot_ViewID    vid=viewState.view_id;
	Object        coord=vid.coord;
	long          id=vid.ltime;
	Vector        members=new Vector();


	if(my_addr == null && viewState.members != null && viewState.nmembers == 1) {
	    my_addr=viewState.members[0];
	    System.out.println("My address is " + my_addr);
	}

	for(int i=0; i < viewState.members.length; i++)
	    members.addElement(viewState.members[i]);

	v=new View(coord, id, members);

	if(my_view == null)
	    my_view=v;

	if(!receive_views)
	    return;

	try {
	    mq.Add(new Event(Event.VIEW_CHANGE, v));
	}
	catch(Exception e) {
	    System.err.println("EnsChannel.AceptedView: " + e);
	}
    }



    public void Heartbeat(Hot_GroupContext gctx, Object env, int rate) {}


    public void Block(Hot_GroupContext gctx, Object env) {
	if(!receive_blocks)
	    return;
	try {
	    mq.Add(new Event(Event.BLOCK));
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }


    public void Exit(Hot_GroupContext gctx, Object env) {
	System.out.println("EnsChannel.Exit(): received EXIT message !");
    }


    /* ------------------------------------------------------------------ */
}
