package JavaGroups;

import java.util.*;

/**
 * A channel represents one member of a specific group in a Java VM. Only a single object 
 * at a time is allowed to connect to a channel, but there can be more than one channel in
 * an application (for different or the same groups).<p>
 * Messages can be multicast to all objects connected to all channels with the same name 
 * using the <em>Send</em> method and messages can be received using the
 * <em>Receive</em> method.<p>
 * A channel instance is created using a <em>ChannelFactory</em>. Each implementation of a channel
 * must provide a subclass of <code>Channel</code> and an implementation of <code>ChannelFactory</code>.
 * <p>
 * Various degrees of sophistication in message exchange can be achieved using building
 * blocks on top of channels, e.g. light-weight groups, synchronous message invocation, or
 * remote method calls. Channels are on the same abstraction level as sockets, and should really
 * be simple to use. Higher-level abstractions are all built on top of channels.
 */

public abstract class ChannelNew implements Transportable {

    /** Private constructor. Channels should be created through ChannelFactories.
    private ChannelNew() {}


    /** Connects a client to a channel. The client is now able to receive messages, views and block 
	events (depending on the options set) and to send messages. If the channel is already 
	connected, this is a null operation. All channels with the same name form a group.
	@param group_address The name of the group to be joined.
    */

    abstract public void    Connect(String group_address);




    /** Disconnects a client from a channel. This is a null operation if not connected.
     */

    abstract public void    Disconnect();




    /** Sends a message to a destination. The message contains
	<ol>
	<li>a destination address (Object). If it is null, the message is sent to all members of
	    the group.
	<li>a source address. Can be left empty. Will be filled in by the protocol stack.
	<li>a byte buffer. The message contents.
	<li>several additional fields. They can be used by application programs (or patterns). E.g.
            a message ID, a <code>oneway</code> field which determines whether a response is expected etc.
	</ol>
	@param msg The message to be sent. Destination and buffer should be set. A null destination
	           (default) means send to all members of the group.
	@exception NotConnected The channel is not connected. Messages can only be sent in the
	           connected state.
    */

    abstract public void    Send(Message msg) throws NotConnected;






    /** Receives a message, a view change or a block event. By using <code>SetOpt</code>, 
	the type of objects to be received can be determined (e.g. not views and blocks, just messages).
	The channel must be connected to receive messages. The possible types returned can be:
	<ol>
	<li><code>Message</code>. Normal message
	<li><code>View</code>. A view change.
	<li><code>BlockEvent</code>. A block event indicating an impending view change.
	</ol>
	The <code>instanceof</code> operator can be used to discriminate between different types returned.
	@param timeout Value in milliseconds. Value <= 0 means wait forever
	@return A Message, View or BlockEvent object, depending on what is on top of the internal queue.
	@exception TimeoutException Thrown when a timeout has occurred.
	@exception NotConnected The channel is not connected. Messages can only be received
	           in the connected state.
     */

    abstract public Object  Receive(long timeout) throws NotConnected, TimeoutException;





    /** Gets the current view. This does <em>not</em> retrieve a new view, use <code>Receive</code>
	to do so. The view may only be available after a successful <code>Connect</code>. The result of
	calling	this method on an unconnected channel is implementation defined (may return null).
	@return The current view.
     */

    abstract public View    GetView();






    /** 
	Returns the channel's address. The result of calling this method on an unconnected channel 
	is implementation defined (may return null).
	@return The channel's address. Generated by the underlying transport, and opaque.
	        Addresses can be used as destination in the <code>Send</code> operation.
    */

    abstract public Object  GetAddress();








    /** 
	Sets an option. The following options are currently recognized:
	<ol>
	<li><code>BLOCK</code> (0). Turn the reception of BLOCK events on/off (value is Boolean).
	    Default is off. If set to on, receiving VIEW events will be set to on, too.
	<li><code>VIEW</code>  (1). Turn the reception of VIEW events on/off  (value is Boolean).
	    Default is off.
	<li><code>LOCAL</code> (2). Receive its own broadcast messages to the group (value is Boolean).
            Default is on.
	</ol>
	This method can be called on an unconnected channel.
     */

    abstract public void    SetOpt(int option, Object value);





    /** 
	Gets an option. This method can be called on an unconnected channel.
	@param option  The option to be returned.
	@return The object associated with an option.
     */
    abstract public Object  GetOpt(int option);





    /** Called to acknowledge a Block (callback in <code>MembershipListener</code>). 
	After sending BlockOk, no messages should be sent until a new view has been received. 
    */

    abstract public void    BlockOk();
}
