package JavaGroups;

import java.net.*;
import java.util.Vector;
import JavaGroups.*;
import JavaGroups.JavaStack.Address;

/**
   Keeps track of messages according to their sequence numbers. Allows messages to be added out of
   order, and with gaps between sequence numbers. Method <code>Remove</code> removes the first
   message with a sequence number that is 1 higher than <code>next_to_remove</code> (this variable
   is then incremented), or it returns null if no message is present, or if no message's sequence
   number is 1 higher.
 */
public class SlidingWindow implements Runnable {


    /**
       Retransmit command (see Gamma et al.) used by the sliding window table to retrieve missing
       messages. It tries <em>twice</em> to retransmit by sending a pt2pt message to the sender
       of the message. If this does not furnish the message, a multicast message is sent to the 
       group (<em>just once</em>). If this does still not furnish the message, the sender of the
       message is suspected, and subsequently removed, and its entry removed from the sliding window
       table.
     */
    public interface RetransmitCommand {

	/**
	   Get the missing message with sequence number <code>seq_no</code> by sending a retransmit
	   message to destination <code>dest</code>.
	   @param seq_no The sequence number of the missing message
	   @param sender The destination of the object to which the retransmit request is sent.
	   @param mcast_to_group Mcast to group instead of <code>sender</code> to get message
	   @param suspect The <code>sender</code> does not respond, should be removed from the group
	 */
	public void Retransmit(long seq_no, Object sender, boolean mcast_to_group, boolean suspect);
    }


    class Entry {	
	long      seq_no=0;
	long      timestamp=System.currentTimeMillis();
	Message   msg;
	int       num_tries=0;

	Entry(long seq_no, Message msg) {
	    this.seq_no=seq_no;
	    this.msg=msg;
	}

	public String toString() {
	    StringBuffer ret=new StringBuffer();
	    ret.append(seq_no);
	    if(msg == null)
		ret.append("-");
	    else
		ret.append("+");
	    return ret.toString();
	}
    }



    private Object              sender=null;
    private long                next_to_remove=1, highest=1;
    private Vector              msgs=new Vector();
    private RetransmitCommand   command=null;
    private long                retransmit_period=3000; // may be dynamically adjusted
    private int                 max_num_tries=2;        // number of retries before mcasting
    private Thread              retransmit_thread=null;


    public SlidingWindow() {}


    public SlidingWindow(Object sender, RetransmitCommand command) {
	this.sender=sender;
	this.command=command;
	Start();
    }


    public void finalize() {
	Stop();
    }
    

    public void Start() {
	if(retransmit_thread == null) {
	    retransmit_thread=new Thread(this, "SlidingWindow.RetransmitThread");
	    retransmit_thread.start();
	}
    }

    public void Stop() {
	if(retransmit_thread != null) {
	    retransmit_thread.stop();
	    retransmit_thread=null;
	}
    }


    /**
       Adds a message according to its sequence number (ordered). Fill gaps with entries which have
       'null' messages.
     */
    public void Add(long seq_no, Message msg) {
	Entry    e=new Entry(seq_no, msg), current=null;
	boolean  inserted=false;
	
	if(seq_no < next_to_remove) {
	    System.err.println("SlidingWindow.Add(): seq_no " + seq_no + " is smaller than the " +
			       "threshold (" + next_to_remove + "); discarding message");
		return;
	}

	synchronized(msgs) {
	    if(seq_no >= highest) {
		for(long i=highest; i <= seq_no; i++) {
		    if(i == seq_no) {
			msgs.addElement(new Entry(i, msg));  // add at end
		    }
		    else {
			msgs.addElement(new Entry(i, null));
		    }
		}
		highest=seq_no+1;
		msgs.notify();                       // notify retransmit thread
	    }
	    else if(msg != null) {  // insert msg into existing slot
		for(int i=0; i < msgs.size(); i++) {
		    current=(Entry)msgs.elementAt(i);

		    if(seq_no == current.seq_no) {
			if(current.msg == null) {
			    current.msg=msg;
			    msgs.notify();           // notify retransmit thread
			}
			return;
		    }

		}
	    }

	}

    }






    /** 
	Removes an entry if its sequence number if 1 higher than <code>next_to_remove</code>.
	Returns null otherwise 
    */
    public Message Remove() {
	Entry    e;
	Message  retval=null;

	synchronized(msgs) {
	    if(msgs.size() == 0)
		return null;
	    e=(Entry)msgs.elementAt(0);
	    if(e.seq_no == next_to_remove && e.msg != null) {
		retval=e.msg;
		msgs.removeElementAt(0);
		next_to_remove++;
		return retval;
	    }
	    return retval;
	}
    }


    /** Deletes all entries */
    public void Reset() {
	synchronized(msgs) {
	    msgs.removeAllElements();
	}
	next_to_remove=0;
    }


    public String toString() {return msgs.toString();}


    /**
       Check for each message whether it needs to be retransmitted (by checking age of
       msg). If this is the case, call <code>retransmit_command</code> with the
       corresponding parameters (pt2pt, mcast or suspect). While iterating through msgs,
       compute the min. wait time for all messages. At the end of an iteration, sleep for
       that period of time. This gives the chance to add / remove messages (otherwise this
       would be blocked).  
    */
    public void run() {
	Entry   e;
	long    current_time=0, min_time, time_to_wait;

	while(true) {
	    synchronized(msgs) {
		while(msgs.size() < 1) {
		    try {
			msgs.wait();
		    }
		    catch(Exception ex) {}
		}
		while(msgs.size() > 0) {
		    min_time=retransmit_period + 1000;  // greater than any possible timeout value
		    for(int i=0; i < msgs.size(); i++) {
			e=(Entry)msgs.elementAt(i);
			if(e.msg != null)
			    continue;
			current_time=System.currentTimeMillis();
			time_to_wait=current_time - e.timestamp;
			min_time=Math.min(min_time, time_to_wait);
			if(time_to_wait >= retransmit_period) {
			    if(e.num_tries < max_num_tries) {        // send pt2pt
				command.Retransmit(e.seq_no, sender, false, false);
			    }
			    else if(e.num_tries == max_num_tries) {  // mcast to group
				command.Retransmit(e.seq_no, sender, true, false);
			    }
			    else {                                   // suspect
				command.Retransmit(e.seq_no, sender, false, true);
			    }
			    e.timestamp=System.currentTimeMillis();
			    e.num_tries++;
			}
		    }
		    if(min_time > 0) {
			try {
			    msgs.wait(min_time); // gives others a chance to add / remove messages
			}
			catch(Exception ee) {}
		    }
		}
	    }
	}
	

    }







    public static void main(String args[]) {


	class Retransmit implements RetransmitCommand {
	    SlidingWindow win=null;

	    void SetWindow(SlidingWindow w) {win=w;}

	    public synchronized void Retransmit(long seq_no, Object sender, 
						boolean mcast_to_group, boolean suspect) {
		System.out.println("RetransmitRequest: seq_no=" + seq_no + ", sender=" + sender +
				   ", mcast=" + mcast_to_group + ", suspect=" + suspect);
		win.Add(seq_no, new Message(null, null, null));
	    }

	}


	class Remover extends Thread {
	    SlidingWindow w=null;
	    Remover(SlidingWindow w) {this.w=w;}
	    public void run() {
		Message m;
		while(true) {
		    while((m=w.Remove()) != null)
		    try {
			sleep(3000);
		    }
		    catch(Exception e) {
			System.err.println(e);
		    }
		}
	    }
	}


	SlidingWindow w;
	Message       msg=new Message(null, null, null);
	Address       sender=null;
	Retransmit    retransmitter=new Retransmit();

	try {
	    sender=new Address(InetAddress.getByName("janet"), 5555);
	}
	catch(Exception e) {
	    System.err.println(e);
	}

	w=new SlidingWindow(sender, retransmitter);
	retransmitter.SetWindow(w);
	

	new Remover(w).start();


	for(int i=0; i < 10; i++) {
	    try {
		if(i % 7 == 0)
		    continue;
		w.Add(i, msg);
		System.out.println("Adding " + i);
		Thread.currentThread().sleep(1000);
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	}


    }
    
}
