package JavaGroups;


import java.io.Serializable;
import java.util.*;
import JavaGroups.JavaStack.*;


class RequestCorrelatorHeader implements Serializable {
    public static final int REQ = 0;
    public static final int RSP = 1;

    int      type=REQ;
    long     id=0;
    boolean  rsp_expected=true;
    String   name=null;

    public RequestCorrelatorHeader(int type, long id, boolean rsp_expected, String name) {
	this.type=type; this.id=id; this.rsp_expected=rsp_expected; this.name=name;
    }

    public String toString() {
	StringBuffer ret=new StringBuffer();
	ret.append("[RequestCorrelatorHeader: name=" + name + ", type=");
	ret.append(type == REQ ? "REQ" : type == RSP ? "RSP" : "<unknown>");
	ret.append(", id=" + id);
	ret.append(", rsp_expected=" + rsp_expected + "]");
	return ret.toString();
    }
}






/**
   Framework to send requests and receive matching responses (matching on request ID).
   Multiple requests can be sent at a time. Whenever a response is received, the correct
   <code>RspCollector</code> is looked up (key = id) and its method 
   <code>ReceiveResponse</code> invoked. A caller may use <code>Done</code> to signal that no
   more responses are expected, and that the corresponding entry may be removed.<p>
   <code>RequestCorrelator</code> can be installed at both client and server sides, 
   it can also switch roles dynamically, i.e. send a request and at the same time process 
   an incoming request (when local delivery is enabled, this is actually the default).
 */
public class RequestCorrelator implements Runnable {
    protected Transport       transport=null;
    protected Vector          requests=new Vector();
    protected RequestHandler  request_handler=null;
    protected Queue           mq=new Queue();
    protected Thread          request_handler_thread=null;
    protected String          name=null; // makes the instance unique (together with IDs)
    protected boolean         multithreaded=true;


    class RequestEntry {
	long           id=0;
	RspCollector   coll=null;

	RequestEntry(long id, RspCollector coll) {
	    this.id=id; this.coll=coll;
	}

	public boolean equals(Object other) {
	    if(other == null || !(other instanceof RequestEntry))
		return false;
	    return id == ((RequestEntry)other).id ? true : false;
	}
    }


    private void AddEntry(RequestEntry entry) {
	synchronized(requests) {
	    if(!requests.contains(entry))
		requests.addElement(entry);
	    else
		System.err.println("RequestCorrelator.AddEntry(): entry " + 
				   entry + " already present !");
	}
    }


    private RspCollector FindEntry(long id) {
	RequestEntry entry;

	synchronized(requests) {
	    for(int i=0; i < requests.size(); i++) {
		entry=(RequestEntry)requests.elementAt(i);
		if(entry.id == id)
		    return entry.coll;
	    }
	}
	return null;
    }









    /**
       Constructor. Uses transport to send messages. If <code>handler</code> is not null,
       all incoming requests will be dispatched to it (via <code>Handle(Message)</code>).
       @param name Used to differentiate between diferent RequestCorrelators (e.g. in 
                   different protocol layers). Has to be unique if multiple request
		   correlators are used.
       @param transport Use to send requests. Only <code>Send</code> is used, not
                        <code>Receive</code> !
       @param handler Request handler. Method <code>Handle(Message)</code> will be called
                      when a request is received.
     */
    public RequestCorrelator(String name, Transport transport, RequestHandler handler) {
	this.name=name;
	this.transport=transport;
	request_handler=handler;
	if(handler != null)
	    Start();
    }


    /**
       Constructor. Uses transport to send messages. If <code>handler</code> is not null,
       all incoming requests will be dispatched to it (via <code>Handle(Message)</code>).
       @param name Used to differentiate between diferent RequestCorrelators (e.g. in 
                   different protocol layers). Has to be unique if multiple request
		   correlators are used.
       @param transport Use to send requests. Only <code>Send</code> is used, not
                        <code>Receive</code> !
       @param handler Request handler. Method <code>Handle(Message)</code> will be called
                      when a request is received.
       @param multithreaded If true (default) a request will be put in a request queue where it
                            is handled by a separater request thread (1 thread for all requests).
			    This can be turned off, e.g. when ordering is important. If turned off,
			    each request is handled by the receiver thread directly, without being
			    put in the request queue (and no request handler thread will be started).
			    If turned off, deadlock situations might arise, e.g. when the servicing of
			    a request in user code triggers another (synchronous) request to itself.
     */
    public RequestCorrelator(String name, Transport transport, RequestHandler handler, 
			     boolean multithreaded) {
	this.name=name;
	this.transport=transport;
	request_handler=handler;
	this.multithreaded=multithreaded;
	if(handler != null)
	    Start();
    }



    public void finalize() {
	Stop();
    }



    public void SetRequestHandler(RequestHandler handler) {
	request_handler=handler;
	Start();
    }


   
    /**
       Send a request to a group. If no response collector is given, no responses are
       expected (making the call asynchronous).
       @param id The request ID. Must be unique for this JVM (e.g. current time in millisecs)
       @param msg The request to be sent. The body of the message carries the request data
       @param coll A response collector (usually the object that invokes this method). Its methods
       <code>ReceiveResponse</code> and <code>Suspect</code> will be invoked when
       a message has been received or a member is suspected, respectively.
    */
    public void SendRequest(long id, Message msg, RspCollector coll) {
	RequestCorrelatorHeader hdr;

	hdr=new RequestCorrelatorHeader(RequestCorrelatorHeader.REQ, id, 
					coll != null ? true : false, name);
	msg.AddHeader(hdr);

	if(coll != null)
	    AddEntry(new RequestEntry(hdr.id, coll));
	
	if(transport != null) {
	    try {
		transport.Send(msg);
	    }
	    catch(Exception e) {}
	}
	else
	    System.err.println("RequestCorrelator.SendRequest(): protocol is not available !");
    }





    /**
       Used to signal that a certain request may be garbage collected as all responses
       have been received.
    */
    public void Done(long id) {
	RequestEntry entry;

	synchronized(requests) {
	    for(int i=0; i < requests.size(); i++) {
		entry=(RequestEntry)requests.elementAt(i);
		if(entry.id == id) {
		    requests.removeElementAt(i);
		    break;
		}
	    }
	}
    }





    /**
       <b>Callback</b>. All response collectors currently registered will be notified that
       <code>mbr</code> may have crashed, so they won't wait for its response.
    */
    public void Suspect(Object mbr) {
	RequestEntry entry;

	synchronized(requests) {
	    for(int i=0; i < requests.size(); i++) {
		entry=(RequestEntry)requests.elementAt(i);
		if(entry.coll != null)
		    entry.coll.Suspect(mbr);
	    }
	}
    }





    /**
       <b>Callback</b>. Called by the protocol below when a message has been received.
       The algorithm should test whether the message is destined for us and, if not,
       return false.  Otherwise, it should remove the header and check whether the
       message is a request or response. In the first case, the message will be
       delivered to the request handler registered (calling its <code>Handle</code>
       method), in the second case, the corresponding response collector is looked up
       and the message delivered.  
    */
    public boolean Receive(Message msg) {
	Object                    tmp;
	RequestCorrelatorHeader   hdr;


	tmp=msg.PeekHeader();
	if(tmp == null || !(tmp instanceof RequestCorrelatorHeader))
	    return false;  // message is not handled by us (e.g. unicast message) !

	hdr=(RequestCorrelatorHeader)msg.PeekHeader();

	// check whether the message was sent by a request correlator with the same name
	// (there may be multiple request correlators in the same protocol stack...)
	if(hdr.name == null || !hdr.name.equals(name)) {
	    System.err.println("RequestCorrelator.Receive(): name of request correlator header (" +
			       hdr.name + ") is different from ours (" + name + "). Msg not accepted");
	    return false;
	}

	switch(hdr.type) {
	case RequestCorrelatorHeader.REQ:
	    if(request_handler == null) {
		System.err.println("RequestCorrelator.Receive(): there is no request handler " +
				   "installed to deliver request !");
		return true;  // message was handled !
	    }
	    if(!multithreaded) {
		//request_handler.Handle(msg);
		HandleRequest(msg);
	    }
	    else {
		try {
		    mq.Add(msg);
		}
		catch(Exception e) {}
	    }
	    break;
	case RequestCorrelatorHeader.RSP:
	    msg.RemoveHeader();
	    RspCollector coll=FindEntry(hdr.id);
	    if(coll != null)
		coll.ReceiveResponse(msg);
	    break;
	default:
	    msg.RemoveHeader();
	    System.err.println("RequestCorrelator.Receive(): header's type is " +
			       "neither REQ nor RSP !");
	    break;
	}
	return true;  // message was handled by us
    }




    /**
       Remove requests from mq and invoke request_handler.Handle(). Only started if 
       <code>multithreaded</code> is true.
    */
    public void run() {
	Message  req=null;

	while(true) {
	    try {
		req=(Message)mq.Remove();
		HandleRequest(req);
	    }
	    catch(QueueClosed e) {
		break;
	    }
	}
    }



    void HandleRequest(Message req) {
	Object                   retval;
	byte[]                   rsp_buf;
	RequestCorrelatorHeader  hdr, rsp_hdr;
	Message                  rsp=null;


	hdr=(RequestCorrelatorHeader)req.RemoveHeader();
	
	retval=request_handler.Handle(req);
	rsp_buf=null;
	
	if(hdr.rsp_expected) {
	    rsp=req.MakeReply();
	    
	    try {
		rsp_buf=retval != null ? Util.ObjectToByteBuffer((Serializable)retval) : null;
	    }
	    catch(Exception e) {
		System.err.println("RequestCorrelator.HandleRequest(): failure sending response: " +
				   "return value is not serializable. Sending null value");
		rsp_buf=null;
	    }
	    if(rsp_buf != null)
		rsp.SetBuffer(rsp_buf);
	    
	    rsp_hdr=new RequestCorrelatorHeader(RequestCorrelatorHeader.RSP, hdr.id, false, name);
	    rsp.AddHeader(rsp_hdr);
	    
	    if(transport != null) {
		try {
		    transport.Send(rsp);
		}
		catch(Exception e) {}
	    }
	    else
		System.err.println("RequestCorrelator.run(): failure sending response. " + 
				   "No protocol available ! ");
	}
    }



    public void Start() {
	if(multithreaded == false) return;
	if(request_handler_thread == null) {
	    request_handler_thread=new Thread(this, "RequestCorrelator.RequestHandlerThread");
	    request_handler_thread.start();
	}
    }


    public void Stop() {
	if(request_handler_thread != null) {
	    request_handler_thread.stop();
	    request_handler_thread=null;
	}
    }






    public static void main(String args[]) {
	RequestCorrelatorHeader hdr, hdr2, hdr3;
	Vector v=new Vector();

	hdr=new RequestCorrelatorHeader(RequestCorrelatorHeader.REQ, 
					System.currentTimeMillis(), false, "bla");
	v.addElement(hdr);


	hdr2=new RequestCorrelatorHeader(RequestCorrelatorHeader.REQ, 
					 System.currentTimeMillis(), false, "bla");
	hdr2.id=hdr.id+1;
	v.addElement(hdr2);

	System.out.println(v);

	hdr3=new RequestCorrelatorHeader(RequestCorrelatorHeader.RSP, 
					 System.currentTimeMillis(), false, "bla");

	if(v.contains(hdr3))
	    System.out.println("Contains");
	else
	    System.out.println("Does no contain");

	
    }
}
