package JavaGroups.JavaStack;

import java.util.*;
import JavaGroups.*;

/**
 * A ProtocolStack manages a number of protocols layered above each other. It creates all
 * protocol classes, initializes them and, when ready, starts all of them, beginning with the 
 * bottom most protocol. It also dispatches messages received from the stack to registered
 * objects (e.g. channel, GMP) and sends messages sent by those objects down the stack.<p>
 * The ProtocolStack makes use of the Configurator to setup and initialize stacks, and to
 * detroy them again when not needed anymore.<p>
 * <b>The bottommost instance has to implement interface <code>LowLevelCommunication</code></b>.
 * This is necessary to be able to retrieve one's own (transport specific) address, join multicast
 * addresses etc.
 */
public class ProtocolStack extends Protocol implements Transport {
    private Protocol                top_prot=null;
    private Protocol                bottom_prot;
    private Configurator            conf=new Configurator();
    private String                  setup_string;
    private JChannel                channel=null;
    private Integer                 start_mutex=new Integer(0);
    private Integer                 stop_mutex=new Integer(0);
    private boolean                 start_ok_received=false;
    private Object                  local_addr=null;


    
    public ProtocolStack(JChannel channel, String setup_string) {
	this.setup_string=setup_string;
	this.channel=channel;
    }
    

    public void Setup() throws Exception {
	if(top_prot == null) {
	    top_prot=conf.SetupProtocolStack(setup_string, this);
	    if(top_prot == null)
		throw new Exception("ProtocolStack.Setup(): couldn't create protocol stack");
	    top_prot.SetUpProtocol(this);
	    bottom_prot=conf.GetBottommostProtocol(top_prot);
	    conf.StartProtocolStack(bottom_prot);        // sets up queues and threads
	}
    }



    public void Destroy() throws Exception {
	if(top_prot != null) {
	    conf.StopProtocolStack(top_prot);           // destroys msg queues and threads
	    top_prot=null;
	}	
    }


    
    /**
       Start all layers. A START event is sent from the bottom layer up the stack. Each layer
       can perform some initialization.
     */
    public void Start() throws Exception {
	synchronized(start_mutex) {
	    bottom_prot.Up(new Event(Event.START));
	    start_mutex.wait(5000);                  // will be notified by START_OK
	}
	if(start_ok_received == false)
	    throw new ChannelException("ProtocolStack.Start(): timed out waiting for START_OK");
    }



    public void StartInternal() {
	// do nothing, DON'T REMOVE !!!!
    }

    
    /** Passes a STOP event down the stack, waits for the STOP_OK event from the bottom layer.
	Clears all message queues */
    public void Stop() {
	synchronized(stop_mutex) {
	    top_prot.Down(new Event(Event.STOP));
	    try {
		stop_mutex.wait(5000);               // will be notified by STOP_OK
	    }
	    catch(Exception e) {
	    }
	}
    }

    public void StopInternal() {
	// do nothing, DON'T REMOVE !!!!
    }





    /*--------------------------- Transport interface ------------------------------*/

    public void     Send(Message msg) throws Exception {
	Down(new Event(Event.MSG, msg));
    }

    public Object   Receive(long timeout) throws Exception {
	throw new Exception("ProtocolStack.Receive(): not implemented !");
    }
    /*------------------------- End of  Transport interface ---------------------------*/





    /*--------------------------- Protocol functionality ------------------------------*/
    public String GetName()  {return "ProtocolStack";}




    public void Up(Event evt) {
	switch(evt.GetType()) {
	    
	case Event.SET_LOCAL_ADDRESS:
	    local_addr=evt.GetArg();
	    break;

	case Event.START_OK:
	    synchronized(start_mutex) {
		start_ok_received=true;
		start_mutex.notify();
	    }
	    return;
	case Event.STOP_OK:
	    synchronized(stop_mutex) {
		stop_mutex.notify();
	    }
	    return;
	}
	if(channel != null)
	    channel.Up(evt);
    }




    public void Down(Event evt) {
	if(top_prot != null)
	    top_prot.ReceiveDownEvent(evt);
	else
	    System.err.println("ProtocolStack.Down() at " + ": no down protocol available !");
    }



    protected void ReceiveUpEvent(Event evt) {
	Up(evt);
    }


    
    /** Override with null functionality: we don't need any threads to be started ! */
    public void StartWork() {}

    /** Override with null functionality: we don't need any threads to be started ! */
    public void StopWork()  {}


    /*----------------------- End of Protocol functionality ---------------------------*/




}
