package JavaGroups;

import java.io.*;
import java.util.Vector;


/**
 * Collection of various utility routines that can not be assigned to other classes.
 */
public class Util {


    /**
     * Creates an object from a byte buffer
     */
    public static Object ObjectFromByteBuffer(byte[] buffer) throws Exception {
	Object               retval=null;
	if(buffer == null) return null;
	ByteArrayInputStream in_stream=new ByteArrayInputStream(buffer);
	ObjectInputStream    in=new ObjectInputStream(in_stream);	
	retval=in.readObject();
	if(retval == null)
	    return null;
	return retval;
    }

    /**
     * Serializes an object into a byte buffer. 
     * The object has to implement interface Serializable.
     */
    public static byte[] ObjectToByteBuffer(Object obj) throws Exception {
	ByteArrayOutputStream out_stream=new ByteArrayOutputStream();
	ObjectOutputStream    out=new ObjectOutputStream(out_stream);	
	out.writeObject(obj);
	return out_stream.toByteArray();
    }


    public static void Sleep(long timeout) {
	try {
	    Thread.currentThread().sleep(timeout);
	}
	catch(Exception e) {}
    }


    /** Sleeps between 1 and timeout milliseconds, chosen randomly. Timeout must be > 1 */
    public static void SleepRandom(long timeout) {
	if(timeout <= 1) {
	    System.err.println("Util.SleepRandom(" + timeout + "): timeout must be > 1 !");
	    return;
	}

	long r=(int)((Math.random() * 100000) % timeout) +1;
	try {
	    Thread.currentThread().sleep(r);
	}
	catch(Exception e) {}
    }


    public static void DumpStack(boolean exit) {
	try {
	    throw new Exception("Dumping stack:");
	}
	catch(Exception e) {
	    e.printStackTrace();
	    if(exit)
		System.exit(0);
	}
    }


    public static void Crash() {
	System.exit(-1);
    }

    
    public static String PrintEvent(Event evt) {
	Message msg;

	if(evt.GetType() == Event.MSG) {
	    msg=(Message)evt.GetArg();
	    if(msg != null && msg.GetBuffer() != null)
		return PrintMessage(msg);
	    else
		return msg.PrintObjectHeaders();
	}
	return evt.toString();
    }

    
    /** Tries to read an object from the message's buffer and prints it */
    public static String PrintMessage(Message msg) {
	Object obj;

	if(msg == null)
	    return "";
	byte[] buf=msg.GetBuffer();
	if(buf == null)
	    return null;

	try {
	    obj=ObjectFromByteBuffer(buf);
	    return obj.toString();
	}
	catch(Exception e) {  // it is not an object
	    return "";
	}
	
    }



    
    /** Tries to read a <code>MethodCall</code> object from the message's buffer and prints it.
     Returns empty string if object is not a method call */
    public static String PrintMethodCall(Message msg) {
	Object obj;
	if(msg == null)
	    return "";
	byte[] buf=msg.GetBuffer();
	if(buf == null)
	    return "";

	try {
	    obj=ObjectFromByteBuffer(buf);
	    return obj.toString();
	}
	catch(Exception e) {  // it is not an object
	    return "";
	}
	
    }



    public static void Print(String msg) {
	//synchronized(System.out) {
	    System.out.println(msg); System.out.flush();
	    //}
    }



    public static void PrintThreads() {
	Thread threads[]=new Thread[Thread.activeCount()];
	int num=Thread.enumerate(threads);
	System.out.println("------- Threads -------");
	for(int i=0; i < threads.length; i++) {
	    System.out.println("#" + i + ": " + threads[i]);
	}
	System.out.println("------- Threads -------\n");
    }



    /**
       Fragments a byte buffer into smaller fragments of (max.) frag_size.
       Example: a byte buffer of 1024 bytes and a frag_size of 248 gives 4 fragments
       of 248 bytes each and 1 fragment of 32 bytes.
       @return An array of byte buffers (<code>byte[]</code>).
     */
    public static byte[][] FragmentBuffer(byte[] buf, int frag_size) {
	byte[]                retval[];
	long                  total_size=buf.length;
	int                   accumulated_size=0;
	byte[]                fragment;
	int                   tmp_size=0;
	int                   num_frags;
	int                   index=0;
	
	num_frags=buf.length % frag_size == 0 ? buf.length / frag_size : buf.length / frag_size +1;
	retval=new byte[num_frags][];
	
	while(accumulated_size < total_size) {
	    if(accumulated_size + frag_size <= total_size)
		tmp_size=frag_size;
	    else
		tmp_size=(int)(total_size - accumulated_size);
	    fragment=new byte[tmp_size];
	    System.arraycopy(buf, accumulated_size, fragment, 0, tmp_size);
	    retval[index++]=fragment;
	    accumulated_size+=tmp_size;
	}
	return retval;
    }



    /**
       Concatenates smaller fragments into entire buffers.
       @param fragments An array of byte buffers (<code>byte[]</code>)
       @return A byte buffer
     */
    public static byte[] DefragmentBuffer(byte[] fragments[]) {
	int                   total_length=0;
	byte[]                ret;
	int                   index=0;

	if(fragments == null) return null;
	for(int i=0; i < fragments.length; i++) {
	    if(fragments[i] == null) continue;
	    total_length+=fragments[i].length;
	}
	ret=new byte[total_length];
	for(int i=0; i < fragments.length; i++) {
	    if(fragments[i] == null) continue;
	    System.arraycopy(fragments[i], 0, ret, index, fragments[i].length);
	    index+=fragments[i].length;
	}
	return ret;
    }



    public static void PrintFragments(byte[] frags[]) {
	for(int i=0; i < frags.length ;i++)
	    System.out.println("'" + new String(((byte[])frags[i])) + "'");
    }


    
    /**
       Peeks for view on the channel until n views have been received or timeout has elapsed.
       Used to determine the view in which we want to start work. Usually, we start as only
       member in our own view (1st view) and the next view (2nd view) will be the full view
       of all members, or a timeout if we're the first member. If a non-view (a message or
       block) is received, the method returns immediately.
       @param channel The channel used to peek for views. Has to be operational.
       @param number_of_views The number of views to wait for. 2 is a good number to ensure that,
              if there are other members, we start working with them included in our view.
       @param timeout Number of milliseconds to wait until view is forced to return. A value
              of <= 0 means wait forever.
     */
    public static View PeekViews(Channel channel, int number_of_views, long timeout) {
	View     retval=null;
	Object   obj=null;
	int      num=0;
	long     start_time=System.currentTimeMillis();

	if(timeout <= 0) {
	    while(true) {
		try {
		    obj=channel.Peek(0);
		    if(obj == null || !(obj instanceof View))
			break;
		    else {
			retval=(View)channel.Receive(0);
			num++;
			if(num >= number_of_views)
			    break;
		    }
		}
		catch(Exception ex) {
		    break;
		}
	    }
	}
	else {
	    while(timeout > 0) {
		try {
		    obj=channel.Peek(timeout);
		    if(obj == null || !(obj instanceof View))
			break;
		    else {
			retval=(View)channel.Receive(timeout);
			num++;
			if(num >= number_of_views)
			    break;
		    }
		}
		catch(Exception ex) {
		    break;
		}
		timeout=timeout - (System.currentTimeMillis() - start_time);
	    }
	}

	return retval;
    }




    public static String Array2String(long[] array) {
	StringBuffer ret=new StringBuffer("[");

	if(array != null) {
	    for(int i=0; i < array.length; i++)
		ret.append(array[i] + " ");
	}

	ret.append("]");
	return ret.toString();
    }

    public static String Array2String(int[] array) {
	StringBuffer ret=new StringBuffer("[");

	if(array != null) {
	    for(int i=0; i < array.length; i++)
		ret.append(array[i] + " ");
	}

	ret.append("]");
	return ret.toString();
    }

    public static String Array2String(boolean[] array) {
	StringBuffer ret=new StringBuffer("[");

	if(array != null) {
	    for(int i=0; i < array.length; i++)
		ret.append(array[i] + " ");
	}
	ret.append("]");
	return ret.toString();
    }

    
    /**
       Selects a random subset of members according to subset_percentage and returns them.
       Picks no member twice from the same membership. If the percentage is smaller than 1 -> picks 1 member.
     */
    public static Vector PickSubset(Vector members, double subset_percentage) {
	Vector ret=new Vector(), tmp_mbrs;
	int    num_mbrs=members.size(), subset_size, index;

	if(num_mbrs == 0) return ret;
	subset_size=(int)Math.ceil(num_mbrs * subset_percentage);

	tmp_mbrs=(Vector)members.clone();
	
	for(int i=subset_size; i > 0 && tmp_mbrs.size() > 0; i--) {
	    index=(int)((Math.random() * num_mbrs) % tmp_mbrs.size());
	    ret.addElement(tmp_mbrs.elementAt(index));
	    tmp_mbrs.removeElementAt(index);
	}

	return ret;
    }





    public static void main(String args[]) {
  	byte[]  buf=new String("Hello world from Bela Ban and Jeannette Fechner").getBytes();
  	byte[]  fragments[]=FragmentBuffer(buf, 8);

  	System.out.println("length: " + buf.length);
  	System.out.println(fragments.length + " fragments");
  	PrintFragments(fragments);


  	byte[] new_buf=DefragmentBuffer(fragments);
  	System.out.println("Result is: '" + new String(new_buf) + "' (Length: " + 
  			   new_buf.length + ")");


	Vector mbrs=new Vector();
	for(int i=0; i < 3; i++)
	    mbrs.addElement(new Long(i));
	Vector ret=Util.PickSubset(mbrs, 0.1);
	System.out.println("mbrs=" + mbrs + "\nsubset=" + ret);



    }

}
