package JavaGroups.JavaStack;


import java.util.*;
import JavaGroups.*;



class MessageProtocolHeader implements java.io.Serializable {
    String name;
    public MessageProtocolHeader(String name) {this.name=name;}
    public String toString() {return "MessageProtocolHeader(name=" + name + ")";}
}




/**
   Based on Protocol, but incorporates RequestCorrelator and GroupRequest: the latter can 
   be used to mcast messages to all members and receive their reponses.<p>
   A protocol based on this template can send messages to all members and receive all, a single,
   n, or none responses. Requests directed towards the protocol can be handled by overriding
   method <code>Handle</code>.<p>
   Requests and responses are in the form of <code>Message</code>s, which would typically need to
   contain information pertaining to the request/response, e.g. in the form of objects contained
   in the message. To use remote method calls, use <code>RpcProtocol</code> instead.<p>
   Typical use of of a <code>MessageProtocol</code> would be when a protocol needs to interact with
   its peer protocols at each of the members' protocol stacks. A simple protocol like fragmentation,
   which does not need to interact with other instances of fragmentation, may simply subclass
   <code>Protocol</code> instead.
 */
public abstract class MessageProtocol extends Protocol implements RequestHandler {
    protected RequestCorrelator   _corr=null;
    protected Vector              members=new Vector();
        



   /**
       Cast a message to all members, and wait for <code>mode</code> responses. The responses are
       returned in a response list, where each response is associated with its sender.<p>
       Uses <code>GroupRequest</code>.
       @param dests The members from which responses are expected. If it is null, replies from all members
                    are expected. The request itself is multicast to all members.
       @param msg The message to be sent to n members
       @param mode Defined in <code>GroupRequest</code>. The number of responses to wait for:
                   <ol>
		   <li>GET_FIRST: return the first response received.
		   <li>GET_ALL: wait for all responses (minus the ones from suspected members)
		   <li>GET_MAJORITY: wait for a majority of all responses (relative to the grp size)
		   <li>GET_ABS_MAJORITY: wait for majority (absolute, computed once)
		   <li>GET_N: wait for n responses (may block if n > group size)
		   <li>GET_NONE: wait for no responses, return immediately (non-blocking)
		   </ol>
       @param timeout If 0: wait forever. Otherwise, wait for <code>mode</code> responses 
                      <em>or</em> timeout time.
       @return RspList A list of responses. Each response is an <code>Object</code> and associated
                       to its sender.
   */    
    public RspList CastMessage(Vector dests, Message msg, int mode, long timeout) {
	GroupRequest  _req=null;
	Vector        real_dests=dests != null ? dests : members;
	
	// This marks message as sent by us ! (used in Up()
	// msg.AddHeader(new MsgProtHeader(GetName()));   ++ already done by RequestCorrelator

	_req=new GroupRequest(msg, _corr, real_dests, mode, timeout, 0);
	_req.Execute();

	return _req.GetResults();
    }





    
    /**
       Sends a message to a single member (destination = msg.dest) and returns the response. 
       The message's destination must be non-zero !
    */
    public Object SendMessage(Message msg, int mode, long timeout) throws Timeout, Suspected {
	Vector        mbrs=new Vector();
	RspList       rsp_list=null;
	Object        dest=msg.GetDest();
	Rsp           rsp;
	Object        retval=null;
	GroupRequest  _req=null;

	if(dest == null) {
	    System.out.println("MessageProtocol.SendMessage(): the message's destination is null ! " +
			       "Cannot send message !");
	    return null;
	}

	
	mbrs.addElement(dest);   // dummy membership (of destination address)	


	_req=new GroupRequest(msg, _corr, mbrs, mode, timeout, 0);
	_req.Execute();

	if(mode == GroupRequest.GET_NONE)
	    return null;


	rsp_list=_req.GetResults();
	
	if(rsp_list.size() == 0) {
	    System.err.println("MessageProtocol.SendMessage(): response list is empty !");
	    return null;	    
	}
	if(rsp_list.size() > 1)
	    System.err.println("MessageProtocol.SendMessage(): response list contains " +
			       "more that 1 response; returning first response !");
	rsp=(Rsp)rsp_list.elementAt(0);
	if(rsp.WasSuspected())
	    throw new Suspected(dest);
	if(!rsp.WasReceived())
	    throw new Timeout();
	return rsp.GetValue();
    }







    /**
       Processes a request destined for this layer. The return value is sent as response.
    */
    public Object Handle(Message req) {
	System.out.println("MessageProtocol.Handle(): this method should be overridden !");
	return null;
    }




    
    public final void Up(Event evt) {
	Message                msg;
	Object                 hdr;

	switch(evt.GetType()) {

	case Event.START:
	    Start();
	    HandleUpEvent(evt);
	    PassUp(evt);
	    break;

	case Event.STOP_OK:
	    PassUp(evt);
	    Stop();
	    break;

	case Event.VIEW_CHANGE:
	    UpdateView((View)evt.GetArg());
	    PassUp(evt);
	    break;
	    
	default:

	    // if we have a message with a MessageProtocolHeader, we don't need to go through the
	    // request correlator (see Down()), but can pass it up immediately. Otherwise, we let
	    // the layer do something (e.g. remove its own header) and decide whether to pass it up.
	    if(evt.GetType() == Event.MSG) {
		HandleUpEvent(evt);
		msg=(Message)evt.GetArg();
		hdr=msg.PeekHeader();
		if(hdr instanceof MessageProtocolHeader) {
		    msg.RemoveHeader();
		    PassUp(evt);
		    return;
		}
	    }

	    if(HandleUpEvent(evt)) {  // yes, pass up event
		if(_corr != null)
		    _corr.Receive(evt);
		else
		    System.err.println("MessageProtocol.Up(): request correlator is null ! " +
				       "Event is " + Util.PrintEvent(evt));
	    }
	    break;
	}
    }




    /**
       This message is not originated by this layer, therefore we can just pass it down without having
       to go through the request correlator. We do this ONLY for messages !
     */
    public final void Down(Event evt) {
	if(evt.GetType() == Event.VIEW_CHANGE)
	    UpdateView((View)evt.GetArg());
	
	if(evt.GetType() == Event.MSG) {
	    HandleDownEvent(evt);
	    ((Message)evt.GetArg()).AddHeader(new MessageProtocolHeader(GetName()));
	    PassDown(evt);
	    return;
	}

	if(HandleDownEvent(evt))
	    PassDown(evt);	
    }




    void UpdateView(View new_view) {
	Vector new_mbrs=new_view.GetMembers();
	if(new_mbrs != null) {
	    members.removeAllElements();
	    for(int i=0; i < new_mbrs.size(); i++)
		members.addElement(new_mbrs.elementAt(i));
	}
    }
    

    /**
       Handle up event. Return false if it should not be passed up the stack.
     */
    protected boolean HandleUpEvent(Event evt) {
	// override in subclasses
	return true;
    }

    /**
       Handle down event. Return false if it should not be passed down the stack.
     */
    protected boolean HandleDownEvent(Event evt) {
	// override in subclasses
	return true;
    }






    protected void Start() {
	if(_corr == null)
	    _corr=new RequestCorrelator(GetName(), this, this);
    }


    protected void Stop() {
	if(_corr != null) {
	    _corr.Stop();
	    _corr=null;
	}
    }


}
