package JavaGroups;

import java.io.*;
import java.util.*;
// import JavaGroups.Algorithms.*;



/**
 * Provides the abstraction of a java.util.Hashtable that is replicated at several
 * locations. Any change to the hashtable (clear, put, remove etc) will transparently be
 * propagated to all replicas in the group. All read-only methods will always access the
 * local replica.<p>
 * Both keys and values added to the hashtable <em>must be serializable</em>, the reason
 * being that they will be sent across the network to all replicas of the group. Having said
 * this, it is now for example possible to add RMI remote objects to the hashtable as they
 * are derived from <code>java.rmi.server.RemoteObject</code> which in turn is serializable.
 * This allows to lookup shared distributed objects by their name and invoke methods on them,
 * regardless of one's onw location. A <code>DistributedHashtable</code> thus allows to
 * implement a distributed naming service in just a couple of lines.<p>
 * An instance of this class will contact an existing member of the group to fetch its
 * initial state (using the state exchange funclet <code>StateExchangeFunclet</code>.
 */
public class DistributedHashtable extends Hashtable implements MessageListener, Cloneable {

    public interface Notification {
	void EntrySet(Object key, Object value);
	void EntryRemoved(Object key);
    }


    private transient Channel               channel;
    private transient RpcDispatcher         disp=null;
    private transient String                groupname=null;
    private transient Vector                notifs=new Vector();



    public DistributedHashtable(String groupname, ChannelFactory factory, String properties) {
	this.groupname=groupname;
	try {
	    channel=factory != null ? factory.CreateChannel(properties) : new JChannel(properties);
	    disp=new RpcDispatcher(channel, this, null, this);
	    channel.Connect(groupname);
	    channel.SetOpt(Channel.GET_STATE_EVENTS, new Boolean(true));
	    boolean rc=channel.GetState(null, 0);
	}
	catch(Exception e) {
	    System.err.println("DistributedHashtable(): " + e);
	}
    }



    public void AddNotifier(Notification n) {
	if(!notifs.contains(n))
	    notifs.addElement(n);
    }

    
    public Object put(Object key, Object value) {
	Object prev_val=get(key);
	try {	    
	    disp.CallRemoteMethods(null, "_put", key, value, GroupRequest.GET_ALL, 0);
	    return prev_val;
	}
	catch(Exception e) {
	    return null;
	}
    }

    public void clear() {
	try {
	    disp.CallRemoteMethods(null, "_clear", GroupRequest.GET_ALL, 0);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }

    public Object remove(Object key) {
	Object retval=get(key);
	try {
	    disp.CallRemoteMethods(null, "_remove", key, GroupRequest.GET_ALL, 0);
	    return retval;
	}
	catch(Exception e) {
	    return null;
	}
    }



    /*------------------------ Callbacks -----------------------*/

    public synchronized Object _put(Object key, Object value) {
	Object retval=super.put(key, value);
	for(int i=0; i < notifs.size(); i++)
	    ((Notification)notifs.elementAt(i)).EntrySet(key, value);
	return retval;
    }


    public synchronized void _clear() {super.clear();}


    public synchronized Object _remove(Object key) {
	Object retval=super.remove(key);
	for(int i=0; i < notifs.size(); i++)
	    ((Notification)notifs.elementAt(i)).EntryRemoved(key);
	return retval;
    }

    /*----------------------------------------------------------*/



    /*-------------------- State Exchange ----------------------*/

    public void Receive(Message msg) { }

    public Object GetState() {
	Object    key, val;
	Hashtable copy=new Hashtable();
	for(Enumeration e=keys(); e.hasMoreElements();) {
	    key=e.nextElement();
	    val=get(key);
	    copy.put(key, val);
	}
	return copy;
    }

    public void SetState(Object new_state) {
	Hashtable new_copy=(Hashtable)new_state;
	Object    key;
	if(new_copy == null)
	    return;

	_clear(); // remove all elements	
	for(Enumeration e=new_copy.keys(); e.hasMoreElements();) {
	    key=e.nextElement();
	    _put(key, new_copy.get(key));
	}
    }




}
