package JavaGroups;


import java.util.*;

/**
   Stores received messages according to their sequence numbers.  Any message is added
   in order, duplicate messages are discarded. Every message received is ACK'ed (even
   duplicates). Messages are removed in order; the first message has to have its seqno
   equal to <code>next_to_remove</code> (which will be incremented when a message is
   removed).  
*/
public class AckReceiverWindow {
    long        initial_seqno=0, next_to_remove=0;
    SortedList  msgs=new SortedList();



    class Entry implements Comparable {	
	long      seqno=0;
	Message   msg;
	
	Entry(long seqno, Message msg) {
	    this.seqno=seqno;
	    this.msg=msg;
	}

	public int Compare(Object other) {
	    Entry e=(Entry)other;
	    return seqno == e.seqno ? 0 : seqno < e.seqno ? -1 : 1;
	}
	
	public String toString() {return "" + seqno;}
    }




    public AckReceiverWindow(long initial_seqno) {
	this.initial_seqno=initial_seqno;
	next_to_remove=initial_seqno;
    }


    public void Add(long seqno, Message msg) {
	synchronized(msgs) {
	    if(seqno < next_to_remove) {
		//System.out.println("AckReceiverWindow.Add(): discarded msg with seqno=" + seqno +
		//	   ": next msg to receive is " + next_to_remove);
		return;
	    }
	    msgs.Add(new Entry(seqno, msg));
	}
    }

    /**
       Removes a message whose seqno is equal to <code>next_to_remove</code>, increments the latter.
       Returns message that was removed, or null, if no message can be removed. Messages are thus
       removed in order.
     */
    public Message Remove() {
	Entry   e;

	synchronized(msgs) {
	    e=(Entry)msgs.PeekAtHead();
	    if(e == null)
		return null;
	    if(e.seqno == next_to_remove) {
		msgs.RemoveFromHead();
		next_to_remove++;
		return e.msg;
	    }
	}
	return null;
    }

    
    public void Reset() {
	msgs.RemoveAll();
	next_to_remove=initial_seqno;
    }


    public String toString() {
	return msgs.toString();
    }




    public static void main(String[] args) {
	AckReceiverWindow win=new AckReceiverWindow(33);
	Message           m=new Message(), ret;

	win.Add(37, m);
	System.out.println(win);

	while((ret=win.Remove()) != null) {
	    System.out.println("Removed message, win is " + win);
	}


	win.Add(35, m);
	System.out.println(win);
	
	win.Add(36, m);
	System.out.println(win);

	while((ret=win.Remove()) != null) {
	    System.out.println("Removed message, win is " + win);
	}


	
	win.Add(33, m);
	System.out.println(win);

	win.Add(34, m);
	System.out.println(win);

	
	win.Add(38, m);
	System.out.println(win);


	while((ret=win.Remove()) != null) {
	    System.out.println("Removed message, win is " + win);
	}
	

	win.Add(35, m);
	System.out.println(win);
	
	win.Add(332, m);
	System.out.println(win);


    }
    
}
