package JavaGroups.JavaStack.Protocols;

import java.io.*;
import java.util.*;
import JavaGroups.*;
//import JavaGroups.Subject;
//import JavaGroups.Observer;
//import JavaGroups.Algorithms.*;
import JavaGroups.JavaStack.*;




class StateTransferRequest implements Serializable {
    static final int MAKE_COPY    = 1;  // arg = originator of request
    static final int RETURN_STATE = 2;  // arg = orginator of request

    int    type=0;
    Object arg;


    public StateTransferRequest(int type, Object arg) {
	this.type=type; this.arg=arg;
    }

    public int    GetType() {return type;}
    public Object GetArg()  {return arg;}

    public String toString() {
	return "[StateTransferRequest: type=" + Type2Str(type) + ", arg=" + arg + "]";
    }

    static String Type2Str(int t) {
	switch(t) {
	case MAKE_COPY:     return "MAKE_COPY";
	case RETURN_STATE:  return "RETURN_STATE";
	default:            return "<unknown>";
	}
    }
}



/**
   State transfer layer. Upon receiving a GET_STATE event from JChannel, a MAKE_COPY message is 
   sent to all members. When the originator receives MAKE_COPY, it queues all messages to the 
   channel.
   When another member receives the message, it asks the JChannel to provide it with a copy of
   the current state (GetStateEvent is received by application, ReturnState() sends state down the
   stack). Then the current layer sends a unicast RETURN_STATE message to the coordinator, which
   returns the cached copy.
   When the state is received by the originator, the GET_STATE sender is unblocked with a
   GET_STATE_OK event up the stack (unless it already timed out).<p>
   Requires QUEUE layer on top.
 */

public class STATE_TRANSFER extends Protocol implements RequestHandler {
    boolean               trace=false;
    Object                local_addr=null;
    Vector                members=new Vector();
    Message               m=null;
    boolean               is_server=false;
    Object                cached_state=null;
    Integer               state_xfer_mutex=new Integer(0); // get state from appl (via channel).
    long                  timeout_get_appl_state=5000;
    long                  timeout_return_state=5000;
    RequestCorrelator     corr=null;
    Vector                observers=new Vector();




    
    /** All protocol names have to be unique ! */
    public String  GetName() {return "STATE_TRANSFER";}


    
    
    public boolean SetProperties(Properties props) {
	String     str;
	
	this.props=props;
	str=props.getProperty("trace");
	if(str != null) {
	    trace=new Boolean(str).booleanValue();
	    props.remove("trace");
	}

	// Milliseconds to wait for application to provide requested state, events are
	// STATE_TRANSFER up and STATE_TRANSFER_OK down
	str=props.getProperty("timeout_get_appl_state");
	if(str != null) {
	    timeout_get_appl_state=new Long(str).longValue();
	    props.remove("timeout_get_appl_state");
	}

	// Milliseconds to wait for 1 or all members to return its/their state. 0 means wait
	// forever. States are retrieved using GroupRequest/RequestCorrelator
	str=props.getProperty("timeout_return_state");
	if(str != null) {
	    timeout_return_state=new Long(str).longValue();
	    props.remove("timeout_return_state");
	}

	if(props.size() > 0) {
	    System.err.println("STATE_TRANSFER.SetProperties(): the following " +
			       "properties are not recognized:");
	    props.list(System.out);
	    return false;
	}
	return true;
    }
    


    public Vector RequiredUpServices() {
	Vector ret=new Vector();
	ret.addElement(new Integer(Event.START_QUEUEING));
	ret.addElement(new Integer(Event.STOP_QUEUEING));
	return ret;
    }
    
    
    
    
    public void Up(Event evt) {
	Message              msg;
	Object               obj, sender;

	switch(evt.GetType()) {

	case Event.START:
	    corr=new RequestCorrelator(GetName(), this, this);
	    break;
	    
	case Event.BECOME_SERVER:
	    is_server=true;
	    break;

	case Event.SET_LOCAL_ADDRESS:
	    local_addr=evt.GetArg();
	    break;

	case Event.TMP_VIEW:
	case Event.VIEW_CHANGE:
	    Vector new_members=(Vector)((View)evt.GetArg()).GetMembers();
	    synchronized(members) {
		members.removeAllElements();
		if(new_members != null && new_members.size() > 0)
		    for(int k=0; k < new_members.size(); k++)
			members.addElement(new_members.elementAt(k));
	    }
	    break;	    
	}            // end switch


	if(corr != null)
	    corr.Receive(evt); // will consume or pass up, depending on header
	else
	    PassUp(evt);
    }




    public void Down(Event evt) {
	Integer            i;
	Object             coord, state;
	Vector             event_list=null;
	StateTransferInfo  info;


	switch(evt.GetType()) {

	case Event.STOP:
	    if(corr != null) {
		corr.Stop();
		corr=null;
	    }
	    break;

	case Event.TMP_VIEW:
	case Event.VIEW_CHANGE:
	    Vector new_members=(Vector)((View)evt.GetArg()).GetMembers();
	    synchronized(members) {
		members.removeAllElements();
		if(new_members != null && new_members.size() > 0)
		    for(int k=0; k < new_members.size(); k++)
			members.addElement(new_members.elementAt(k));
	    }
	    break;

	case Event.GET_STATE:       // generated by JChannel.GetState()
	    info=(StateTransferInfo)evt.GetArg();
	    coord=DetermineCoordinator();
	    
	    if(coord == null || coord.equals(local_addr)) {
		if(trace) System.out.println("++Coordinator is null");
		event_list=new Vector();
		event_list.addElement(new Event(Event.GET_STATE_OK, null));
		PassUp(new Event(Event.STOP_QUEUEING, event_list));
		return;             // don't pass down any further !
	    }

	    SendMakeCopyMessage();  // multicast MAKE_COPY to all members (including me)
	    
	    if(info.type == StateTransferInfo.GET_FROM_MANY)
		state=GetStateFromMany(info.targets);
	    else
		state=GetStateFromSingle(info.target);

	    /* Pass up the state to the application layer (insert into JChannel's event queue */
	    event_list=new Vector();
	    event_list.addElement(new Event(Event.GET_STATE_OK, state));

	    /* Now stop queueing */
	    PassUp(new Event(Event.STOP_QUEUEING, event_list));
	    return;                 // don't pass down any further !
	    
	case Event.GET_APPLSTATE_OK:
	    synchronized(state_xfer_mutex) {
		cached_state=evt.GetArg();
		state_xfer_mutex.notify();
	    }
	    return;                 // don't pass down any further !
	    
	}
 
	PassDown(evt);              // pass on to the layer below us
    }







    /* ---------------------- Interface RequestHandler -------------------------- */
    public Object Handle(Message msg) {
	StateTransferRequest req;

	try {
	    req=(StateTransferRequest)Util.ObjectFromByteBuffer(msg.GetBuffer());

	    switch(req.GetType()) {
	    case StateTransferRequest.MAKE_COPY:
		MakeCopy(req.GetArg());
		return null;
	    case StateTransferRequest.RETURN_STATE:
		if(is_server)
		    return cached_state;
		else {
		    System.err.println("STATE_TRANSFER.Handle(RETURN_STATE): returning null" +
				       "as I'm not yet an operational state server !");
		    return null;
		}
	    default:
		System.err.println("STATE_TRANSFER.Handle(): type " + req.GetType() + 
				   "is unknown in StateTransferRequest !");
		return null;
	    }
	}
	catch(Exception e) {
	    System.err.println("STATE_TRANSFER.Handle(): " + e);
	    return null;
	}
    }
    /* ------------------- End of Interface RequestHandler ---------------------- */








    Object GetStateFromSingle(Object target) {
	Vector                dests=new Vector();
	Message               msg;
	StateTransferRequest  r=new StateTransferRequest(StateTransferRequest.RETURN_STATE,
							 local_addr);
	RspList               rsp_list;
	Rsp                   rsp;
	Object                dest;
	GroupRequest          req;
	int                   num_tries=0;


	try {
	    msg=new Message(null, null, Util.ObjectToByteBuffer(r));
	}
	catch(Exception e) {
	    System.err.println("STATE_TRANSFER.GetStateFromCord(): " + e);
	    return null;
	}

	while(members.size() > 1 && num_tries++ < 3) {  // excluding myself
	    dest=target != null ? target : DetermineCoordinator();
	    if(dest == null)
		return null;
	    msg.SetDest(dest);
	    dests.removeAllElements();
	    dests.addElement(dest);
	    req=new GroupRequest(msg, corr, dests, GroupRequest.GET_FIRST, timeout_return_state, 0);
	    req.Execute();
	    rsp_list=req.GetResults();
	    for(int i=0; i < rsp_list.size(); i++) {  // get the first non-suspected result
		rsp=(Rsp)rsp_list.elementAt(i);
		if(rsp.WasReceived())
		    return rsp.GetValue();
	    }
	    Util.Sleep(1000);
	}

	return null;
    }






    Vector GetStateFromMany(Vector targets) {
	Vector                dests=new Vector();
	Message               msg;
	StateTransferRequest  r=new StateTransferRequest(StateTransferRequest.RETURN_STATE, 
							 local_addr);
	RspList               rsp_list;
	Vector                retval=new Vector();
	GroupRequest          req;
	int                   i;

	
	if(targets != null)
	    for(i=0; i < targets.size(); i++)
		if(!local_addr.equals(targets.elementAt(i)))
		    dests.addElement(targets.elementAt(i));
	else
	    for(i=0; i < members.size(); i++)
		if(!local_addr.equals(members.elementAt(i)))
		    dests.addElement(members.elementAt(i));
	
	if(dests.size() == 0)
	    return null;
	
	msg=new Message();
	try {
	    msg.SetBuffer(Util.ObjectToByteBuffer(r));
	}
	catch(Exception e) {}

	req=new GroupRequest(msg, corr, dests, GroupRequest.GET_ALL, timeout_return_state, 0);
	req.Execute();
	rsp_list=req.GetResults();
	return rsp_list.GetResults();
    }





    void SendMakeCopyMessage() {
	GroupRequest           req;
	Message                msg=new Message();
	StateTransferRequest   r=new StateTransferRequest(StateTransferRequest.MAKE_COPY,
							  local_addr);
	Vector                 dests=new Vector();


	for(int i=0; i < members.size(); i++)  // don't add myself twice in dests !
	    if(!local_addr.equals(members.elementAt(i)))
		dests.addElement(members.elementAt(i));
	
	if(dests.size() == 0)
	    return;

	try {
	    msg.SetBuffer(Util.ObjectToByteBuffer(r));
	}
	catch(Exception e) {}

	req=new GroupRequest(msg, corr, dests, GroupRequest.GET_ALL, timeout_return_state, 0);
	req.Execute();
    }



    /** Return the first element of members which is not me. Otherwise return null. */
    Object DetermineCoordinator() {
	Object ret=null;
	if(members != null && members.size() > 1) {
	    for(int i=0; i < members.size(); i++)
		if(!local_addr.equals(members.elementAt(i)))
		    return members.elementAt(i);
	}
	return ret;
    }



    /**
       If server, ask application to send us a copy of its state (STATE_TRANSFER up,
       STATE_TRANSFER down). If client, start queueing events. Queuing will be stopped when
       state has been retrieved (or not) from single or all member(s).
     */
    void MakeCopy(Object sender) {
	if(sender.equals(local_addr)) { // was sent by us, has to start queueing
	    PassUp(new Event(Event.START_QUEUEING));
	}
	else {               // only retrieve state from appl when not in client state anymore
	    if(is_server) {  // get state from application and store it locally
		synchronized(state_xfer_mutex) {
		    cached_state=null;

		    PassUp(new Event(Event.GET_APPLSTATE, local_addr));
		    try {
			state_xfer_mutex.wait(timeout_get_appl_state); // wait for STATE_TRANSFER_OK
		    }
		    catch(Exception e) {}
		}
	    }
	}
    }
    


}
