package JavaGroups;

import java.io.*;
import java.util.*;
import JavaGroups.JavaStack.Address;

/**
   Keeps track of ACKs from receivers for each message. When a new message is sent, it is tagged with
   a sequence number and the receiver set (set of members to which the message is sent) and added
   to a hashtable (key = sequence number, val = message + receiver set). Each incoming ACK is noted
   and when all ACKs for a specific sequence number haven been received, the corresponding entry is 
   removed from the hashtable. A retransmission thread periodically re-sends the message point-to-point
   to all receivers from which no ACKs have been received yet. A view change or suspect message causes
   the corresponding non-existing receivers to be removed from the hashtable.<p>
   This class may need flow control in order to avoid needless retransmissions because of
   timeouts.

   @author Bela Ban June 9 1999
 */
public class AckMcastSenderWindow implements Runnable {
    Hashtable          msgs=new Hashtable();     // long (seqno) -- Entry
    Thread             retransmitter=null;
    RetransmitCommand  command=null;
    long               retransmit_timeout=2000;  // +++ modify
    Vector             stable_msgs=new Vector();
    boolean            waiting=false;



    /**
       Called by retransmitter thread whenever a message needs to be re-sent to a destination.
       <code>dest</code> has to be set in the <code>dst</code> field of <code>msg</code>, as the
       latter was sent multicast, but now we are sending a unicast message. Message has to be
       copied before sending it (as headers will be appended and therefore the message changed !).
     */
    public interface RetransmitCommand {
	void Retransmit(long seqno, Message msg, Object dest);
    }



    class Entry {
	Message    msg;
	Hashtable  senders=new Hashtable();  // key=sender, value=boolean (true=received, false=not)
	int        num_received=0;
	long       timestamp=System.currentTimeMillis();

	Entry(Message msg, Vector dests) {
	    this.msg=msg;
	    for(int i=0; i < dests.size(); i++)
		senders.put(dests.elementAt(i), new Boolean(false));
	}

	boolean AllReceived() {return num_received >= senders.size() ? true : false;}

	public String toString() {
	    StringBuffer ret=new StringBuffer();
	    
	    ret.append("num_received=" + num_received + ", received msgs=" + senders);

	    return ret.toString();
	}
    }







    public AckMcastSenderWindow(RetransmitCommand command) {
	this.command=command;
	Start();
    }
    

    /** Returns copy of stable messages, or null (if non available). Removes all stable messages
     afterwards */
    public Vector GetStableMessages()   {
	Vector retval=stable_msgs.size() > 0 ? (Vector)stable_msgs.clone() : null;

	synchronized(stable_msgs) {
	    if(stable_msgs.size() > 0)
		stable_msgs.removeAllElements();
	}
	return retval;
    }
    

    public long Size() {return msgs.size();}



    /**
       Adds a new message to the hash table.
       @param seqno The sequence number associated with the message
       @param msg The message (should be a copy !)
       @param receivers The set of addresses to which the message was sent and from which 
                        consequently an ACK is expected
     */
    public void Add(long seqno, Message msg, Vector receivers) {
	if(waiting) {
	    //System.err.println("AckMcastSenderWindow.Add(): waiting to flush contents; "+
	    //	       "no entry may be added at this time. Discarding entry.");
	    return;
	}

	if(msgs.get(new Long(seqno)) != null) {
	    //System.err.println("AckMcastSenderWindow.Add(): seqno number " + seqno + " already present !");
	    return;
	}
	msgs.put(new Long(seqno), new Entry(msg, receivers));	
	if(retransmitter != null) {
	    synchronized(retransmitter) {
		retransmitter.notify();  // wake up retransmitter in case it is sleeping
	    }
	}
    }





    /**
       An ACK has been received from <code>sender</code>. Tag the sender in the hash table as
       'received'. If all ACKs have been received, remove the entry altogether.
       @param seqno  The sequence number of the message for which an ACK has been received.
       @param sender The sender which sent the ACK
     */
    public void Ack(long seqno, Object sender) {
	Entry    entry=null;
	Boolean  received;

	entry=(Entry)msgs.get(new Long(seqno));
	if(entry == null)
	    return;

	synchronized(entry) {
	    received=(Boolean)entry.senders.get(sender);
	    if(received == null)
		return;
	    if(received.booleanValue() == false) { // not yet received
		entry.senders.put(sender, new Boolean(true));
		entry.num_received++;
	    }
	    else
		return;
	    
	    if(entry.AllReceived()) {
		msgs.remove(new Long(seqno));
		stable_msgs.addElement(new Long(seqno));
		synchronized(msgs) {
		    msgs.notify();   // wake up WaitUntilAllAcksReceived() method
		}
	    }
	}	
    }


    /**
       Remove <code>obj</code> from all receiver sets and wake up retransmission thread.
     */
    public void Remove(Object obj) {
	Long  key;
	Entry entry;
	
	for(Enumeration e=msgs.keys(); e.hasMoreElements();) {
	    key=(Long)e.nextElement();
	    entry=(Entry)msgs.get(key);
	    entry.senders.remove(obj);
	    
	    if(entry.AllReceived()) {
		//System.out.println("AckMcastSenderWindow.Suspect(): received all ACKs for " + key +
		//	   ": removing entry from hashtable");
		msgs.remove(key);
		stable_msgs.addElement(key);
		synchronized(msgs) {
		    msgs.notify();   // wake up WaitUntilAllAcksReceived() method
		}
	    }
	}
    }


    /**
       Process with address <code>suspected</code> is suspected: remove it from all receiver sets.
       This means that no ACKs are expected from this process anymore.
       @param suspected The suspected process
     */
    public void Suspect(Object suspected) {
	Remove(suspected);
    }




    /** Remove all entries from the hashtable. Stop the retransmitter thread */
    public void Reset() {
	if(waiting) {
	    //System.err.println("AckMcastSenderWindow.Reset(): waiting to flush contents; cant reset now !");
	    return;
	}

	synchronized(msgs) {
	    msgs.clear();
	    msgs.notify();
	}
	Stop();
	Start(); // +++ ????
    }



    /**
       Waits until all outstanding messages have been ACKed by all receivers. Takes into account
       suspicions and view changes. Returns when there are no entries left in the hashtable.
       While waiting, no entries can be added to the hashtable (they will be discarded).
       @param timeout Miliseconds to wait. 0 means wait indefinitely.
     */
    public void WaitUntilAllAcksReceived(long timeout) {
	long time_to_wait=timeout, start_time, current_time;
	
	waiting=true;
	if(timeout <= 0) {
	    synchronized(msgs) {
		while(msgs.size() > 0) {
		    try {
			msgs.wait();
		    }
		    catch(Exception e) {}
		}
	    }
	}
	else {
	    start_time=System.currentTimeMillis();
	    synchronized(msgs) {
		while(msgs.size() > 0) {
		    current_time=System.currentTimeMillis();
		    time_to_wait=timeout - (current_time - start_time);
		    if(time_to_wait <= 0)
			break;
		    try {
			msgs.wait(time_to_wait);
		    }
		    catch(Exception e) {
			System.out.println(e);
		    }
		}
	    }

	}
	waiting=false;
    }



    /**
       Retransmitter thread. While there are messages available for which not all ACKs have been
       received yet, retransmit those without ACKs.
     */
    public void run() {
	Entry entry;
	Long  key;

	if(command == null) {
	    System.err.println("AckMcastSenderWindow.run(): command is null. Cannot retransmit " +
			       "messages !");
	    return;
	}

	while(true) {	    
	    synchronized(retransmitter) {
		while(msgs.size() == 0) { // wait until new msg added (or don't wait if msgs available)
		    try {
			retransmitter.wait(0);
		    }
		    catch(Exception e) {}
		}
	    }
	    
	    Util.Sleep(retransmit_timeout);
			
	    if(msgs.size() == 0)
		continue;
	    for(Enumeration e=msgs.keys(); e.hasMoreElements();) {
		key=(Long)e.nextElement();
		entry=(Entry)msgs.get(key);
		if(System.currentTimeMillis() - entry.timestamp >= retransmit_timeout) {
		    synchronized(entry) {
			Object  sender;
			Boolean received;
			
			for(Enumeration e2=entry.senders.keys(); e2.hasMoreElements();) {
			    sender=e2.nextElement();
			    received=(Boolean)entry.senders.get(sender);
			    if(received.booleanValue() == false) {
				//System.out.println("AckMcastSenderWindow.run(): --> retransmitting" +
				//	   " msg #" + key + " to " + sender);
				command.Retransmit(key.longValue(), entry.msg.Copy(), sender);
			    }
			}
		    }
		}
	    }
	}
    }







    public String toString() {
  	StringBuffer ret=new StringBuffer();
  	Entry        entry;
  	Long         key;
	
  	for(Enumeration e=msgs.keys(); e.hasMoreElements();) {
  	    key=(Long)e.nextElement();
  	    entry=(Entry)msgs.get(key);
  	    ret.append("Key=" + key + ", value=" + entry + "\n");
  	}
	
  	return ret.toString();
    }




    public synchronized void Start() {
	if(retransmitter == null) {
	    retransmitter=new Thread(this, "AckMcastSenderWindow.RetransmitterThread");
	    retransmitter.start();
	}
	else {
	    if(!retransmitter.isAlive()) {
		retransmitter.stop();
		retransmitter=new Thread(this, "AckMcastSenderWindow.RetransmitterThread");
		retransmitter.start();
	    }
	    else
		;  // don't do anything, thread is still alive & kicking
	}
    }


    public synchronized void Stop() {
	if(retransmitter != null) {
	    retransmitter.stop();
	    retransmitter=null;
	}
    }










    public static void main(String[] args) {
	AckMcastSenderWindow  win;
	Message               m=new Message();
	Object                recv1=new Address("janet", 5555);
	Object                recv2=new Address("janet", 6666);
	Object                recv3=new Address("janet", 7777);
	Vector                receivers=new Vector();


	class retr implements AckMcastSenderWindow.RetransmitCommand {

	    public void Retransmit(long seqno, Message msg, Object dest) {

	    }

	}


	class Acker extends Thread {
	    AckMcastSenderWindow win;
	    Object r1, r2, r3;
	    
	    Acker(AckMcastSenderWindow win, Object r1, Object r2, Object r3) {
		this.win=win;
		this.r1=r1;
		this.r2=r2;
		this.r3=r3;
	    }

	    public void run() {
		System.out.println("Acker thread started");
		win.Ack(2, r3);
		Util.Sleep(1000);		

		win.Ack(1, r2);
		Util.Sleep(400);
		win.Suspect(r1);

		System.out.println("Acker: contents are " + win);
		Util.Sleep(1000);

		win.Ack(1, r1);
		win.Ack(2, r1);
		win.Ack(2, r2);
		System.out.println("Acker: contents are " + win);
		Util.Sleep(5000);
		win.Ack(1, r3);
	    }
	}




	receivers.addElement(recv1);
	receivers.addElement(recv2);
	receivers.addElement(recv3);


	win=new AckMcastSenderWindow(new retr());
	win.Start();

	win.Add(1, m, receivers);
	win.Add(2, m, receivers);
	new Acker(win, recv1, recv2, recv3).start();
	win.WaitUntilAllAcksReceived(7500);

	win.Stop();
    }

}
