package JavaGroups;




/**
   Implementation of a priority scheduler. The scheduler maintains a queue to the end of which
   all tasks are added. It continually looks at the first queue element, assigns a thread to it,
   runs the thread and waits for completion. When a new <em>priority task</em> is added, it
   will be added to the head of the queue and the scheduler will be interrupted. In this case,
   the currently handled task is suspended, and the one at the head of the queue handled. This
   is recursive: a priority task can always be interrupted by another priority task.
   Resursion ends when no more priority tasks are added, or when the thread pool is
   exhausted.
 */
public class Scheduler implements Runnable {
    Queue               queue=new Queue();
    Thread              sched_thread=null;
    Task                current_task=null;
    ThreadPool          pool=null;
    int                 NUM_THREADS=5;
    final int           WAIT_FOR_THREAD_AVAILABILITY=3000;
    SchedulerListener   listener=null;
    Object              queue_mutex=new Object();
    


    public class Task {
	Thread   thread=null;
	Runnable target=null;
	boolean  suspended=false;

	Task(Runnable target) {this.target=target;}

	public String toString() {
	    return "[thread=" + thread + ", target=" + target + ", suspended=" + suspended + "]";
	}
    }


    public Scheduler() {}


    public Scheduler(int num_threads) {this.NUM_THREADS=num_threads;}

    public void SetListener(SchedulerListener l) {
	listener=l;
    }


    public void run() {
	while(true) {
	    try {
		current_task=(Task)queue.Peek(0); // get the first task in the queue
		if(current_task.suspended) {
		    current_task.suspended=false;
		    current_task.thread.resume();
		    if(listener != null) listener.Resumed(current_task.target);
		}
		else {
		    if(current_task.thread == null) {
			current_task.thread=pool.GetThread();
			if(current_task.thread == null) { // thread pool exhausted
			    Util.Sleep(WAIT_FOR_THREAD_AVAILABILITY);
			    continue;
			}
			// The notification has to be *before* handling the message, otherwise 
			// the header will already have been removed !
			if(listener != null) listener.Started(current_task.target);
			((ReusableThread)current_task.thread).AssignTask(current_task.target);
		    }
		    else {
			if(listener != null) listener.Started(current_task.target);
			((ReusableThread)current_task.thread).AssignTask(current_task.target);
		    }
		}

		if(sched_thread.isInterrupted()) // will continue at catch(InterruptedException)
		    sched_thread.interrupt();
		
		synchronized(current_task.thread) {
		    while(!((ReusableThread)current_task.thread).Done())
			current_task.thread.wait();
		}

		if(listener != null) listener.Stopped(current_task.target);
		queue.RemoveElement(current_task);
	    }
	    catch(InterruptedException interrupted) {
		// System.out.println("Scheduler was interrupted");
		if(current_task.thread != null) {
		    current_task.thread.suspend();
		    if(listener != null) listener.Suspended(current_task.target);
		    current_task.suspended=true;
		}
		continue;
	    }
	    catch(QueueClosed closed_ex) {
		sched_thread=null;
		return;
	    }
	    catch(Exception ex) {
		System.err.println("Scheduler.run(): " + ex);
		continue;
	    }
	}
    }



    
    public void AddPrio(Runnable task) {
	Task new_task=new Task(task);
	
	try {
	    synchronized(queue_mutex) {
		if(queue.Size() == 0)
		    queue.Add(new_task);
		else {
		    queue.AddAtHead(new_task);
		    sched_thread.interrupt();
		}
	    }
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }



    

    public void Add(Runnable task) {
	Task new_task=new Task(task);

	try {
	    synchronized(queue_mutex) {
		queue.Add(new_task);
	    }
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }


    
    
    public void Reset() {
	queue.Reset();
    }



    public void Start() {
	if(sched_thread == null) {
	    pool=new ThreadPool(NUM_THREADS);
	    sched_thread=new Thread(this, "Scheduler main thread");
	    sched_thread.start();
	}
    }



    public void Stop() {
	if(sched_thread != null) {
	    queue.Close(false); // will stop thread
	    sched_thread.stop();
	    sched_thread=null;
	    pool.Destroy();
	    pool=null;
	}
    }








    public static void main(String[] args) {

	class MyThread implements Runnable {
	    String name;

	    MyThread(String name) {this.name=name;}

	    public void run() {
		long sleep_time=(long)(Math.random() * 1000);
		Util.Print("\n--> " + name + ": sleeping for " + sleep_time + " ms");
		try {
		    Util.Sleep(sleep_time);
		    Util.Print("--> " + name + ": Done");
		}
		catch(Exception e) {
		    System.err.println(e);
		}
	    }
	}


	class Listener implements SchedulerListener {
	    public void Started(Runnable t) {
		System.out.println("--> Started: " + t);
	    }
	    public void Stopped(Runnable t) {
		System.out.println("--> Stopped: " + t);
	    }
	    public void Suspended(Runnable t) {
		System.out.println("--> Suspended: " + t);
	    }
	    public void Resumed(Runnable t) {
		System.out.println("--> Resumed: " + t);
	    }
	}


	Scheduler sch=new Scheduler();
	sch.SetListener(new Listener());
	sch.Add(new MyThread("Bela"));
	sch.Add(new MyThread("Janet"));
	sch.Add(new MyThread("Ralph"));
	sch.Start();
	sch.Add(new MyThread("Frank"));
	sch.Add(new MyThread("Marco"));

	Util.Sleep(1000);
	sch.AddPrio(new MyThread("Gabi"));
	sch.Add(new MyThread("Rolf"));
	Util.Sleep(100);
	sch.AddPrio(new MyThread("Gabi2"));
	Util.Sleep(100);
	sch.AddPrio(new MyThread("Gabi3"));
	Util.Sleep(100);
	sch.AddPrio(new MyThread("Gabi4"));
	
    }
    
}
