package JavaGroups;


import java.io.Serializable;
import java.util.Stack;
import java.util.*;
import JavaGroups.JavaStack.*;
import JavaGroups.Scheduler.Task;


class RequestCorrelatorHeader implements Serializable {
    public static final int REQ = 0;
    public static final int RSP = 1;

    int       type=REQ;
    long      id=0;
    boolean   rsp_expected=true;  // msg is synchronous if true
    String    name=null;
    Stack     call_stack=null;    // contains senders (e.g. P --> Q --> R)

    public RequestCorrelatorHeader(int type, long id, boolean rsp_expected, String name) {
	this.type=type; this.id=id; this.rsp_expected=rsp_expected; this.name=name;
    }

    public String toString() {
	StringBuffer ret=new StringBuffer();
	ret.append("[RequestCorrelatorHeader: name=" + name + ", type=");
	ret.append(type == REQ ? "REQ" : type == RSP ? "RSP" : "<unknown>");
	ret.append(", id=" + id);
	ret.append(", rsp_expected=" + rsp_expected + "]");
	return ret.toString();
    }
}





/**
   Framework to send requests and receive matching responses (matching on request ID).
   Multiple requests can be sent at a time. Whenever a response is received, the correct
   <code>RspCollector</code> is looked up (key = id) and its method 
   <code>ReceiveResponse</code> invoked. A caller may use <code>Done</code> to signal that no
   more responses are expected, and that the corresponding entry may be removed.<p>
   <code>RequestCorrelator</code> can be installed at both client and server sides, 
   it can also switch roles dynamically, i.e. send a request and at the same time process 
   an incoming request (when local delivery is enabled, this is actually the default).
 */
public class RequestCorrelator {
    protected Protocol          transport=null;
    protected Vector            requests=new Vector();
    protected RequestHandler    request_handler=null;
    protected String            name=null; // makes the instance unique (together with IDs)
    protected Scheduler         scheduler=null;
    protected Object            local_address=null;
    protected Stack             call_stack=null;
    protected boolean           deadlock_detection=true;
    protected CallStackSetter   call_stack_setter=new CallStackSetter();


    /** Listens for scheduler events and sets the current call chain (stack) whenever a thread is
	started, or a suspended thread resumed. Does this only for synchronous requests.
	(<code>Runnable</code> is actually a <code>Request</code>).
    */
    class CallStackSetter implements SchedulerListener {

	public void Started(Runnable r) {
	    SetCallStack(r);
	}

	public void Stopped(Runnable r) {	    
	    SetCallStack(null);
	}

	public void Suspended(Runnable r) {
	    SetCallStack(null);
	}

	public void Resumed(Runnable r) {
	    SetCallStack(r);
	}

	void SetCallStack(Runnable r) {
	    Stack                    new_stack;
	    Message                  req;
	    RequestCorrelatorHeader  hdr;
	    Object                   obj;

	    if(r == null) {
		call_stack=null;
		return;
	    }
	    req=((Request)r).req;
	    if(req == null) return;
	    obj=req.PeekHeader();
	    if(obj == null || !(obj instanceof RequestCorrelatorHeader)) return;
	    hdr=(RequestCorrelatorHeader)obj;
	    if(hdr.rsp_expected == false) 
		return;
	    
	    new_stack=hdr.call_stack;
	    if(new_stack != null)
		call_stack=(Stack)new_stack.clone();
	}

    }



    class Request implements Runnable {
	Message req=null;
	
	public Request(Message req) {this.req=req;}
	
	public void run() {
	    HandleRequest(req);
	}
	
    }




    class RequestEntry {
	long           id=0;
	RspCollector   coll=null;

	RequestEntry(long id, RspCollector coll) {
	    this.id=id; this.coll=coll;
	}

	public boolean equals(Object other) {
	    if(other == null || !(other instanceof RequestEntry))
		return false;
	    return id == ((RequestEntry)other).id ? true : false;
	}
    }


    private void AddEntry(RequestEntry entry) {
	synchronized(requests) {
	    if(!requests.contains(entry))
		requests.addElement(entry);
	    else
		System.err.println("RequestCorrelator.AddEntry(): entry " + 
				   entry + " already present !");
	}
    }


    private RspCollector FindEntry(long id) {
	RequestEntry entry;

	synchronized(requests) {
	    for(int i=0; i < requests.size(); i++) {
		entry=(RequestEntry)requests.elementAt(i);
		if(entry.id == id)
		    return entry.coll;
	    }
	}
	return null;
    }









    /**
       Constructor. Uses transport to send messages. If <code>handler</code> is not null,
       all incoming requests will be dispatched to it (via <code>Handle(Message)</code>).
       @param name Used to differentiate between diferent RequestCorrelators (e.g. in 
                   different protocol layers). Has to be unique if multiple request
		   correlators are used.
       @param transport Use to send requests. Only <code>Send</code> is used, not
                        <code>Receive</code> !
       @param handler Request handler. Method <code>Handle(Message)</code> will be called
                      when a request is received.
     */
    public RequestCorrelator(String name, Protocol transport, RequestHandler handler) {
	this.name=name;
	this.transport=transport;
	request_handler=handler;
	Start();
    }




    /**
       Constructor. Uses transport to send messages. If <code>handler</code> is not null,
       all incoming requests will be dispatched to it (via <code>Handle(Message)</code>).
       @param name Used to differentiate between diferent RequestCorrelators (e.g. in 
                   different protocol layers). Has to be unique if multiple request
		   correlators are used.
       @param transport Use to send requests. Only <code>Send</code> is used, not
                        <code>Receive</code> !
       @param handler Request handler. Method <code>Handle(Message)</code> will be called
                      when a request is received.
       @param deadlock_detection When enabled (true) recursive synchronous message calls will
                                 be detected and processed with higher priority in order to
				 solve deadlocks. Slows down processing a little bit when enabled
				 due to runtime checks involved.
     */
    public RequestCorrelator(String name, Protocol transport, RequestHandler handler, 
			     boolean deadlock_detection) {
	this.deadlock_detection=deadlock_detection;
	this.name=name;
	this.transport=transport;
	request_handler=handler;
	Start();
    }



    public void finalize() {Stop();}


    public void SetDeadlockDetection(boolean flag) {deadlock_detection=flag;}


    public void SetRequestHandler(RequestHandler handler) {
	request_handler=handler;
	Start();
    }


   
    /**
       Send a request to a group. If no response collector is given, no responses are
       expected (making the call asynchronous).
       @param id The request ID. Must be unique for this JVM (e.g. current time in millisecs)
       @param msg The request to be sent. The body of the message carries the request data
       @param coll A response collector (usually the object that invokes this method).
                   Its methods <code>ReceiveResponse</code> and <code>Suspect</code> will
		   be invoked when a message has been received or a member is suspected,
		   respectively.
    */
    public void SendRequest(long id, Message msg, RspCollector coll) {
	RequestCorrelatorHeader hdr;

	hdr=new RequestCorrelatorHeader(RequestCorrelatorHeader.REQ, id, 
					coll != null ? true : false, name);

	// add call stack for deadlock detection (if enabled)
	if(deadlock_detection && coll != null) {  // 'coll != null' means synchronous call
	    Stack new_call_stack=call_stack != null ? (Stack)call_stack.clone() : new Stack();
	    if(local_address != null)
		new_call_stack.push(local_address);
	    else
		System.err.println("RequestCorrelator.SendRequest(): local address is null !");
	    hdr.call_stack=new_call_stack;
	}	

	msg.AddHeader(hdr);

	if(coll != null)
	    AddEntry(new RequestEntry(hdr.id, coll));

	if(transport != null) {
	    try {
		transport.PassDown(new Event(Event.MSG, msg));  // send request
	    }
	    catch(Exception e) {}
	}
	else
	    System.err.println("RequestCorrelator.SendRequest(): transport is not available !");
    }





    /**
       Used to signal that a certain request may be garbage collected as all responses
       have been received.
    */
    public void Done(long id) {
	RequestEntry entry;

	synchronized(requests) {
	    for(int i=0; i < requests.size(); i++) {
		entry=(RequestEntry)requests.elementAt(i);
		if(entry.id == id) {
		    requests.removeElementAt(i);
		    break;
		}
	    }
	}
    }





    /**
       <b>Callback</b>. All response collectors currently registered will be notified that
       <code>mbr</code> may have crashed, so they won't wait for its response.
    */
    protected void Suspect(Object mbr) {
	RequestEntry entry;

	synchronized(requests) {
	    for(int i=0; i < requests.size(); i++) {
		entry=(RequestEntry)requests.elementAt(i);
		if(entry.coll != null)
		    entry.coll.Suspect(mbr);
	    }
	}
    }


    /**
       Mark all responses from members that are not in new_view as NOT_RECEIVED.
     */
    protected void ProcessViewChange(View new_view) {
	RequestEntry entry;
	
	synchronized(requests) {
	    for(int i=0; i < requests.size(); i++) {
		entry=(RequestEntry)requests.elementAt(i);
		if(entry.coll != null)
		    entry.coll.ViewChange(new_view);
	    }
	}
    }




    /**
       <b>Callback</b>. Called by the protocol below when a message has been
       received.  The algorithm should test whether the message is destined for us
       and, if not, pass it up to the next layer. Otherwise, it should remove the
       header and check whether the message is a request or response. In the first
       case, the message will be delivered to the request handler registered (calling
       its <code>Handle</code> method), in the second case, the corresponding
       response collector is looked up and the message delivered.
    */
    public void Receive(Event evt) {
	Object                    tmp;
	RequestCorrelatorHeader   hdr;
	Message                   msg;
	Request                   req=null;
	int                       type=evt.GetType();

	if(type != Event.MSG) {
	    switch(type) {
	    case Event.SUSPECT:             // don't wait for responses from faulty members
		Suspect(evt.GetArg());
		break;
	    case Event.VIEW_CHANGE:         // adjust number of responses to wait for
		ProcessViewChange((View)evt.GetArg());
		break;
	    case Event.SET_LOCAL_ADDRESS:
		local_address=evt.GetArg();
		break;
	    }
	    transport.PassUp(evt);
	    return;
	}
	
	// System.out.println("RequestCorr " + name + ": received msg");
	msg=(Message)evt.GetArg();
	tmp=msg.PeekHeader();
	if(tmp == null || !(tmp instanceof RequestCorrelatorHeader)) {	  // message is not handled by us !
	    transport.PassUp(evt);
	    return;
	}

	hdr=(RequestCorrelatorHeader)msg.PeekHeader();

	// check whether the message was sent by a request correlator with the same name
	// (there may be multiple request correlators in the same protocol stack...)
	if(hdr.name == null || !hdr.name.equals(name)) {
	    //System.err.println("RequestCorrelator.Receive(): name of request correlator" +
	    //	       "header (" + hdr.name + ") is different from ours (" + 
	    //	       name + "). Msg not accepted, passed up");
	    transport.PassUp(evt);
	    return;
	}

	switch(hdr.type) {
	case RequestCorrelatorHeader.REQ:
	    if(request_handler == null) {
		System.err.println("RequestCorrelator.Receive(): there is no request handler " +
				   "installed to deliver request !");
		return;
	    }

	    req=new Request(msg);

	    // check whether priority: if synchronous and call stack contains address that equals
	    // local address -> add priority request. Else add normal request.

	    if(deadlock_detection) {
		Stack st=hdr.call_stack;
		if(hdr.rsp_expected && st != null && local_address != null) {
		    if(st.contains(local_address)) { // was sent (directly or indirectly) by me
			scheduler.AddPrio(req);
			return;
		    }
		}
	    }

	    scheduler.Add(req);	    
	    break;

	case RequestCorrelatorHeader.RSP:
	    msg.RemoveHeader();
	    RspCollector coll=FindEntry(hdr.id);
	    if(coll != null)
		coll.ReceiveResponse(msg);
	    break;

	default:
	    msg.RemoveHeader();
	    System.err.println("RequestCorrelator.Receive(): header's type is " +
			       "neither REQ nor RSP !");
	    break;
	}
    }




    void HandleRequest(Message req) {
	Object                   retval;
	byte[]                   rsp_buf;
	RequestCorrelatorHeader  hdr, rsp_hdr;
	Message                  rsp=null;


	hdr=(RequestCorrelatorHeader)req.RemoveHeader();	
	retval=request_handler.Handle(req);
	rsp_buf=null;
	
	if(hdr.rsp_expected) {
	    rsp=req.MakeReply();	    
	    try {
		rsp_buf=retval != null ? Util.ObjectToByteBuffer(retval) : null;
	    }
	    catch(Exception e) {
		System.err.println("RequestCorrelator.HandleRequest(): failed sending response:" +
				   "return value is not serializable. Sending null value");
		rsp_buf=null;
	    }
	    if(rsp_buf != null)
		rsp.SetBuffer(rsp_buf);
	    
	    rsp_hdr=new RequestCorrelatorHeader(RequestCorrelatorHeader.RSP, hdr.id, false, name);
	    rsp.AddHeader(rsp_hdr);
	    
	    if(transport != null) {
		try {
		    transport.PassDown(new Event(Event.MSG, rsp));  // send response
		}
		catch(Exception e) {}
	    }
	    else
		System.err.println("RequestCorrelator.run(): failure sending response. " + 
				   "No protocol available ! ");
	}
    }




    public void Start() {
	if(scheduler == null) {
	    scheduler=new Scheduler();
	    if(deadlock_detection && call_stack_setter != null)
		scheduler.SetListener(call_stack_setter);
	    scheduler.Start();
	}
    }


    public void Stop() {
	if(scheduler != null) {
	    scheduler.Stop();
	    scheduler=null;
	}
    }






    public static void main(String args[]) {
	RequestCorrelatorHeader hdr, hdr2, hdr3;
	Vector v=new Vector();

	hdr=new RequestCorrelatorHeader(RequestCorrelatorHeader.REQ, 
					System.currentTimeMillis(), false, "bla");
	v.addElement(hdr);


	hdr2=new RequestCorrelatorHeader(RequestCorrelatorHeader.REQ, 
					 System.currentTimeMillis(), false, "bla");
	hdr2.id=hdr.id+1;
	v.addElement(hdr2);

	System.out.println(v);

	hdr3=new RequestCorrelatorHeader(RequestCorrelatorHeader.RSP, 
					 System.currentTimeMillis(), false, "bla");

	if(v.contains(hdr3))
	    System.out.println("Contains");
	else
	    System.out.println("Does no contain");

	
    }
}
