package JavaGroups;

import java.io.*;
import java.util.*;
import JavaGroups.JavaStack.*;




/**
   Used on top of channel to implement group requests. Client's <code>Handle()</code>
   method is called when request is received. Is the equivalent of RpcProtocol on the
   application instead of protocol level.
 */
public class MessageDispatcher extends Protocol implements UpHandler, RequestHandler {
    protected Channel             channel=null;
    protected RequestCorrelator   corr=null;
    protected MessageListener     msg_listener=null;
    protected MembershipListener  membership_listener=null;
    protected RequestHandler      req_handler=null;
    protected Vector              members=new Vector();





    public MessageDispatcher(Channel channel, MessageListener l, MembershipListener l2) {
	this.channel=channel;
	SetMessageListener(l);
	SetMembershipListener(l2);
	if(channel != null)
	    channel.SetUpHandler(this);
	Start();
    }



    public MessageDispatcher(Channel channel, MessageListener l, MembershipListener l2, 
			     RequestHandler req_handler) {
	this(channel, l, l2);
	SetRequestHandler(req_handler);
    }






    public void finalize() {
	Stop();
    }


    public void Start() {
	if(corr == null) {
	    corr=new RequestCorrelator("MessageDispatcher", this, this, true);  // deadlock detection
	    corr.Start();
	}
    }


    public void Stop() {
	if(corr != null) {
	    corr.Stop();
	    corr=null;
	}
    }


    public void SetMessageListener(MessageListener l) {
	channel.SetOpt(Channel.GET_STATE_EVENTS, new Boolean(true));
	msg_listener=l;
    }


    public void SetMembershipListener(MembershipListener l) {
	channel.SetOpt(Channel.VIEW, new Boolean(true));
	membership_listener=l;
    }

    public void SetRequestHandler(RequestHandler rh) {
	req_handler=rh;
    }



    public void Send(Message msg) throws ChannelNotConnected, ChannelClosed {
	if(channel != null)
	    channel.Send(msg);
	else
	    System.err.println("MessageDispatcher.Send(): channel is null !");
    }
    




   /**
       Cast a message to all members, and wait for <code>mode</code> responses. The responses are
       returned in a response list, where each response is associated with its sender.<p>
       Uses <code>GroupRequest</code>.
       @param dests The members to which then message is to be sent. If it is null, then the message 
                    is sent to all members
       @param msg The message to be sent to n members
       @param mode Defined in <code>GroupRequest</code>. The number of responses to wait for:
                   <ol>
		   <li>GET_FIRST: return the first response received.
		   <li>GET_ALL: wait for all responses (minus the ones from suspected members)
		   <li>GET_MAJORITY: wait for a majority of all responses (relative to the grp size)
		   <li>GET_ABS_MAJORITY: wait for majority (absolute, computed once)
		   <li>GET_N: wait for n responses (may block if n > group size)
		   <li>GET_NONE: wait for no responses, return immediately (non-blocking)
		   </ol>
       @param timeout If 0: wait forever. Otherwise, wait for <code>mode</code> responses 
                      <em>or</em> timeout time.
       @return RspList A list of responses. Each response is an <code>Object</code> and associated
                       to its sender.
   */    
    public RspList CastMessage(Vector dests, Message msg, int mode, long timeout) {
	GroupRequest  _req=null;
	Vector        real_dests=dests != null ? dests : members;
	
	// This marks message as sent by us ! (used in Up()
	// msg.AddHeader(new MsgProtHeader(GetName()));   ++ already done by RequestCorrelator

	//System.out.println("--> " + real_dests);
	_req=new GroupRequest(msg, corr, real_dests, mode, timeout, 0);
	_req.Execute();

	return _req.GetResults();
    }





    
    /**
       Sends a message to a single member (destination = msg.dest) and returns the response. 
       The message's destination must be non-zero !
    */
    public Object SendMessage(Message msg, int mode, long timeout) throws Timeout, Suspected {
	Vector        mbrs=new Vector();
	RspList       rsp_list=null;
	Object        dest=msg.GetDest();
	Rsp           rsp;
	Object        retval=null;
	GroupRequest  _req=null;

	if(dest == null) {
	    System.out.println("MessageProtocol.SendMessage(): the message's destination is null ! " +
			       "Cannot send message !");
	    return null;
	}
	
	mbrs.addElement(dest);   // dummy membership (of destination address)	

	_req=new GroupRequest(msg, corr, mbrs, mode, timeout, 0);
	_req.Execute();

	if(mode == GroupRequest.GET_NONE)
	    return null;


	rsp_list=_req.GetResults();
	
	if(rsp_list.size() == 0) {
	    System.err.println("MessageProtocol.SendMessage(): response list is empty !");
	    return null;	    
	}
	if(rsp_list.size() > 1)
	    System.err.println("MessageProtocol.SendMessage(): response list contains " +
			       "more that 1 response; returning first response !");
	rsp=(Rsp)rsp_list.elementAt(0);
	if(rsp.WasSuspected())
	    throw new Suspected(dest);
	if(!rsp.WasReceived())
	    throw new Timeout();
	return rsp.GetValue();
    }






    
    /* ------------------------ RequestHandler Interface ---------------------- */
    public Object Handle(Message msg) {
	if(req_handler != null)
	    return req_handler.Handle(msg);
	else
	    return null;
    }
    /* -------------------- End of RequestHandler Interface ------------------- */








    /* ------------------------- Protocol Interface --------------------------- */

    public String GetName() {return "MessageDispatcher";}

    public void StartUpHandler() {
	// do nothing, DON'T REMOVE !!!!
    }

    public void StartDownHandler() {
	// do nothing, DON'T REMOVE !!!!
    }


    public void StopInternal() {
	// do nothing, DON'T REMOVE !!!!
    }

    protected void ReceiveUpEvent(Event evt) {
    }

    protected void ReceiveDownEvent(Event evt) {
    }


    /** Called by request correlator when message was not generated by it. We handle it and call the
	message listener's corresponding methods
    */
    public void PassUp(Event evt) {
	switch(evt.GetType()) {
	case Event.MSG:
	    if(msg_listener != null)
		msg_listener.Receive((Message)evt.GetArg());
	    break;	    

	case Event.GET_APPLSTATE: // reply with GET_APPLSTATE_OK
	    if(msg_listener != null)
		channel.ReturnState(msg_listener.GetState());
	    else
		channel.ReturnState(null);
	    break;

	case Event.GET_STATE_OK:
	    if(msg_listener != null) {
		try {
		    msg_listener.SetState(evt.GetArg());
		}
		catch(ClassCastException cast_ex) {
		    System.err.println("MessageDispatcher.PassUp(): received SetStateEvent, " +
				       "but argument " + evt.GetArg() + 
				       " is not serializable ! Discarding message.");
		}		
	    }
	    break;

	case Event.VIEW_CHANGE:
	    View    v=(View)evt.GetArg();
	    Vector  new_mbrs=v.GetMembers();

	    if(new_mbrs != null) {
		members.removeAllElements();
		for(int i=0; i < new_mbrs.size(); i++)
		    members.addElement(new_mbrs.elementAt(i));
	    }

	    if(membership_listener != null)
		membership_listener.ViewAccepted(v);
	    break;
	    
	case Event.SUSPECT:
	    if(membership_listener != null)
		membership_listener.Suspect(evt.GetArg());
	    break;
	    
	case Event.BLOCK:
	    if(membership_listener != null)
		membership_listener.Block();
	    break;
	}


    }


    public void PassDown(Event evt) {
	Down(evt);
    }


    /** Called by channel (we registered before) when event is received. This is the UpHandler interface.
     */
    public void Up(Event evt) {
	if(corr != null)
	    corr.Receive(evt);
	else
	    System.err.println("MessageDispatcher.Up(): request correlator is null !");
    }


    public void Down(Event evt) {
	if(channel != null)
	    channel.Down(evt);
	else
	    System.err.println("MessageDispatcher.Down(): channel is null !");
    }
    /* ----------------------- End of Protocol Interface ----------------------- */

}
