package JavaGroups;

import java.io.*;
import java.util.*;

/**
   Doubly-linked list. Elements can be added at head or tail and removed from head/tail.
   This class is tuned for element access at either head or tail, random access to elements
   is not very fast; in this case use Vector. Concurrent access is supported: a thread is blocked
   while another thread adds/removes an object. When no objects are available, removal returns null.
 */
public class List implements Externalizable {
    protected Element   head=null, tail=null;
    protected int       size=0;
    protected Object    mutex=new Object();

    class Element {
	Object  obj=null;
	Element next=null;
	Element prev=null;

	Element(Object o) {obj=o;}
    }

    
    public List() {}



    /**
       Adds an object at the tail of the list.
     */
    public void Add(Object obj) {
	Element el=new Element(obj);

	synchronized(mutex) {
	    if(head == null) {
		head=el;
		tail=head;
		size=1;
	    }
	    else {
		el.prev=tail;
		tail.next=el;
		tail=el;
		size++;	    
	    }
	}	
    }


    /**
       Adds an object at the head of the list.
     */
    public void AddAtHead(Object obj) {
	Element el=new Element(obj);

	synchronized(mutex) {
	    if(head == null) {
		head=el;
		tail=head;
		size=1;
	    }
	    else {
		el.next=head;
		head.prev=el;
		head=el;
		size++;
	    }
	}
    }


    /**
       Removes an object from the tail of the list. Returns null if no elements available
    */
    public Object Remove() {
	Element retval=null;
	
	synchronized(mutex) {
	    if(tail == null)
		return null;
	    retval=tail;
	    if(head == tail) { // last element
		head=null; tail=null;
	    }
	    else {
		tail.prev.next=null;
		tail=tail.prev;
		retval.prev=null;
	    }

	    size--;
	}
	return retval.obj;
    }


    /** Removes an object from the head of the list. Returns null if no elements available */
    public Object RemoveFromHead() {
	Element retval=null;

	synchronized(mutex) {
	    if(head == null)
		return null;
	    retval=head;
	    if(head == tail) { // last element
		head=null; tail=null;
	    }
	    else {
		head=head.next;
		head.prev=null;
		retval.next=null;
	    }
	    size--;
	}
	return retval.obj;
    }
    



    /**
       Returns element at the tail (if present), but does not remove it from list.
     */
    public Object Peek() {
	synchronized(mutex) {
	    return tail != null ? tail.obj : null;
	}
    }



    /**
       Returns element at the head (if present), but does not remove it from list.
     */
    public Object PeekAtHead() {
	synchronized(mutex) {
	    return head != null ? head.obj : null;
	}
    }






    /**
       Removes element <code>obj</code> from the list, checking for equality using the <code>equals</code>
       operator. Only the first duplicate object is removed.
     */
    public void RemoveElement(Object obj) {
	Element el=head;

	synchronized(mutex) {
	    while(el != null) {
		if(el.obj.equals(obj)) {
		    
		    if(head == tail) {           // only 1 element left in the list
			head=null; tail=null;
		    }
		    else if(el.prev == null) {  // we're at the head
			head=el.next;
			head.prev=null;
			el.next=null;
		    }
		    else if(el.next == null) {  // we're at the tail
			tail=el.prev;
			tail.next=null;
			el.prev=null;
		    }
		    else {                      // we're somewhere in the middle of the list
			el.prev.next=el.next;
			el.next.prev=el.prev;
			el.next=null;
			el.prev=null;
		    }
		    size--;
		    break;
		}

		el=el.next;
	    }
	}
    }




    public void RemoveAll() {
	synchronized(mutex) {
	    size=0;
	    head=null;
	    tail=null;
	}
    }



    public int Size() {
	return size;
    }

    public String toString() {
	StringBuffer ret=new StringBuffer("[");
	Element el=head;
	
	while(el != null) {
	    if(el.obj != null)
		ret.append(el.obj + " ");
	    el=el.next;
	}
	ret.append("]");
	return ret.toString();
    }


    public String Dump() {
	StringBuffer ret=new StringBuffer("[");
	for(Element el=head; el != null; el=el.next)
	    ret.append(el.obj + " ");
	
	return ret.toString() + "]";
    }



    public Vector GetContents() {
	Vector   retval=new Vector(size);
	Element  el;

	synchronized(mutex) {
	    el=head;
	    while(el != null) {
		retval.addElement(el.obj);
		el=el.next;
	    }
	}
	return retval;
    }


    public Enumeration Elements() {
	return new ListEnumerator(head);
    }

    
    public boolean Contains(Object obj) {
	Element el=head;
	
	while(el != null) {
	    if(el.obj != null && el.obj.equals(obj))
		return true;
	    el=el.next;
	}
	return false;
    }



    public List Copy() {
	List    retval=new List();
	
	synchronized(mutex) {
	    for(Element el=head; el != null; el=el.next)
		retval.Add(el.obj);
	}
	
	return retval;
    }
    


    public void writeExternal(ObjectOutput out) throws IOException {
	Element el;
	
	synchronized(mutex) {
	    el=head;
	    out.writeInt(size);
	    for(int i=0; i < size; i++) {
		out.writeObject(el.obj);
		el=el.next;
	    }
	}
    }



    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
	Object obj;
	int    new_size=in.readInt();

	if(new_size == 0)
	    return;
	for(int i=0; i < new_size; i++) {
	    obj=in.readObject();
	    Add(obj);
	}
    }




    class ListEnumerator implements Enumeration {
	Element curr=null;

	ListEnumerator(Element start) {
	    curr=start;
	}
	
	public boolean hasMoreElements() {
	    return curr != null;
	}
	
	public Object nextElement() {
	    Object retval;

	    if(curr == null)
		throw new NoSuchElementException();
	    retval=curr.obj;
	    curr=curr.next;
	    return retval;
	}

    }




    public static void main(String args[]) {
	List l=new List();

	l.Add("Bela");
	l.Add("Janet");
	l.Add("Marco");
	l.Add("Ralph");

	for(Enumeration e=l.Elements(); e.hasMoreElements();) {
	    System.out.println(e.nextElement());
	}

	System.out.println(l + ".Contains(\"Bela\"): " + l.Contains("Bela"));


	l.Add(new Integer(1));
	l.Add(new Integer(2));
	l.Add(new Integer(5));
	l.Add(new Integer(6));


	System.out.println(l + ".Contains(2): " + l.Contains(new Integer(2)));
    }
    





//      public static void main(String[] args) {
//  	List l=new List();
//  	Long n;


//  	l.AddAtHead(new Integer(1));
//  	l.AddAtHead(new Integer(2));
//  	l.AddAtHead(new Integer(3));
//  	l.AddAtHead(new Integer(4));
//  	l.AddAtHead(new Integer(5));

//  	System.out.println("Removed from head: " + l.RemoveFromHead());
//  	System.out.println("Removed from head: " + l.RemoveFromHead());
//  	System.out.println("Removed from head: " + l.RemoveFromHead());
//  	System.out.println("Removed from head: " + l.RemoveFromHead());
//  	System.out.println("Removed from head: " + l.RemoveFromHead());
//  	System.out.println("Removed from head: " + l.RemoveFromHead());
//  	System.out.println("Removed from head: " + l.RemoveFromHead());

	
//  	System.out.print("Adding 50000 numbers:");
//  	for(long i=0; i < 50000; i++) {
//  	    n=new Long(i);
//  	    if(i % 2 == 0) {
//  		l.AddAtHead(n);
//  	    }
//  	    else {
//  		l.Add(n);
//  	    }
//  	}
//  	System.out.println(" OK");

//  	long   num=0;
//  	Object obj;
//  	System.out.print("Removing all elements: ");
//  	while((obj=l.Remove()) != null)
//  	      num++;
//  	System.out.println("OK, removed " + num + " objects");
//      }



//      public static void main(String[] args) {
//  	byte[]  buf;
//  	List    l=new List(), l2, l3;


//  	try {
//  	    for(int i=0; i < 10; i++)
//  		l.Add(new Integer(i));
//  	    System.out.println(l.GetContents());

//  	    buf=Util.ObjectToByteBuffer(l);
//  	    l2=(List)Util.ObjectFromByteBuffer(buf);
//  	    System.out.println(l2.GetContents());

//  	    l3=new List(); l3.Add("Bela"); l3.Add("Janet");


//  	    l2.Add(new Integer(300));
	    
//  	    FileOutputStream    outstream=new FileOutputStream("list.out");
//  	    ObjectOutputStream  out=new ObjectOutputStream(outstream);
//  	    out.writeObject(l2); out.writeObject(l3);
	    

//  	    FileInputStream instream=new FileInputStream("list.out");
//  	    ObjectInputStream in=new ObjectInputStream(instream);
//  	    l=(List)in.readObject();
//  	    System.out.println(l.GetContents());

//  	    l=(List)in.readObject();
//  	    System.out.println(l.GetContents());

//  	    //l=(List)in.readObject();
//  	    //System.out.println(l.GetContents());
//  	}
//  	catch(Exception e) {
//  	    System.err.println(e);
//  	}
	
//      }


}
