package JavaGroups.JavaStack.Protocols;

import java.io.*;
import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;


class UnicastHeader implements Serializable {
    static final int DATA=0;
    static final int DATA_ACK=1;

    int     type=DATA;
    long    seqno=0;   // First msg is 0
    boolean first=false;


    public UnicastHeader(int type, long seqno) {
	this.type=type == DATA_ACK ? DATA_ACK : DATA;
	this.seqno=seqno;
    }

    public String toString() {
	return "[UNICAST: " + Type2Str(type) + ", seqno=" + seqno + "]";
    }
    
    public String Type2Str(int t) {
	switch(t) {
	case DATA: return "DATA";
	case DATA_ACK: return "DATA_ACK";
	default: return "<unknown>";
	}
    }
}



/**
   Reliable unicast layer. Uses acknowledgement scheme similar to TCP to provide lossless transmission
   of unicast messages (for reliable multicast see NAKACK layer). When a message is sent to a peer for
   the first time, we add the pair <peer_addr, Entry> in the hashtable (peer address is the key). All
   messages sent to that peer will be added to hashtable.peer_addr.sent_msgs. When we receive a
   message from a peer for the first time, another entry will be created and added to the hashtable
   (unless already existing). Msgs will then be added to hashtable.peer_addr.received_msgs.<p> This
   layer is used to reliably transmit point-to-point messages, that is, either messages sent to a
   single receiver (vs. messages multicast to a group) or for example replies to a multicast message. The 
   sender uses an <code>AckSenderWindow</code> which retransmits messages for which it hasn't received
   an ACK, the receiver uses <code>AckReceiverWindow</code> which keeps track of the lowest seqno
   received so far, and keeps messages in order.
*/
public class UNICAST extends Protocol implements AckSenderWindow.RetransmitCommand {
    boolean    trace=false, is_server=false;
    Vector     members=new Vector();
    Hashtable  connections=new Hashtable(); // Object (peer sender or peer receiver) -- Entries


    class Entry {
	AckReceiverWindow  received_msgs=null;  // stores all msgs rcvd by a certain peer in seqno-order
	AckSenderWindow    sent_msgs=null;      // stores (and retransmits) msgs sent by us to a certain peer
	long               sent_msgs_seqno=GetInitialSeqno();  // seqno for msgs sent by us
	

	void Reset() {
	    if(sent_msgs != null)
		sent_msgs.Reset();
	    if(received_msgs != null)
		received_msgs.Reset();
	}
    }


    /** All protocol names have to be unique ! */
    public String  GetName() {return "UNICAST";}


    public boolean SetProperties(Properties props) {
	String     str;

	this.props=props;
	str=props.getProperty("trace");
	if(str != null) {
	    trace=new Boolean(str).booleanValue();
	    props.remove("trace");
	}
	if(props.size() > 0) {
	    System.err.println("UNICAST.SetProperties(): these properties are not recognized:");
	    props.list(System.out);
	    return false;
	}
	return true;
    }



    public void Up(Event evt) {
	Message        msg;
	Object         dst, src, obj;
	UnicastHeader  hdr;
	Entry          entry;

	switch(evt.GetType()) {

	case Event.MSG:
	    msg=(Message)evt.GetArg();
	    dst=msg.GetDest();
	    src=msg.GetSrc();
	    if(dst == null || ((Address)dst).IsMulticastAddress())  // only handle unicast messages
		break;

	    obj=msg.PeekHeader();
	    if(obj == null || !(obj instanceof UnicastHeader)) {
		PassUp(evt);
		return;
	    }

	    hdr=(UnicastHeader)msg.RemoveHeader();	    
	    switch(hdr.type) {
	    case UnicastHeader.DATA:      // received regular message
		HandleDataReceived(src, hdr.seqno, hdr.first, msg);
		SendAck(src, hdr.seqno);
		break;
	    case UnicastHeader.DATA_ACK:  // received ACK for previously sent message
		HandleAckReceived(src, hdr.seqno);
		break;
	    default:
		System.err.println("UNICAST.Up(MSG): UnicastHeader type " + hdr.type + " not known !");
		break;
	    }

	    return;
	}

	PassUp(evt);            // Pass up to the layer above us
    }





    public void Down(Event evt) {
	Message        msg;
	Object         dst, mbr;
	Entry          entry;
	UnicastHeader  hdr;

	switch(evt.GetType()) {

	case Event.BECOME_SERVER:
	    is_server=true;
	    break;

	case Event.VIEW_CHANGE:
	    Vector new_members=(Vector)((View)evt.GetArg()).GetMembers();
	    synchronized(members) {
		members.removeAllElements();
		if(new_members != null && new_members.size() > 0)
		    for(int i=0; i < new_members.size(); i++)
			members.addElement(new_members.elementAt(i));		
	    }

	    // Remove connections to peers that are not members anymore !
	    synchronized(connections) {
		for(Enumeration e=connections.keys(); e.hasMoreElements();) {
		    mbr=e.nextElement();
		    if(!members.contains(mbr)) {
			entry=(Entry)connections.get(mbr);
			entry.Reset();
			System.out.println("********** UNICAST.Down(VIEW_CHANGE): removing entry for " + mbr);
			connections.remove(mbr);
		    }
		}
	    }
	    break;

        // Add UnicastHeader, add to AckSenderWindow and pass down
	case Event.MSG:
	    msg=(Message)evt.GetArg();
	    dst=msg.GetDest();

	    /* only handle unicast messages */
	    if(dst == null || ((Address)dst).IsMulticastAddress())
		break;

	    entry=(Entry)connections.get(dst);
	    if(entry == null) {
		entry=new Entry();
		connections.put(dst, entry);
	    }

	    hdr=new UnicastHeader(UnicastHeader.DATA, entry.sent_msgs_seqno);
	    if(entry.sent_msgs == null) { // first msg to peer 'dst'
		hdr.first=true;
		entry.sent_msgs=new AckSenderWindow(this, entry.sent_msgs_seqno);
	    }
	    msg.AddHeader(hdr);

	    System.out.println("UNICAST: --> DATA(" + dst + ": #" + entry.sent_msgs_seqno + ", first=" + 
			       hdr.first + ")");

	    entry.sent_msgs.Add(entry.sent_msgs_seqno, msg.Copy());  // add *including* UnicastHeader
	    entry.sent_msgs_seqno++;
	    break;
	    
	}  // switch

	PassDown(evt);          // Pass on to the layer below us
    }




    /** Returns random initial sequence number between 1 and 100 */
    long GetInitialSeqno() {
	long ret=(long)((Math.random() * 100) % 100);
	return ret;
    }



    /** Called by AckSenderWindow to resend messages for which no ACK has been received yet */
    public void Retransmit(long seqno, Message msg, int num_tries) {
	Object dst=msg.GetDest();
	Entry  entry;

	System.out.println("-------> UNICAST: retransmitting msg #" + seqno + " to " + dst);

	if(is_server) {
	    if(!members.contains(dst)) {
		System.err.println("**** UNICAST.Retransmit(" + seqno + "): dest " + dst + " is not member " +
				   "any longer; removing entry !");
		
		synchronized(connections) {
		    entry=(Entry)connections.get(dst);
		    if(entry != null)
			entry.Reset();
		    connections.remove(dst);
		}
		return;
	    }
	}

	PassDown(new Event(Event.MSG, msg.Copy()));
    }





    /**
       Check whether the hashtable contains an entry e for <code>sender</code> (create if not). If
       e.received_msgs is null and <code>first</code> is true: create a new AckReceiverWindow(seqno) and
       add message. Set e.received_msgs to the new window. Else just add the message. If first is false,
       but we don't yet have hashtable.received_msgs, then just discard the message. If first is true, but
       hashtable.received_msgs already exists, also discard the message (redundant message).
     */
    void HandleDataReceived(Object sender, long seqno, boolean first, Message msg) {
	Entry    entry;
	Message  m;

	System.out.println("UNICAST: <-- DATA(" + sender + ": #" + seqno + ", first=" + first + ")");
	
	entry=(Entry)connections.get(sender);
	if(entry == null) {
	    entry=new Entry();
	    connections.put(sender, entry);
	}

	if(entry.received_msgs == null) {
	    if(first)
		entry.received_msgs=new AckReceiverWindow(seqno);
	    else {
		System.err.println("UNICAST.HandleDataReceived(): seqno " + seqno + " from " + sender +
				   " is not tagged as the first message sent by " + sender + "; however, " +
				   "the table for received messages from " + sender + " is still null !" +
				   "We probably haven't received the first message from " + sender + " !" +
				   "Discarding message");
		return;
	    }
	}

	entry.received_msgs.Add(seqno, msg);


	// Try to remove (from the AckReceiverWindow) as many messages as possible as pass them up
	while((m=entry.received_msgs.Remove()) != null)
	    PassUp(new Event(Event.MSG, m));
    }




    /**
       Add the ACK to hashtable.sender.sent_msgs
     */
    void HandleAckReceived(Object sender, long seqno) {
	Entry           entry;
	AckSenderWindow win;

	System.out.println("UNICAST: <-- ACK(" + sender + ": #" + seqno + ")");

	entry=(Entry)connections.get(sender);
	if(entry == null || entry.sent_msgs == null) {
	    System.err.println("UNICAST.HandleAckReceived(): entry for seqno " + seqno + " from " +
			       sender + " not found; discarding ACK !");
	    return;
	}
	win=(AckSenderWindow)entry.sent_msgs;
	win.Ack(seqno); // removes message from retransmission
    }



    void SendAck(Object dst, long seqno) {
	Message ack=new Message(dst, null, null);
	ack.AddHeader(new UnicastHeader(UnicastHeader.DATA_ACK, seqno));
	
	System.out.println("UNICAST: --> ACK(" + dst + ": #" + seqno + ")");

	PassDown(new Event(Event.MSG, ack));
    }



}
