

package JavaGroups.JavaStack.Protocols;

import java.io.*;
import java.net.*;
import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;




/**
   Replacement for UDP. Instead of sending packets via UDP, a TCP connection is opened to a JRouter,
   the IP address/port of which was given using channel properties <code>router_host</code> and 
   <code>router_port</code>. All outgoing traffic is sent via this TCP socket to the JRouter which
   distributes it to all connected TUNNELs in this group. Incoming traffic received from JRouter will
   simply be passed up the stack.
   <p>A TUNNEL layer can be used to penetrate a firewall, most firewalls allow creating TCP connections
   to the outside world, however, they do not permit outside hosts to initiate a TCP connection to a host
   inside the firewall. Therefore, the connection created by the inside host is reused by JRouter to
   send traffic from an outside host to a host inside the firewall.
 */
public class TUNNEL extends Protocol implements Runnable {
    private Properties          properties=null;
    private String              channel_name=null;
    private boolean             trace=false;
    private Vector              members=new Vector();    
    private String              router_host=null;
    private int                 router_port=0;
    private Address             local_addr=null;  // sock's local addr and local port
    private Socket              sock=null;        // connection to JRouter
    private Thread              receiver=null;
    private OutputStream        out_stream=null;
    private ObjectOutputStream  out=null;
    private InputStream         in_stream=null;
    private ObjectInputStream   in=null;


    public TUNNEL() {}



    public String toString() {
	return "Protocol TUNNEL(local_addr=" + local_addr + ")";
    }





    /* ------------------------------ Private methods -------------------------------- */




    public synchronized void HandleIncomingMessage(Message msg) {
	TunnelHeader hdr=(TunnelHeader)msg.RemoveHeader();

	if(trace) Util.Print("TUNNEL.HandleIncomingMessage(): received msg " + msg);
		

	/* Discard all messages destined for a channel with a different name */

	String ch_name=hdr != null ? hdr.channel_name : null;
	if(ch_name != null && !channel_name.equals(ch_name))
	    return;
	
	PassUp(new Event(Event.MSG, msg));	
    }
    
    



    private void HandleDownEvent(Event evt) {
	switch(evt.GetType()) {

	case Event.TMP_VIEW:
	case Event.VIEW_CHANGE:
	    synchronized(members) {
		members.removeAllElements();
		Vector tmpvec=(Vector)((View)evt.GetArg()).GetMembers();
		for(int i=0; i < tmpvec.size(); i++)
		    members.addElement(tmpvec.elementAt(i));		
	    }
	    break;

	case Event.GET_LOCAL_ADDRESS:   // return local address -> Event(SET_LOCAL_ADDRESS, local)
	    PassUp(new Event(Event.SET_LOCAL_ADDRESS, local_addr));
	    break;

	case Event.CLEANUP:
	    PassUp(new Event(Event.CLEANUP_OK));
	    break;

	case Event.SET_LOCAL_ADDRESS:
	    local_addr=(Address)evt.GetArg();
	    break;
	    
	case Event.CONNECT:
	    channel_name=(String)evt.GetArg();
	    PassUp(new Event(Event.CONNECT_OK));
	    break;

//  	case Event.CONNECT:
//  	    try {
//  		channel_name=(String)evt.GetArg();
//  		StartWork();

//  		if(trace)
//  		    System.out.println("TUNNEL.Down(CONNECT): passing up local address=" + local_addr);

//  		PassUp(new Event(Event.SET_LOCAL_ADDRESS, local_addr));
//  		PassUp(new Event(Event.CONNECT_OK));
//  	    }
//  	    catch(ClassCastException cce) {
//  		System.err.println("TUNNEL.HandleDownEvent(CONNECT): channel name must be a string " +
//  				   ", but was " + evt.GetArg());
//  		return;
//  	    }
//  	    break;


	case Event.DISCONNECT:
	    StopWork();
	    PassUp(new Event(Event.DISCONNECT_OK));
	    break;
	}
    }



    /* ------------------------------------------------------------------------------- */







    public void run() {
	Object o;

	try {
	    while(true) {
		o=in.readObject();
		if(!(o instanceof Message)) {
		    System.err.println("TUNNEL.run(): received a request of type " +
				       o.getClass().getName() + ", discarding");
		    continue;
		}
		HandleIncomingMessage((Message)o);
	    }
	}
	catch(Exception e) {
	    System.err.println("TUNNEL.run(): " + e);
	}
    }






    /*------------------------------ Protocol interface ------------------------------ */

    public String GetName() {return "TUNNEL";}


    /** Setup the Protocol instance acording to the configuration string */
    public boolean SetProperties(Properties props) {
	String               str;

	this.props=props;
	str=props.getProperty("router_host");
	if(str != null) {
	    router_host=new String(str);
	    props.remove("router_host");
	}

	str=props.getProperty("router_port");
	if(str != null) {
	    router_port=new Integer(str).intValue();
	    props.remove("router_port");
	}

	str=props.getProperty("trace");
	if(str != null) {
	    trace=new Boolean(str).booleanValue();
	    props.remove("trace");
	}

	if(trace) 
	    Util.Print("TUNNEL(router_host=" + router_host + ";router_port=" + router_port +")");

	if(router_host == null || router_port == 0) {
	    System.err.println("TUNNEL.SetProperties(): both router_host and router_port have " +
			       "to be set !");
	    System.exit(-1);
	}


	if(props.size() > 0) {
	    System.err.println("TUNNEL.SetProperties(): the following properties are not recognized:");
	    props.list(System.out);
	    return false;
	}
	return true;
    }



    public void Up(Event evt) {
	if(evt.GetType() == Event.START) {
	    PassUp(evt);
	    StartWork();
	    PassUp(new Event(Event.SET_LOCAL_ADDRESS, local_addr));
	    PassUp(new Event(Event.START_OK));
	    return;
	}	
	PassUp(evt);
    }


//      public void Up(Event evt) {
//  	if(evt.GetType() == Event.START) {
//  	    PassUp(evt);
//  	    PassUp(new Event(Event.START_OK));
//  	    return;
//  	}	
//  	PassUp(evt);
//      }





    /**
     * Caller by the layer above this layer. We just call Route(msg) to pass it on to
     * the remote router.
     */
    public synchronized void Down(Event evt) {
	Message      msg;
	Object       dest_addr, tmp_addr;
	TunnelHeader hdr;

	if(trace) Util.Print("TUNNEL.Down(): event is " + evt);
	
	if(evt.GetType() != Event.MSG) {
	    HandleDownEvent(evt);
	    return;
	}

	hdr=new TunnelHeader(channel_name);
	msg=(Message)evt.GetArg();
	msg.AddHeader(hdr);


	if(msg.GetSrc() == null)
	    msg.SetSrc(local_addr);
       
	try {
	    out.writeObject(msg);
	    out_stream.flush();
	}
	catch(Exception e) {
	    System.err.println("TUNNEL.Down(): " + e);
	}
	
    }





    private void StartWork() {
	RegistrationRequest  reg;

	if(channel_name == null)
	    System.err.println("TUNNEL.StartWork(): cannot get channel name; will not " +
			       "be able to discard messages destined for different channels");

	if(sock == null) {
	    try {
		sock=new Socket(router_host, router_port);
		out_stream=sock.getOutputStream();
		out=new ObjectOutputStream(out_stream);
		in_stream=sock.getInputStream();
		reg=new RegistrationRequest(channel_name, null);
		out.writeObject(reg);
		out_stream.flush();
		in=new ObjectInputStream(in_stream);
		local_addr=(Address)in.readObject();
		receiver=new Thread(this);
		receiver.start();
	    }
	    catch(Exception e) {
		System.err.println("TUNNEL.SetProperties(): " + e);
	    }
	}

    }




    private synchronized void StopWork() {
	RegistrationRequest reg=new RegistrationRequest(channel_name, local_addr);
	reg.unregister=true;
	try {
	    out.writeObject(reg);
	    out_stream.flush();
	}
	catch(Exception e) {
	    System.err.println(e);
	}

	if(receiver != null) {
	    receiver.stop();
	    receiver=null;
	}

	try {
	    sock.close();
	    sock=null;
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }

    /*--------------------------- End of Protocol interface -------------------------- */





    
}
