package JavaGroups.JavaStack.Protocols;


import java.io.*;
import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;




class NakAckHeader implements Externalizable { //Serializable {
    public static final int NAK_MSG          = 1;  // asynchronous msg
    public static final int NAK_ACK_MSG      = 2;  // synchronous msg
    public static final int WRAPPED_MSG      = 3;  // a wrapped msg, needs to be ACKed
    public static final int RETRANSMIT_MSG   = 4;  // retransmit msg
    public static final int NAK_ACK_RSP      = 5;  // ack to NAK_ACK_MSG, seqno contains ACKed
    public static final int OUT_OF_BAND_MSG  = 6;  // out-of-band msg
    public static final int OUT_OF_BAND_RSP  = 7;  // ack for out-of-band msg

    int     type=0;
    long    seqno=-1;          // either reg. NAK_ACK_MSG or first_seqno in retransmissions
    long    last_seqno=-1;     // used for retransmissions
    ViewId  vid=null;
    Vector  stable_msgs=null;  // the messages from this sender can be deleted safely (OutOfBander)
    Object  sender=null;       // In case of WRAPPED_MSG: the address to which an ACK has to be sent


    public NakAckHeader() {}
    


    public NakAckHeader(int type, long seqno, ViewId vid) {
	this.type=type;
	this.seqno=seqno;
	this.vid=vid;
    }




    public void writeExternal(ObjectOutput out) throws IOException {
	out.writeInt(type);
	out.writeLong(seqno);
	out.writeLong(last_seqno);
	out.writeObject(vid);
	out.writeObject(stable_msgs);
	out.writeObject(sender);
    }



    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
	type=in.readInt();
	seqno=in.readLong();
	last_seqno=in.readLong();
	vid=(ViewId)in.readObject();
	stable_msgs=(Vector)in.readObject();
	sender=in.readObject();
    }


    public NakAckHeader Copy() {
	NakAckHeader ret=new NakAckHeader(type, seqno, vid);
	ret.last_seqno=last_seqno;
	ret.stable_msgs=stable_msgs != null ? (Vector)stable_msgs.clone() : null;
	ret.sender=sender;
	return ret;
    }


    public static String Type2Str(int t) {
	switch(t) {
	case NAK_MSG:          return "NAK_MSG";
	case NAK_ACK_MSG:      return "NAK_ACK_MSG";
	case WRAPPED_MSG:      return "WRAPPED_MSG";
	case RETRANSMIT_MSG:   return "RETRANSMIT_MSG";
	case NAK_ACK_RSP:      return "NAK_ACK_RSP";
	case OUT_OF_BAND_MSG:  return "OUT_OF_BAND_MSG";
	case OUT_OF_BAND_RSP:  return "OUT_OF_BAND_RSP";
	default:               return "<undefined>";
	}
    }

    public String toString() {
	StringBuffer ret=new StringBuffer();
	ret.append("[NAKACK: " + Type2Str(type) + ", seqno=" + seqno + ", last_seqno=" + last_seqno +
		   ", vid=" + vid );
	if(type == WRAPPED_MSG)
	    ret.append(", sender=" + sender);
	ret.append("]");

	return ret.toString();
    }

}


/**
   Negative AcKnowledgement layer (NAKs), paired with positive ACKs. The default is to send a message
   using NAKs: the sender sends messages with monotonically increasing seqnos, receiver requests 
   retransmissions of missing messages (gaps). When a SWITCH_NAK_ACK event is received, the mode
   is switched to using NAK_ACKS: the sender still uses monotonically increasing seqnos, but the receiver
   acknowledges every message. NAK and NAK_ACK seqnos are the same, when switching the mode, the current
   seqno is reused. Both NAK and NAK_ACK messages use the current view ID in which the message is sent to
   queue messages destined for an upcoming view, or discard messages sent in a previous view. Both modes
   reset their seqnos to 0 when receiving a view change. The NAK_ACK scheme is used for broadcasting
   view changes.
   <p>
   The third mode is for out-of-band control messages (activated by SWITCH_OUT_OF_BAND): this mode does
   neither employ view IDs, nor does it use the same seqnos as NAK and NAK_ACK. It uses its own seqnos,
   unrelated to the ones used by NAK and NAK_ACK, and never resets them. In combination with the sender's
   address, this makes every out-of-band message unique. Out-of-band messages are used for example for
   broadcasting FLUSH messages.<p>
   Once a mode is set, it remains in effect until exactly 1 message has been sent, afterwards the default
   mode NAK is used again.
   
   The following communication between 2 peers exists (left side is initiator,
   right side receiver): <pre>


   send_out_of_band
   -------------->       synchronous     (1)
   <-------------
        ack


      send_nak
   -------------->       asynchronous    (2)


    send_nak_ack
   -------------->       synchronous     (3)
   <--------------
        ack


      retransmit
   <--------------       asynchronous    (4)


   </pre>
   
   When a message is sent, it will contain a header describing the type of the
   message, and containing additional data, such as sequence number etc. When a
   message is received, it is fed into either the OutOfBander or NAKer, depending on the
   header's type.<p>
   Note that in the synchronous modes, ACKs are sent for each request. If a reliable unicast protocol layer 
   exists somewhere underneath this layer, then even the ACKs are transmitted reliably, thus increasing
   the number of messages exchanged. However, since it is envisaged that ACK/OUT_OF_BAND are not used
   frequently, this problem is currently not addressed.

  */
public class NAKACK extends Protocol {
    NAKer                 naker=new NAKer();
    OutOfBander           out_of_bander=new OutOfBander();
    LosslessTransmission  transmitter=naker;             // NAK is default
    ViewId                vid=null;
    View                  view=null;
    boolean               is_server=false, trace=false;
    Object                local_addr=null;
    List                  queued_msgs=new List();      // msgs for next view (vid > current vid)
    Vector                members=null;                // for OutOfBander: this is the destination set to
                                                       // send messages to
    Vector                old_members=null;
    boolean               send_next_msg_out_of_band=false;
    boolean               send_next_msg_acking=false;
    long                  retransmit_timeout=3000;     // time to wait before requesting retransmission
    long                  rebroadcast_timeout=0;       // until all outstanding ACKs recvd (rebcasting)



    
    class LosslessTransmission {
	void  Start()                                                             {}
	void  Stop()                                                              {}
	void  SetAcks(boolean f)                                                  {}
	void  Reset()                                                             {}
	void  Send(Message msg)                                                   {}
	void  Receive(long id, Message msg, Vector stable_msgs)                   {}
	void  Suspect(Object mbr)                                                 {}
	void  Stable(long[] seqnos)                                               {}
    }
    




    
    class NAKer extends LosslessTransmission implements NakReceiverWindow.RetransmitCommand,
							AckMcastSenderWindow.RetransmitCommand {
	long                  seqno=0;                       // current message sequence number
	Hashtable             received_msgs=new Hashtable(); // ordered by sender -> NakReceiverWindow
	Hashtable             sent_msgs=new Hashtable();     // ordered by seqno (sent by me !)
	AckMcastSenderWindow  sender_win=new AckMcastSenderWindow(this);
	boolean               acking=false;                  // require acks when sending msgs
	long                  deleted_up_to=0;


	long GetNextSeqno() {return seqno++;}




	/** Returns an array of the highest sequence numbers consumed by the application so far,
	    its order corresponding with <code>mbrs</code>. Used by coordinator as argument when
	    sending initial FLUSH request to members */
	long[] GetHighestSeqnosDelivered() {
	    long[]             highest_deliv=members != null ? new long[members.size()] : null;
	    Object             mbr;
	    NakReceiverWindow  win;

	    if(highest_deliv == null) return null;

	    for(int i=0; i < highest_deliv.length; i++) highest_deliv[i]=-1;

	    synchronized(members) {
		for(int i=0; i < members.size(); i++) {
		    mbr=members.elementAt(i);
		    win=(NakReceiverWindow)received_msgs.get(mbr);
		    if(win != null)
			highest_deliv[i]=win.GetHighestDelivered();
		}
	    }
	    return highest_deliv;
	}



	/**
	   Returns a message digest: for each member in <code>highest_seqnos</code>, my own highest
	   sequence number is added to the digest's array. If my highest seqno for a member P is
	   higher than the one in <code>highest_seqnos</code>, then all messages from P received by us
	   whose seqno is higher are added to the digest's messages. The coordinator will use all digests
	   to compute a set of messages than need to be re-broadcast to the members before installing
	   a new view.
	 */
	Digest ComputeMessageDigest(long[] highest_seqnos) {
	    Digest             digest=highest_seqnos != null ? new Digest(highest_seqnos.length) : null;
	    Object             sender;
	    NakReceiverWindow  win;
	    List               unstable_msgs;

	    if(digest == null) {
		System.err.println("NAKACK.NAKer.ComputeMessageDigest(): highest_seqnos is null, "+
				   "cannot compute digest !");
		return null;
	    }

	    if(highest_seqnos.length != members.size()) {
		System.err.println("NAKACK.NAKer.ComputeMessageDigest(): the membership size and the " +
				   "size of the highest_seqnos array are not equal, cannot compute digest !");
		return null;
	    }

	    for(int i=0; i < digest.highest_seqnos.length; i++)
		digest.highest_seqnos[i]=highest_seqnos[i];

	    for(int i=0; i < highest_seqnos.length; i++) {
		sender=members.elementAt(i);
		if(sender == null) continue;
		win=(NakReceiverWindow)received_msgs.get(sender);
		if(win == null) continue;
		digest.highest_seqnos[i]=win.GetHighestReceived();
		unstable_msgs=win.GetMessagesHigherThan(highest_seqnos[i]);
		if(unstable_msgs != null)
		    for(Enumeration e=unstable_msgs.Elements(); e.hasMoreElements();)
			digest.msgs.Add(e.nextElement());
	    }
	    return digest;
	}




	/**
	   For each non-null member m in <code>range</code>, get messages with sequence numbers between
	   range[m][0] and range[m][1], excluding range[m][0] and including range[m][1].
	 */
	List GetMessagesInRange(long[][] range) {
	    List               retval=new List();
	    List               tmp;
	    NakReceiverWindow  win;
	    Object             sender;

	    for(int i=0; i < range.length; i++) {
		if(range[i] != null) {
		    sender=members.elementAt(i);
		    if(sender == null) continue;
		    win=(NakReceiverWindow)received_msgs.get(sender);
		    if(win == null) continue;
		    tmp=win.GetMessagesInRange(range[i][0], range[i][1]);
		    if(tmp == null || tmp.Size() < 1) continue;
		    for(Enumeration e=tmp.Elements(); e.hasMoreElements();)
			retval.Add(e.nextElement());
		}
	    }
	    return retval;
	}


	void SetAcks(boolean f) {acking=f;}


	/**
	   Vector with messages (ordered by sender) that are stable and can be discarded.
	   This applies to NAK-based sender and receivers.
	 */
	void Stable(long[] seqnos) {
	    int                index;
	    long               seqno;
	    NakReceiverWindow  recv_win;
	    Object             sender;

	    if(members == null || local_addr == null) {
		System.err.println("NAKACK.NAKer.Stable(): members or local_addr are null !");
		return;
	    }
	    index=members.indexOf(local_addr);

	    if(index < 0) {
		System.err.println("NAKACK.NAKer.Stable(): member " + local_addr + " not found in " +
				   members);
		return;
	    }
	    seqno=seqnos[index];
	    if(trace) System.out.println("NAKACK: deleting stable messages [" + deleted_up_to +
					 " - " + seqno + "]");

	    // delete sent messages that are stable (kept for retransmission requests from receivers)
	    for(long i=deleted_up_to; i <= seqno; i++)
		sent_msgs.remove(new Long(i));
	    
	    deleted_up_to=seqno;
	    
	    // delete received msgs that are stable
	    for(int i=0; i < members.size(); i++) {
		sender=members.elementAt(i);
		recv_win=(NakReceiverWindow)received_msgs.get(sender);
		if(recv_win != null)
		    recv_win.Stable(seqnos[i]);  // delete all messages with seqnos <= seqnos[i]
	    }
	}






	void Send(Message msg) {
	    long     id=GetNextSeqno();
	    ViewId   vid_copy;

	    if(vid == null) {
		System.err.println("NAKACK.NAKer.Send(): vid is null !");
		return;
	    }
	    vid_copy=vid.Copy();

	    if(acking) {
		msg.AddHeader(new NakAckHeader(NakAckHeader.NAK_ACK_MSG, id, vid_copy));
		sender_win.Add(id, msg.Copy(), (Vector)members.clone()); // msg must be copied !
	    }
	    else
		msg.AddHeader(new NakAckHeader(NakAckHeader.NAK_MSG, id, vid_copy));

	    sent_msgs.put(new Long(id), msg.Copy());
	    PassDown(new Event(Event.MSG, msg));
	}





	/** Re-broadcast message. Message already contains NakAckHeader (therefore also seqno). 
	    Wrap message (including header) in a new message and bcasts using ACKS. Every receiver 
	    acks the message, unwraps it to get the original message and delivers the original message
	    (if not yet delivered).
	    // send msgs in Vector arg again (they already have a NakAckHeader !)
	    // -> use the same seq numbers
	    // -> destination has to be set to null (broadcast), e.g.: 
	    //      dst=p (me !), src=q --> dst=null, src=q

	    TODO:
	    -----
	    Resend() has to wait until it received all ACKs from all recipients (for all msgs), or until
	    members were suspected. Thus we can ensure that all members received outstanding msgs before
	    we switch to a new view. Otherwise, because the switch to a new view resets NAK and ACK msg
	    transmission, slow members might never receive all outstanding messages.
	 */



	/**
	   1. Set the destination address of the original msg to null
	   2. Add a new header WRAPPED_MSG and send msg. The receiver will ACK the msg,
	      remove the header and deliver the msg
	 */
//  	void Resend(Message msg) {
//  	    Message       copy=msg.Copy();
//  	    NakAckHeader  hdr=(NakAckHeader)copy.RemoveHeader();
//  	    NakAckHeader  hdr2;
//  	    long          id=hdr.seqno;
	    
//  	    if(vid == null) {
//  		System.err.println("NAKACK.NAKer.Resend(): vid is null !");
//  		return;
//  	    }

//  	    hdr2=new NakAckHeader(NakAckHeader.NAK_ACK_MSG, hdr.seqno, hdr.vid);
//  	    copy.SetDest(null);                // broadcast, e.g. dst(p), src(q) -->
//  	                                       //              dst(null), src(q)


//  	    //copy.SetSrc(local_addr); // +++++ modify !!!! This leads to replies sent to the wrong guy !

//  	    copy.AddHeader(hdr2);
	    	    
//  	    sender_win.Add(id, copy.Copy(), (Vector)members.clone());

//  	    System.out.println("RESENDING " + copy.PeekHeader());
//  	    PassDown(new Event(Event.MSG, copy));
//  	}




	void Resend(Message msg) {
	    Message       copy=msg.Copy();
	    NakAckHeader  hdr=(NakAckHeader)copy.PeekHeader();
	    NakAckHeader  wrapped_hdr;
	    long          id=hdr.seqno;
	    
	    if(vid == null) {
		System.err.println("NAKACK.NAKer.Resend(): vid is null !");
		return;
	    }

	    copy.SetDest(null);                // broadcast, e.g. dst(p), src(q) -->
	                                       //              dst(null), src(q)

	    wrapped_hdr=new NakAckHeader(NakAckHeader.WRAPPED_MSG, hdr.seqno, hdr.vid);
	    wrapped_hdr.sender=local_addr;
	    
	    copy.AddHeader(wrapped_hdr);

	    sender_win.Add(id, copy.Copy(), (Vector)members.clone());

	    System.out.println("RESENDING " + copy.PeekHeader());
	    PassDown(new Event(Event.MSG, copy));
	}






	void WaitUntilAllAcksReceived(long timeout) {
	    sender_win.WaitUntilAllAcksReceived(timeout);
	}



	void Receive(long id, Message msg, Vector stable_msgs) {
	    Object             sender=msg.GetSrc();
	    NakReceiverWindow  win=(NakReceiverWindow)received_msgs.get(sender);
	    Message            msg_to_deliver, copy;
	    NakAckHeader       hdr;


	    if(win == null) {
		win=new NakReceiverWindow(sender, this, 0);
		win.SetRetransmitPeriod(retransmit_timeout);
		received_msgs.put(sender, win);
	    }

	    if(trace) System.out.println("NAKACK: RECV <" + sender + "#" + id + ">\n");

	    win.Add(id, msg);  // add in order, then remove and pass up as many msgs as possible
	    while(true) {
		msg_to_deliver=win.Remove();
		if(msg_to_deliver == null)
		    break;
		    
		if(msg_to_deliver.PeekHeader() instanceof NakAckHeader)
		    msg_to_deliver.RemoveHeader();
		PassUp(new Event(Event.MSG, msg_to_deliver));
	    }
	}




	void ReceiveAck(long id, Object sender) {
	    if(trace) System.out.println("NAKACK: <-- ACK <" + sender + "#" + id + ">");
	    sender_win.Ack(id, sender);
	}

       

	
	
	/**
	   Implementation of interface AckMcastSenderWindow.RetransmitCommand.<p>
	   Called by retransmission thread of AckMcastSenderWindow. <code>msg</code> is already
	   a copy, so does not need to be copied again.
	*/
	public void Retransmit(long seqno, Message msg, Object dest) {
	    System.out.println("----> ACK: retransmitting message " + seqno + " to " + dest + 
			       ", header is " + msg.PeekHeader());

	    // check whether dest is member of group. If not, discard retransmission message and
	    // also remove it from sender_win (AckMcastSenderWindow)
	    if(members != null) {
		if(!members.contains(dest)) {
		    System.err.println("NAKACK.Retransmit(" + seqno + ") to " + dest + ": " + dest +
				       " is not a member; discarding retransmission and removing " +
				       dest + " from sender_win");
		    sender_win.Remove(dest);
		    return;
		}
	    }

	    msg.SetDest(dest);
	    PassDown(new Event(Event.MSG, msg));
	}
	


	/** 
	    Implementation of NakReceiverWindow.RetransmitCommand.<p>
	    Called by retransmission thread when gap is detected. Sends retr. request
	    to originator of msg
	*/
	public void Retransmit(long first_seqno, long last_seqno, Object sender) {
	    System.out.println("NAKer ----> Retransmit([" + first_seqno + ", " + last_seqno + 
			       "]) to " + sender + ", vid=" + vid);
	    
	    NakAckHeader hdr=new NakAckHeader(NakAckHeader.RETRANSMIT_MSG, first_seqno, vid.Copy());
	    Message      retransmit_msg=new Message(sender, null, null);

	    hdr.last_seqno=last_seqno;
	    retransmit_msg.AddHeader(hdr);
	    PassDown(new Event(Event.MSG, retransmit_msg));
	}



	// Retransmit from sent-table, called when RETRANSMIT message is received
	void Retransmit(Object dest, long first_seqno, long last_seqno) {
	    Message m, retr_msg;

	    for(long i=first_seqno; i <= last_seqno; i++) {
		m=(Message)sent_msgs.get(new Long(i));
  		if(m == null) {
  		    System.err.println("+++ NAKACK.LosslessTransmission.Retransmit(): message with " +
  				       "seqno=" + i + " not found !");
  		    continue;
  		}
		
		retr_msg=m.Copy();
		retr_msg.SetDest(dest);
		
		try {
		    PassDown(new Event(Event.MSG, retr_msg));
		}
		catch(Exception e) {
		    System.err.println("NAKACK.LosslessTransmission.Retransmit(): " + e);
		}
	    }
	}



	void Stop() {
	    if(sender_win != null)
		sender_win.Stop();
	}


	void Reset() {
	    NakReceiverWindow win;

	    // Only reset if not coord: coord may have to retransmit the VIEW_CHANGE msg to slow members,
	    // since VIEW_CHANGE results in retransmitter resetting, retransmission would be killed, and the
	    // slow mbr would never receive a new view (see ./design/ViewChangeRetransmission.txt)
	    if(!Coordinator())
		sender_win.Reset();

	    sent_msgs.clear();	    
	    for(Enumeration e=received_msgs.elements(); e.hasMoreElements();) {
		win=(NakReceiverWindow)e.nextElement();
		win.Reset();
	    }
	    received_msgs.clear();
	    seqno=0;
	    deleted_up_to=0;
	}



	void Suspect(Object mbr) {
	    NakReceiverWindow w;

	    w=(NakReceiverWindow)received_msgs.get(mbr);
	    if(w != null) {
		w.Reset();
		received_msgs.remove(mbr);
	    }

	    sender_win.Suspect(mbr); // don't keep retransmitting messages to mbr
	}


    }
    







    
    class OutOfBander extends LosslessTransmission implements AckMcastSenderWindow.RetransmitCommand {
	AckMcastSenderWindow    sender_win=new AckMcastSenderWindow(this);
	AckMcastReceiverWindow  receiver_win=new AckMcastReceiverWindow();
	long                    seqno=0;




	void Send(Message msg) {
	    Message      copy;
	    long         id=seqno++;
	    Vector       stable_msgs=(Vector)sender_win.GetStableMessages();
	    NakAckHeader hdr;

	    if(trace) System.out.println("OUT_OF_BAND: SEND #" + id);
	    
	    hdr=new NakAckHeader(NakAckHeader.OUT_OF_BAND_MSG, id, null);
	    hdr.stable_msgs=stable_msgs;
  	    msg.AddHeader(hdr);

	    // msg needs to be copied, otherwise it will be modified by the code below
	    sender_win.Add(id, msg.Copy(), (Vector)members.clone());

	    PassDown(new Event(Event.MSG, msg));
	}




	void Receive(long id, Message msg, Vector stable_msgs) {
	    Object sender=msg.GetSrc();	    

	    // first thing: send ACK back to sender
	    Message      ack_msg=new Message(msg.GetSrc(), null, null);
	    NakAckHeader hdr=new NakAckHeader(NakAckHeader.OUT_OF_BAND_RSP, id, null);
	    ack_msg.AddHeader(hdr);


	    if(trace) System.out.println("OUT_OF_BAND: RECV <" + sender + "#" + id + ">\n");

	    if(receiver_win.Add(sender, id))         // not received previously
		PassUp(new Event(Event.MSG, msg));

	    PassDown(new Event(Event.MSG, ack_msg));  // send ACK
	    if(trace) System.out.println("OUT_OF_BAND: --> ACK <" + sender + "#" + id + ">\n");

	    if(stable_msgs != null)
		receiver_win.Remove(sender, stable_msgs);
	}





	void ReceiveAck(long id, Object sender) {
	    if(trace) System.out.println("OUT_OF_BAND: <-- ACK <" + sender + "#" + id + ">");
	    sender_win.Ack(id, sender);
	}
	
	
	
	/**
	   Called by retransmission thread of AckMcastSenderWindow. <code>msg</code> is already
	   a copy, so does not need to be copied again. All the necessary header are already present;
	   no header needs to be added ! The message is retransmitted as <em>unicast</em> !
	*/
	public void Retransmit(long seqno, Message msg, Object dest) {
	    if(trace) System.out.println("---> NAKACK.OutOfBander.Retransmit(): " + dest + "#" + seqno);
	    msg.SetDest(dest);
	    PassDown(new Event(Event.MSG, msg));
	}
	


	void Reset() {
	    sender_win.Reset();       // +++ ?
	    receiver_win.Reset();     // +++ ?
	}
	
	
	void Suspect(Object mbr) {
	    sender_win.Suspect(mbr);
	    receiver_win.Suspect(mbr);
	}
	    

	void Start() {
	    sender_win.Start();
	}

	void Stop() {
	    if(sender_win != null)
		sender_win.Stop();
	}


	
    }
    





    public  String  GetName()       {return "NAKACK";}


    public Vector ProvidedUpServices() {
	Vector retval=new Vector();
	retval.addElement(new Integer(Event.GET_MSGS_RECEIVED));
	retval.addElement(new Integer(Event.GET_MSG_DIGEST));
	retval.addElement(new Integer(Event.GET_MSGS));
	return retval;
    }



    public Vector ProvidedDownServices() {
	Vector retval=new Vector();
	retval.addElement(new Integer(Event.GET_MSGS_RECEIVED));
	return retval;
    }



    
    /**
       <b>Callback</b>. Called by superclass when event may be handled.<p>
       <b>Do not use <code>PassUp</code> in this method as the event is passed up
       by default by the superclass after this method returns !</b>
       @return boolean Defaults to true. If false, event will not be passed up the stack.
     */
    public void Up(Event evt) {
	Object        obj;
	NakAckHeader  hdr;
	Message       msg, msg_copy;
	int           rc;


  	switch(evt.GetType()) {

	case Event.SUSPECT:
	    transmitter.Suspect(evt.GetArg());
	    break;

	case Event.STABLE:  // generated by STABLE layer. Delete stable messages passed in arg
	    naker.Stable((long[])evt.GetArg());
	    break;	    

	case Event.SET_LOCAL_ADDRESS:
	    local_addr=evt.GetArg();
	    break;

	case Event.GET_MSGS_RECEIVED:  // returns the highest seqnos delivered to the appl. (used by STABLE)
	    long[] highest=naker.GetHighestSeqnosDelivered();
	    PassDown(new Event(Event.GET_MSGS_RECEIVED_OK, highest));
	    break;

	case Event.MSG:
	    msg=(Message)evt.GetArg();
	    obj=msg.PeekHeader();

	    if(obj == null || !(obj instanceof NakAckHeader))
		break;  // pass up
	    
	    hdr=(NakAckHeader)msg.RemoveHeader();
	    
	    switch(hdr.type) {

	    case NakAckHeader.NAK_ACK_MSG:
	    case NakAckHeader.NAK_MSG:
	    case NakAckHeader.WRAPPED_MSG:


		System.out.println("NAKACK.Up(): received msg from " + msg.GetSrc() + 
				   ", hdr is " + hdr);

		// remove wrapped header and send back an ACK to hdr.sender
		if(hdr.type == NakAckHeader.WRAPPED_MSG) {
		    Message      ack_msg=new Message(hdr.sender, null, null);
		    NakAckHeader h=new NakAckHeader(NakAckHeader.NAK_ACK_RSP, hdr.seqno, null);

		    if(hdr.sender == null)
			System.err.println("NAKACK.Up(WRAPPED_MSG): header's 'sender' field is null; " +
					   "cannot send ACK !");
		    ack_msg.AddHeader(h);

		    System.out.println("ACKING " + hdr.seqno + " to " + hdr.sender);
		    PassDown(new Event(Event.MSG, ack_msg));

		    System.out.println("Unwrapping msg " + hdr.seqno + ":");
		    hdr=(NakAckHeader)msg.RemoveHeader();  // that's the original 'unwrapped' header
		    System.out.println("DONE unwrapping: new hdr is " + hdr + ", sender is " + msg.GetSrc());
		}


		if(hdr.type == NakAckHeader.NAK_ACK_MSG) {  // first thing: send ACK back to sender
		    Message      ack_msg=new Message(msg.GetSrc(), null, null);
		    NakAckHeader h=new NakAckHeader(NakAckHeader.NAK_ACK_RSP, hdr.seqno, null);
		    ack_msg.AddHeader(h);
		    PassDown(new Event(Event.MSG, ack_msg));
		}

		// while still a client, we just pass up all messages, without checking for message
		// view IDs or seqnos: other layers further up will discard messages not destined 
		// for us (e.g. based on view IDs).
		// Also: store msg in queue, when view change is received, replay messages with the same
		// vid as the new view
		if(!is_server) {
		    msg_copy=msg.Copy();                // msg without header
		    msg_copy.AddHeader(hdr);            // put header back on as we removed it above
		    queued_msgs.Add(msg_copy);          // need a copy since PassUp will modify msg
		    PassUp(new Event(Event.MSG, msg));
		    return;
		}


		// check for VIDs: is the message's VID the same as ours ?
		if(vid != null && hdr.vid != null) { // only check if our vid and message's vid available
		    Object my_addr=vid.GetCoordAddress(), other_addr=hdr.vid.GetCoordAddress();

		    if(my_addr == null || other_addr == null) {
			System.err.println("NAKACK.Up(): my vid or message's vid does not contain " +
					   "a coordinator; discarding message !");
			return;
		    }
		    if(!my_addr.equals(other_addr)) {
			System.err.println("NAKACK.Up(): creator of own vid (" + my_addr + 
					   ")is different from creator of message's vid (" + 
					   other_addr + "); discarding message !");
			return;
		    }

		    rc=hdr.vid.Compare(vid);
		    if(rc > 0) {           // message is sent in next view -> store !
			System.out.println("NAKACK.Up(): message's vid (" + hdr.vid + "#" + hdr.seqno + 
					   ") is bigger than current vid: (" + vid+
					   ") message is queued !");
			msg.AddHeader(hdr);  // put header back on as we removed it above
			queued_msgs.Add(msg);
			return;
		    }
		    if(rc < 0) {      // message sent in prev. view -> discard !
			System.err.println("NAKACK.Up(): message's vid ("+ hdr.vid +") is smaller than " +
					   "current vid ("+vid+"): message <"+ msg.GetSrc() +
					   ":#" + hdr.seqno + "> is discarded ! " +
					   "Hdr is " + hdr);
			return;
		    }
		    // If we made it down here, the vids are the same --> OK
		}


		msg.AddHeader(hdr);   // stored in received_msgs, re-sent later that's why hdr is added !
		naker.Receive(hdr.seqno, msg, null);
		return;        // transmitter passes message up for us !


	    case NakAckHeader.RETRANSMIT_MSG:
		naker.Retransmit(msg.GetSrc(), hdr.seqno, hdr.last_seqno);
		return;

	    case NakAckHeader.NAK_ACK_RSP:
		naker.ReceiveAck(hdr.seqno, msg.GetSrc());
		return;        // discard, no need to pass up

	    case NakAckHeader.OUT_OF_BAND_MSG:
		out_of_bander.Receive(hdr.seqno, msg, hdr.stable_msgs);
		return;        // transmitter passes message up for us !

	    case NakAckHeader.OUT_OF_BAND_RSP:
		out_of_bander.ReceiveAck(hdr.seqno, msg.GetSrc());
		return;

	    default:
		System.err.println("NAKACK.Up(): NakAck header type " +
				   hdr.type + " not known !");
		break;
	    }
	    
  	}

	PassUp(evt);
    }







    /**
       <b>Callback</b>. Called by superclass when event may be handled.<p>
       <b>Do not use <code>PassDown</code> in this method as the event is passed down
       by default by the superclass after this method returns !</b>
       @return boolean Defaults to true. If false, event will not be passed down the stack.
    */
    public void Down(Event evt) {
	Message msg;

	switch(evt.GetType()) {

	case Event.MSG:
	    msg=(Message)evt.GetArg();
	    if(msg.GetDest() != null && !((Address)msg.GetDest()).IsMulticastAddress())
		break; // unicast address: not null and not mcast, pass down unchanged

	    if(send_next_msg_out_of_band) {
		out_of_bander.Send(msg);
		send_next_msg_out_of_band=false;
	    }
	    else if(send_next_msg_acking) {
		transmitter.SetAcks(true);  // require acks when sending a msg
		transmitter.Send(msg);
		transmitter.SetAcks(false);  // don't require acks when sending a msg
		send_next_msg_acking=false;
	    }
	    else
		transmitter.Send(msg);
	    return;    // don't pass down the stack, transmitter does this for us !
	    
	case Event.GET_MSG_DIGEST:
	    long[] highest_seqnos=(long[])evt.GetArg();
	    Digest digest=naker.ComputeMessageDigest(highest_seqnos);
	    PassUp(new Event(Event.GET_MSG_DIGEST_OK, digest));
	    return;

	case Event.GET_MSGS:
	    List lower_seqnos=naker.GetMessagesInRange((long[][])evt.GetArg());
	    PassUp(new Event(Event.GET_MSGS_OK, lower_seqnos));
	    return;



	case Event.REBROADCAST_MSGS:
	    Vector        v=(Vector)evt.GetArg(), final_v=new Vector();
	    Message       m1, m2;
	    NakAckHeader  h1, h2;
	    
	    // weed out duplicates	    
	outer:
	    for(int i=0; i < v.size(); i++) {
		m1=(Message)v.elementAt(i);
		h1=(NakAckHeader)m1.PeekHeader();

		for(int j=0; j < final_v.size(); j++) {
		    m2=(Message)final_v.elementAt(j);
		    h2=(NakAckHeader)m2.PeekHeader();

		    if(h1.seqno == h2.seqno && m1.GetSrc().equals(m2.GetSrc())) {
			continue outer;
		    }
		}
		final_v.addElement(m1);
	    }

	    System.out.println("NAKACK.REBROADCAST: rebroadcasting " + final_v.size() + " messages");

	    /* Now re-broadcast messages using original NakAckHeader (same seqnos, same sender !) */
	    for(int i=0; i < final_v.size(); i++) {
		m1=(Message)final_v.elementAt(i);
		naker.Resend(m1);
	    }

	    // Wait until all members have acked reception of outstanding msgs. This will empty our
	    // retransmission table (AckMcastSenderWindow)

	    System.out.println("WAIT_UNTIL_ALL_ACKS_RECEIVED:");
	    naker.WaitUntilAllAcksReceived(rebroadcast_timeout);
	    System.out.println("WAIT_UNTIL_ALL_ACKS_RECEIVED: DONE");

	    PassUp(new Event(Event.REBROADCAST_MSGS_OK));
	    break;


	case Event.TMP_VIEW:
	    Vector mbrs=((View)evt.GetArg()).GetMembers();
	    old_members=members != null ? (Vector)members.clone() : null;
	    members=mbrs != null ? (Vector)mbrs.clone(): new Vector();
	    break;


	case Event.VIEW_CHANGE:
	    view=((View)evt.GetArg()).Copy();
	    vid=view.GetVid();

	    members=(Vector)view.GetMembers().clone();

	    naker.Reset();
	    out_of_bander.Reset();

	    is_server=true;  // check vids from now on

	    // deliver messages received previously for this view
	    if(queued_msgs.Size() > 0)
		DeliverQueuedMessages();
	    break;


	case Event.BECOME_SERVER:
	    is_server=true;
	    break;

	case Event.SWITCH_NAK:
	    transmitter.SetAcks(false); // don't require acks when sending a msg
	    return;                     // don't pass down any further

	case Event.SWITCH_NAK_ACK:
	    send_next_msg_acking=true;
	    return;                     // don't pass down any further

	case Event.SWITCH_OUT_OF_BAND:
	    send_next_msg_out_of_band=true;
	    return;

	case Event.STOP:
	    out_of_bander.Stop();
	    naker.Stop();
	    break;

	case Event.GET_MSGS_RECEIVED:  // return the highest seqnos delivered (=consumed by the application)
	    long[] h=naker.GetHighestSeqnosDelivered();
	    PassUp(new Event(Event.GET_MSGS_RECEIVED_OK, h));
	    break;
	}


	PassDown(evt);
    }



    boolean Coordinator() {
	if(members == null || members.size() < 1 || local_addr == null)
	    return false;
	return local_addr.equals(members.elementAt(0));
    }



    /**
       Deliver all messages in the queue where <code>msg.vid == vid</code> holds. Messages were stored
       in the queue because their vid was greater than the current view.
     */
    void DeliverQueuedMessages() {
	NakAckHeader hdr;
	Message      tmpmsg;
	int          rc;
	
	while(queued_msgs.Size() > 0) {
	    tmpmsg=(Message)queued_msgs.RemoveFromHead();
	    hdr=(NakAckHeader)tmpmsg.PeekHeader();
	    rc=hdr.vid.Compare(vid);
	    if(rc == 0) {               // same vid -> OK
		System.out.println("NAKACK.DeliverQueuedMessages: delivering msg: <" + 
				   tmpmsg.GetSrc() + ":#" + hdr.seqno + ">");
		Up(new Event(Event.MSG, tmpmsg));
	    }
	    else if(rc > 0) {
		System.err.println("************ NAKACK.DeliverQueuedMessages(): "+
				   "discarding msg for future view: <" +
				   tmpmsg.GetSrc() + ":#" + hdr.seqno + ">");
		Util.DumpStack(false);
	    }
	    else
		; // can't be the case; only messages for future views are stored !
	}
    }
    


    public boolean SetProperties(Properties props) {
	String     str;
	
	this.props=props;
	str=props.getProperty("trace");
	if(str != null) {
	    trace=new Boolean(str).booleanValue();
	    props.remove("trace");
	}

	str=props.getProperty("retransmit_timeout");
	if(str != null) {
	    retransmit_timeout=new Long(str).longValue();
	    props.remove("retransmit_timeout");
	}

	if(retransmit_timeout <= 0) {
	    System.err.println("NAKACK.SetProperties(): retransmit_timeout cannout be <= 0 !" +
			       "Setting it to 3000");
	    retransmit_timeout=3000;
	}

	if(props.size() > 0) {
	    System.err.println("NAKACK.SetProperties(): these properties are not recognized:");
	    props.list(System.out);
	    return false;
	}
	return true;
    }







    public static void main(String[] args) {
	NakAckHeader   hdr1=new NakAckHeader(NakAckHeader.NAK_MSG, 22, null), hdr2;
	byte           buf[];

	try {
	    System.out.println("hdr1 is " + hdr1);
	    buf=Util.ObjectToByteBuffer(hdr1);
	    hdr2=(NakAckHeader)Util.ObjectFromByteBuffer(buf);
	    System.out.println("hdr2 is " + hdr2);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }

}
