package JavaGroups.JavaStack.Protocols;


import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;







/**
   Group membership protocol. Handles joins/leaves/crashes (suspicions) and emits new views
   accordingly. Use VIEW_ENFORCER on top of this layer to make sure new members don't receive
   any messages until they are members.
 */
public class GMS extends RpcProtocol implements Runnable {
    private GmsImpl             impl=ClientGmsImpl.CreateInstance(this);
    public  Properties          props=null;
    public  Address             local_addr=null;
    public  String              group_addr=null;
    public  Membership          members=new Membership();
    public  ViewId              view_id=null;
    public  long                ltime=0; 
    public  long                initial_mbrs_timeout=5000;
    public  long                join_timeout=5000;
    public  long                join_retry_timeout=2000;
    private long                flush_timeout=0;            // 0=wait forever until FLUSH completes
    private long                rebroadcast_timeout=0;
    private long                view_change_timeout=10000;  // until all HandleViewChange() RPCs have returned
    public  long                leave_timeout=5000;
    public  boolean             trace=false;
    public  Object              impl_mutex=new Object();    // synchronizes event entry into impl
    public  Object              view_mutex=new Object();    // synchronizes view installations
    private Queue               event_queue=new Queue();    // stores SUSPECT, MERGE events
    private Thread              evt_thread=null;
    private Object              flush_mutex=new Object();
    private FlushRsp            flush_rsp=null;
    private Object              rebroadcast_mutex=new Object();
    private boolean             rebroadcast_unstable_msgs=true;


    public String   GetName()          {return "GMS";}

    public Vector   RequiredDownServices() {
	Vector retval=new Vector();
	retval.addElement(new Integer(Event.FLUSH));
	return retval;
    }


    public void SetImpl(GmsImpl new_impl) {
	synchronized(impl_mutex) {
	    impl=new_impl;
	    System.out.println("Changed role to " + new_impl.getClass().getName());
	}
    }




    /**
       Computes the next view. Returns a copy that has <code>old_mbrs</code> and
       <code>suspected_mbrs</code> removed and <code>new_mbrs</code> added.
    */
    public View GetNextView(Vector new_mbrs, Vector old_mbrs, Vector suspected_mbrs) {
	Vector      mbrs;
	long        vid=0;
	View        v;
	Membership  tmp_mbrs;
	Vector      mbrs_to_remove=new Vector();



	if(old_mbrs != null && old_mbrs.size() > 0)
	    for(int i=0; i < old_mbrs.size(); i++)
		mbrs_to_remove.addElement(old_mbrs.elementAt(i));
	if(suspected_mbrs != null && suspected_mbrs.size() > 0)
	    for(int i=0; i < suspected_mbrs.size(); i++)
		if(!mbrs_to_remove.contains(suspected_mbrs.elementAt(i)))
		    mbrs_to_remove.addElement(suspected_mbrs.elementAt(i));
	    
	synchronized(view_mutex) {
	    vid=Math.max(view_id.GetId(), ltime) + 1;
	    ltime=vid;
	    tmp_mbrs=members.Copy();
	    tmp_mbrs.Merge(new_mbrs, mbrs_to_remove);
	    mbrs=(Vector)tmp_mbrs.GetMembers().clone();
	    v=new View(local_addr, vid, mbrs);
	    return v;
	}
    }




    /** Return a copy of the current membership minus the suspected members: FLUSH request is not sent
	to suspected members (because they won't respond, and not to joining members either.
	It *is* sent to leaving members (before they are allowed to leave). */
    Vector ComputeFlushDestination(Vector suspected_mbrs) {
	Vector ret=members.GetMembers(); // *copy* of current membership
	if(suspected_mbrs != null && suspected_mbrs.size() > 0)
	    for(int i=0; i < suspected_mbrs.size(); i++)
		ret.removeElement(suspected_mbrs.elementAt(i));
	return ret;
    }




    /**
       Compute a new view, given the current view, the new members and the suspected/left
       members.  Run view update protocol to install a new view in all members (this involves
       casting the new view to all members). The targets for FLUSH and VIEW mcasts are
       computed as follows:<p> G is the current membership, N is the set of new members, L is
       the set of leaving members, S is the set of suspected members.
       The FLUSH mcast is sent to G minus S (including L). The VIEW mcast is sent to G plus N
       minus S (all existing members, plus the new member, excluding suspected members).  
    */
    public void CastViewChange(Vector new_mbrs, Vector old_mbrs, Vector suspected_mbrs) {
	Rsp       rsp;
	View      new_view;
	ViewId    new_vid;
	Object    mbr;
	Vector    rebroadcast_msgs=new Vector();
	Vector    dests=ComputeFlushDestination(suspected_mbrs);  // members to which FLUSH/VIEW is sent

	if(suspected_mbrs == null)
	    suspected_mbrs=new Vector();

	// next view: current mbrs + new_mbrs - old_mbrs - suspected_mbrs
	new_view=GetNextView(new_mbrs, old_mbrs, suspected_mbrs);  
	new_vid=new_view.GetVid();

	/** FLUSH protocol. 
	    Send to current mbrs - suspected_mbrs (not including new_mbrs, but including old_mbr)
	    Send TMP_VIEW event down, 
	    this allows FLUSH/NAKACK to set membership correctly */
	while(dests.size() > 0) {
	    flush_rsp=null;	
	    synchronized(flush_mutex) {
		PassDown(new Event(Event.FLUSH, dests));  // send FLUSH to members in dests
		try {flush_mutex.wait(flush_timeout);}
		catch(Exception e) {}
	    }
	    if(flush_rsp == null) {
		System.err.println("GMS.CastViewChange(): FLUSH rsp is null due to timeout. FLUSH " +
				   "may not have been successful !");
		break;
	    }

	    if(rebroadcast_unstable_msgs && flush_rsp.unstable_msgs != null && 
	       flush_rsp.unstable_msgs.size() > 0) {
		// System.out.println("FLUSH returned the following unstable msgs:");		    
		Message m;
		for(int i=0; i < flush_rsp.unstable_msgs.size(); i++) {
		    m=(Message)flush_rsp.unstable_msgs.elementAt(i);
		    		    
		    // just add msg, NAKACK.RESEND will weed out duplicates based on
		    // <sender:id> before re-broadcasting msgs
		    rebroadcast_msgs.addElement(m);
		}
	    }

	    if(flush_rsp.result == true)
		break;
	    else {
		if(flush_rsp.failed_mbrs != null) {
		    for(int i=0; i < flush_rsp.failed_mbrs.size(); i++) {
			dests.removeElement(flush_rsp.failed_mbrs.elementAt(i));
			suspected_mbrs.addElement(flush_rsp.failed_mbrs.elementAt(i));
		    }
		}
	    }
	} // while


	// +++ Rebroadcast unstable messages
	if(rebroadcast_unstable_msgs && rebroadcast_msgs.size() > 0) {
	    System.out.println("GMS.CastViewChange(): there are " + rebroadcast_msgs.size() +
			       " messages to be re-broadcast");
	    
	    // NAKACK layer will rebroadcast the msgs (using the same seqnos assigned earlier)
	    synchronized(rebroadcast_mutex) {
		PassDown(new Event(Event.REBROADCAST_MSGS, rebroadcast_msgs));
		try {
		    rebroadcast_mutex.wait(rebroadcast_timeout);
		}
		catch(Exception e) {}
	    }
	}

       

	/* VIEW protocol. Send to current mbrs + new_mbrs - suspected_mbrs - old_mbrs.  Since
	   suspected members were removed from dests during the previous FLUSH round(s), we
	   only need to add the new members.  Send TMP_VIEW event down, this allows
	   FLUSH/NAKACK to set membership correctly */
	
	if(new_mbrs != null && new_mbrs.size() > 0)
	    for(int i=0; i < new_mbrs.size(); i++)
		dests.addElement(new_mbrs.elementAt(i));

	if(old_mbrs != null && old_mbrs.size() > 0)
	    for(int i=0; i < old_mbrs.size(); i++)
		dests.removeElement(old_mbrs.elementAt(i));



	Event view_event=new Event(Event.TMP_VIEW, new_view);
	PassDown(view_event); // needed e.g. by failure detector or UDP

	PassDown(new Event(Event.SWITCH_NAK_ACK));  // use ACK scheme for view bcast
	System.out.println("\n********** ==> 2. Casting VIEW_CHANGE(" + new_vid + ") --> " + dests +
			   " <== ***********");
	RspList rsp_list=CallRemoteMethods(dests, "HandleViewChange",
					   new_vid, dests, GroupRequest.GET_ALL, view_change_timeout);
	System.out.println("*********** ==> VIEW DONE <== ************");

	PassDown(new Event(Event.SWITCH_NAK));  // back to normal NAKs ...
    }



    



    /**
       Assigns the new ltime. Installs view and view_id. Changes role to coordinator if necessary.
       Sends VIEW_CHANGE event up and down the stack.
     */
    public void InstallView(ViewId new_view, Vector mbrs) {
	Object coord;
	int    rc;

	//System.out.println("GMS.InstallView(" + new_view + ")");

	synchronized(view_mutex) {                    // serialize access to views
	    ltime=Math.max(new_view.GetId(), ltime);  // compute Lamport's logical time
	    System.out.println("==> Received VIEW_CHANGE(" + new_view + ")");

	    /* Check for self-inclusion: if I'm not part of the new membership, I just discard it.
	       This ensures that messages sent in view V1 are only received by members of V1 */
	    if(CheckSelfInclusion(mbrs) == false) {
		System.err.println("GMS.InstallView(): I'm not member of " + mbrs + ", discarding");
		return;
	    }


	    if(view_id == null) {
		if(new_view == null) {
		    System.err.println("GMS.InstallView(): view_id and new_view are null !");
		    return;
		}
		else {  // view_id is null, new_view not: just install new_view (we're still a client)
		    view_id=new_view.Copy();
		}
	    }
	    else {
		if(new_view == null) {  // this should never happen though !
		    System.err.println("GMS.InstallView(): new_view is null !");
		    return;
		}
		else {  // both view_id and new_view are not null
		    rc=new_view.Compare(view_id);  // rc should always be a positive number
		    if(rc <= 0) {  // don't accept view id lower than our own
			Util.Print("**** ViewChange, rc is " + rc + ": received view <= current " +
				   "view; discarding it ! ***");
			Util.Print("view_id: " + view_id + "\nnew_view: " + new_view);
			return;
		    }
		    else {  // the check for vid equality was okay, assign new_view to view_id
			
			if(new_view.GetCoordAddress() != null) {
			    view_id=new ViewId(new_view.GetCoordAddress(), new_view.GetId());
			}
			else {
			    view_id=new ViewId(view_id.GetCoordAddress(), new_view.GetId());
			}
		    }
		}
	    }

	    if(mbrs != null && mbrs.size() > 0)
		members.Set(mbrs);



	    // Send VIEW_CHANGE event up and down the stack:
	    Event view_event=new Event(Event.VIEW_CHANGE, MakeView(members.GetMembers()));
	    PassDown(view_event); // needed e.g. by failure detector or UDP
	    PassUp(view_event);

	    coord=DetermineCoordinator();
	    if(coord != null && coord.equals(local_addr) && !(coord.equals(new_view.GetCoordAddress())))
		SetImpl(CoordGmsImpl.CreateInstance(this));
	}
    }



    protected Address DetermineCoordinator() {
	synchronized(members) {
	    return members != null && members.size() > 0 ? (Address)members.elementAt(0) : null;
	}
    }



    /** Returns true if local_addr is member of mbrs, else false */
    protected boolean CheckSelfInclusion(Vector mbrs) {
	Object mbr;
	if(mbrs == null)
	    return false;
	for(int i=0; i < mbrs.size(); i++) {
	    mbr=mbrs.elementAt(i);
	    if(mbr != null && local_addr.equals(mbr))
		return true;
	}
	return false;
    }


    public View MakeView(Vector mbrs) {
	Object  coord=null;
	long    id=0;

	if(view_id != null) {
	    coord=view_id.GetCoordAddress();
	    id=view_id.GetId();
	}
	return new View(coord, id, mbrs);
    }


    public View MakeView(Vector mbrs, ViewId vid) {
  	Object  coord=null;
  	long    id=0;
	
  	if(vid != null) {
  	    coord=vid.GetCoordAddress();
  	    id=vid.GetId();
  	}
  	return new View(coord, id, mbrs);
    }




    /* ------------------------- Request handler methods ----------------------------- */

    public void Join(Address mbr) {
	synchronized(impl_mutex) {
	    impl.Join(mbr);
	}
    }

    public void Leave(Address mbr) {
	synchronized(impl_mutex) {
	    impl.Leave(mbr);
	}
    }

    public void Suspect(Address mbr) {
	synchronized(impl_mutex) {
	    impl.Suspect(mbr);
	}
    }

    public void Merge(Vector new_mbrs) {
	synchronized(impl_mutex) {
	    impl.Merge(new_mbrs);
	}
    }

    public boolean HandleJoin(Address mbr) {
	return impl.HandleJoin(mbr);	
    }

    public void HandleLeave(Address mbr, boolean suspected) {
	impl.HandleLeave(mbr, suspected);	
    }

    public void HandleViewChange(ViewId new_view, Vector mbrs) {
	impl.HandleViewChange(new_view, mbrs);	
    }

    public void HandleMerge(Vector new_mbrs, Vector suspects, long other_ltime) {
	impl.HandleMerge(new_mbrs, suspects, other_ltime);	
    }

    public void HandleSuspect(Address mbr) {
	impl.HandleSuspect(mbr);	
    }

    /* --------------------- End of Request handler methods -------------------------- */



    boolean CheckForViewEnforcer(Protocol up_protocol) {
	String prot_name;

	if(up_protocol == null)
	    return false;
	prot_name=up_protocol.GetName();
	if(prot_name != null && prot_name.equals("VIEW_ENFORCER"))
	    return true;
	return CheckForViewEnforcer(up_protocol.GetUpProtocol());
    }


    
    /**
       <b>Callback</b>. Called by superclass when event may be handled.<p>
       <b>Do not use <code>PassUp</code> in this method as the event is passed up
       by default by the superclass after this method returns !</b>
       @return boolean Defaults to true. If false, event will not be passed up the stack.
     */
    public boolean HandleUpEvent(Event evt) {
	ViewId        vid;
	Object        obj, sender;
	Message       m;


	switch(evt.GetType()) {

	    
	case Event.START:

	    if(CheckForViewEnforcer(up_prot) == false) {
		System.err.println("\nGMS.HandleUpEvent(START): warning, I need protocol layer " +
				   "VIEW_ENFORCER above me to discard messages sent to me while I'm " +
				   "not yet a group member ! Otherwise, these messages will be delivered " +
				   "to the application without checking...\n");
	    }

	    if(_corr != null)
		_corr.SetDeadlockDetection(true);
	    else
		System.err.println("Cannot set deadlock detection in corr, as it is null !");
	    break;


	case Event.CONNECT_OK:     // sent by someone else, but WE are responsible for sending this !
	case Event.DISCONNECT_OK:  // dito (e.g. sent by UDP layer)
	    return false;
	    

	case Event.SET_LOCAL_ADDRESS:
	    local_addr=(Address)evt.GetArg();
	
	    Util.Print("\n----------------------------------------------------------\n" +
		       "GMS: my address is " + local_addr + 
		       "\n----------------------------------------------------------");
	    
	    return true;                         // pass up

	case Event.SUSPECT:
	    try {event_queue.Add(evt);}
	    catch(Exception e) {}
	    return true;                         // pass up

	case Event.MERGE:
	    try {event_queue.Add(evt);}
	    catch(Exception e) {}
	    return false;                        // don't pass up


	case Event.FLUSH_OK:
	    synchronized(flush_mutex) {
		flush_rsp=(FlushRsp)evt.GetArg();
		flush_mutex.notify();
	    }
	    return false;                        // don't pass up	    

	case Event.REBROADCAST_MSGS_OK:
	    synchronized(rebroadcast_mutex) {
		rebroadcast_mutex.notify();
	    }
	    return false;                        // don't pass up
	}

	return impl.HandleUpEvent(evt);
    }




    /**
       <b>Callback</b>. Called by superclass when event may be handled.<p>
       <b>Do not use <code>PassDown</code> in this method as the event is passed down
       by default by the superclass after this method returns !</b>
       @return boolean Defaults to true. If false, event will not be passed down the stack.
    */
    public boolean HandleDownEvent(Event evt) {	
	Address dest;
	Message msg;

	switch(evt.GetType()) {
		
	case Event.CONNECT:
	    try {
		group_addr=(String)evt.GetArg();
	    }
	    catch(ClassCastException cce) {
		System.err.println("GMS.HandleDownEvent(CONNECT): group address must " +
				   "be a string (group name) to make sense");
	    }

	    PassDown(evt);
	    impl.Join(local_addr);
	    PassUp(new Event(Event.CONNECT_OK));
	    StartEventHandlerThread();
	    return false;                        // don't pass down: was already passed down
		
	case Event.DISCONNECT:
	    impl.Leave((Address)evt.GetArg());
	    PassUp(new Event(Event.DISCONNECT_OK));
	    StopEventHandlerThread();
	    return true;                         // pass down
	}	

	return impl.HandleDownEvent(evt);
    }


    // Priority handling, otherwise GMS.Down(DISCONNECT) would block !
    // Smilar to FLUSH protocol
    public void ReceiveDownEvent(Event evt) {
	if(evt.GetType() == Event.BLOCK_OK) {
	    PassDown(evt);
	    return;
	}	
	super.ReceiveDownEvent(evt);
    }




    /** Setup the Protocol instance acording to the configuration string */
    public boolean SetProperties(Properties props) {
	String     str;
	this.props=props;

	str=props.getProperty("trace");
	if(str != null) {
	    trace=new Boolean(str).booleanValue();
	    props.remove("trace");
	}

 	str=props.getProperty("initial_mbrs_timeout");   // time to wait for initial member retrieval
 	if(str != null) {
 	    initial_mbrs_timeout=new Long(str).longValue();
	    props.remove("initial_mbrs_timeout");
	}

 	str=props.getProperty("join_timeout");           // time to wait for JOIN
 	if(str != null) {
 	    join_timeout=new Long(str).longValue();
	    props.remove("join_timeout");
	}

 	str=props.getProperty("view_change_timeout");    // time to wait for VIEW_CHANGE
 	if(str != null) {
 	    view_change_timeout=new Long(str).longValue();
	    props.remove("view_change_timeout");
	}

 	str=props.getProperty("join_retry_timeout");     // time to wait between JOINs
 	if(str != null) {
 	    join_retry_timeout=new Long(str).longValue();
	    props.remove("join_retry_timeout");
	}

 	str=props.getProperty("leave_timeout");           // time to wait until coord responds to LEAVE req.
 	if(str != null) {
 	    leave_timeout=new Long(str).longValue();
	    props.remove("leave_timeout");
	}

 	str=props.getProperty("flush_timeout");           // time to wait until FLUSH completes (0=forever)
 	if(str != null) {
 	    flush_timeout=new Long(str).longValue();
	    props.remove("flush_timeout");
	}

 	str=props.getProperty("rebroadcast_unstable_msgs");  // bcast unstable msgs (recvd from FLUSH)
 	if(str != null) {
 	    rebroadcast_unstable_msgs=new Boolean(str).booleanValue();
	    props.remove("rebroadcast_unstable_msgs");
	}

 	str=props.getProperty("rebroadcast_timeout");     // time to wait until REBROADCAST_MSGS completes
 	if(str != null) {
 	    rebroadcast_timeout=new Long(str).longValue();
	    props.remove("rebroadcast_timeout");
	}

	if(props.size() > 0) {
	    System.err.println("GMS.SetProperties(): the following properties are not recognized:");
	    props.list(System.out);
	    return false;
	}
	return true;
    }



    public void run() {
	Event evt;

	while(true) {
	    try {
		evt=(Event)event_queue.Remove();
		switch(evt.GetType()) {
		case Event.SUSPECT:
		    impl.Suspect((Address)evt.GetArg());
		    break;
		case Event.MERGE:
		    impl.Merge((Vector)evt.GetArg());
		    break;
		default:
		    System.out.println("GMS.run(): event handler thread encountered event of type " +
				       Event.Type2String(evt.GetType()) + ": not handled by me !");
		    break;
		}
	    }
	    catch(QueueClosed closed) {
		break;
	    }
	}
    }



    /* ------------------------------- Private Methods --------------------------------- */

    private void StartEventHandlerThread() {
	if(evt_thread == null) {
	    evt_thread=new Thread(this, "GMS.EventHandlerThread");
	    evt_thread.start();
	}
    }


    private void StopEventHandlerThread() {
	if(evt_thread != null) {
	    evt_thread.stop();
	    evt_thread=null;
	    event_queue.Reset();
	}
    }


}
