package JavaGroups.JavaStack.Protocols;

import java.io.*;
import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;


class FdHeader implements Serializable {

    static final int HEARTBEAT     = 0;
    static final int HEARTBEAT_ACK = 1;
    static final int SUSPECT       = 2;
    static final int REGULAR       = 3;


    int     type=HEARTBEAT;
    Object  suspected_mbr=null;

    FdHeader(int type) {this.type=type;}


    public String toString() {
	switch(type) {
	case HEARTBEAT:
	    return "[FD: heartbeat]";
	case HEARTBEAT_ACK:
	    return "[FD: heartbeat ack]";
	case SUSPECT:
	    return "[FD: suspect]";
	case REGULAR:
	    return "[FD: regular message]";
	default:
	    return "[FD: unknown type (" + type + ")]";
	}
    }
}




/**
   Failure detection based on simple heartbeat protocol. Regularly polls members for
   liveness. Passes SUSPECT message up the stack when a member is not reachable. The simple
   algorithms works as follows: the membership is known and ordered. Each HB protocol
   periodically sends a 'are-you-alive' message to its *neighbor*. A neighbor is the next in
   rank in the membership list. It is recomputed upon a view change. When a response hasn't
   been received for n milliseconds and m tries, the corresponding member is suspected (and
   eventually excluded if faulty).<p>
   FD starts when it detects (in a view change notification) that there are at least 
   2 members in the group. It stops running when the membership drops below 2.<p>
   When a message is received from the monitored neighbor member, it causes the pinger thread to
   'skip' sending the next are-you-alive message. Thus, traffic is reduced.
*/

public class FD extends Protocol implements Runnable {
    boolean      trace=false;
    Object       ping_dest=null;
    Object       local_addr=null;
    Thread       pinger=null;
    long         timeout=2000;
    boolean      ack_received=false, skip_heartbeat=false;
    int          num_tries=0;
    int          max_tries=2;
    Vector       members=null;



    public String  GetName() {return "FD";}


    public boolean SetProperties(Properties props) {
	String     str;

	this.props=props;
	str=props.getProperty("trace");
	if(str != null) {
	    trace=new Boolean(str).booleanValue();
	    props.remove("trace");
	}
	
	str=props.getProperty("timeout");
	if(str != null) {
	    timeout=new Long(str).longValue();
	    props.remove("timeout");
	}

	str=props.getProperty("max_tries");  // before suspecting a member
	if(str != null) {
	    max_tries=new Integer(str).intValue();
	    props.remove("max_tries");
	}
	if(props.size() > 0) {
	    System.err.println("FD.SetProperties(): the following properties are not recognized:");
	    props.list(System.out);
	    return false;
	}
	return true;
    }



    Object GetPingDest(Vector members) {
	Object tmp, retval=null;

	if(members == null || members.size() < 2 || local_addr == null)
	    return null;
	for(int i=0; i < members.size(); i++) {
	    tmp=members.elementAt(i);
	    if(local_addr.equals(tmp)) {
		if(i + 1 >= members.size())
		    retval=members.elementAt(0);		    
		else
		    retval=members.elementAt(i+1);
		break;
	    }
	}
	return retval;
    }



    public void Up(Event evt) {
	Message   msg;
	FdHeader  hdr=null;
	Object    sender;

	switch(evt.GetType()) {

	case Event.SET_LOCAL_ADDRESS:
	    local_addr=evt.GetArg();
	    break;
	    
	case Event.MSG:
	    msg=(Message)evt.GetArg();

	    try {
		hdr=(FdHeader)msg.RemoveHeader();
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }

	    switch(hdr.type) {
	    case FdHeader.HEARTBEAT:                       // heartbeat request; send heartbeat ack
		Message  hb_ack=new Message(msg.GetSrc(), null, null);
		FdHeader tmp_hdr=new FdHeader(FdHeader.HEARTBEAT_ACK);
		tmp_hdr.suspected_mbr=local_addr;		
		hb_ack.AddHeader(tmp_hdr);
		PassDown(new Event(Event.MSG, hb_ack));
		return;                                    // don't pass up !
	    case FdHeader.HEARTBEAT_ACK:                   // heartbeat ack
		Object suspect=hdr.suspected_mbr;
		if(ping_dest != null && ping_dest.equals(suspect)) {
		    ack_received=true;
		    num_tries=0;
		}
		else {
		    Stop();
		    ping_dest=GetPingDest(members);
		    if(ping_dest != null)
			Start();
		}
		return;
	    case FdHeader.SUSPECT:
		if(hdr.suspected_mbr != null) {
		    System.out.println("FD: SUSPECT(" + hdr.suspected_mbr + ")");
		    PassUp(new Event(Event.SUSPECT, hdr.suspected_mbr));
		}
		return;
	    default:  // all other messages
		if(ping_dest != null && (sender=msg.GetSrc()) != null) {
		    if(ping_dest.equals(sender)) {
			ack_received=true;
			num_tries=0;
			skip_heartbeat=true;
			//System.out.println("Reset ack_received as message was received from " + sender);
		    }
		}
		break;
	    }	    
	}
	PassUp(evt);                                        // pass up to the layer above us
    }



    public void Down(Event evt) {
	Message msg;
	
	switch(evt.GetType()) {
	case Event.STOP:
	    Stop();
	    PassDown(evt);
	    break;
	    
	case Event.VIEW_CHANGE:
	    synchronized(this) {
		Stop();
		View v=(View)evt.GetArg();
		members=v != null ? v.GetMembers() : null;		
		PassDown(evt);
		ping_dest=GetPingDest(members);
		if(ping_dest != null)
		    Start();
	    }
	    break;

	case Event.MSG:
	    msg=(Message)evt.GetArg();
	    // Do something with the event, e.g. add a header to the message
	    msg.AddHeader(new FdHeader(FdHeader.REGULAR));  // regular message
	    PassDown(evt);
	    break;

	default:
	    PassDown(evt);
	    break;
	}
    }




    public void run() {
	Message   suspect_msg, hb_req;
	FdHeader  hdr;

	while(ping_dest != null) {
	    ack_received=false;
	    if(!skip_heartbeat) {
		hb_req=new Message(ping_dest, null, null);
		hb_req.AddHeader(new FdHeader(FdHeader.HEARTBEAT));  // send heartbeat request
		if(trace) System.out.println("FD: sending are-you-alive msg to " + ping_dest);
		PassDown(new Event(Event.MSG, hb_req));
	    }
	    skip_heartbeat=false;
	    Util.Sleep(timeout);

	    if(!ack_received && num_tries >= max_tries) {
		if(trace)    
		    System.out.println("FD(" + local_addr + "): received no heartbeat ack from " + 
				       ping_dest + ", suspecting it");
		hdr=new FdHeader(FdHeader.SUSPECT);
		hdr.suspected_mbr=ping_dest;
		suspect_msg=new Message(null, null, null);       // mcast SUSPECT to all members
		suspect_msg.AddHeader(hdr);
		PassDown(new Event(Event.MSG, suspect_msg));
		members.removeElement(ping_dest);                // try the next neighbor
		ping_dest=GetPingDest(members);

	    }
	    else {
		if(trace && !skip_heartbeat)
		    System.out.println("FD: received heartbeat ack from " + ping_dest);
		num_tries++;		
	    }
	}
    }





    void Start() {
	if(pinger == null) {
	    pinger=new Thread(this, "FD.PingerThread");
	    pinger.start();
	}
    }

    void Stop() {
	if(pinger != null) {
	    pinger.stop();
	    pinger=null;
	}
    }


}
