package JavaGroups.JavaStack.Protocols;

import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;



/**
   Client part of GMS. Whenever a new member wants to join a group, it starts in the CLIENT role.
   No multicasts to the group will be received and processed until the member has been joined and
   turned into a SERVER (either coordinator or participant, mostly just participant). This class
   only implements <code>Join</code> (called by clients who want to join a certain group, and
   <code>ViewChange</code> which is called by the coordinator that was contacted by this client, to
   tell the client what its initial membership is.
 */
public class ClientGmsImpl extends GmsImpl {
    static ClientGmsImpl     impl=null;
    Vector                   initial_mbrs=new Vector();
    Object                   view_installation_mutex=new Object();
    long                     expected_vid=-1;        // returned by JOIN, VIEW must match this one
    Object                   expected_creator=null;  // returned by JOIN, VIEW must match this one
    boolean                  joined=false;



    
    private ClientGmsImpl() {}

    public static ClientGmsImpl CreateInstance(GMS gms) {
	if(impl == null)
	    impl=new ClientGmsImpl();
	else {  // reset vars to initial default values
	    impl.initial_mbrs.removeAllElements();
	    impl.expected_vid=-1;
	    impl.expected_creator=null;
	    impl.joined=false;
	}
	impl.gms=gms;
	return impl;
    }





    /**
       Will generate a CONNECT_OK event. Determines the coordinator and sends a unicast 
       Join() message to it. If successful, we wait for a ViewChange (can time out). 
       If view change is received, impl is changed to an instance of ParticipantGmsImpl. 
       Otherwise, we continue trying to send Join() messages to	the coordinator, 
       until we succeed (or there is no member in the group. In this case, we create
       our own singleton group).
       @param mbr Our own address (assigned through SET_LOCAL_ADDRESS)
    */
    public void Join(Address mbr) {
	Object    coord=null;
	Event     view_evt;

	while(true) {
	    FindInitialMembers();

	    synchronized(view_installation_mutex) {
		if(joined) {
		    System.out.println("******** ClientGmsImpl.Join(): successful ! *********");
		    return;
		}
		
		if(initial_mbrs.size() < 1) {
		    System.out.println("No initial members found; starting as first member");
		    gms.view_id=new ViewId(mbr);       // create singleton view with mbr as only member
		    gms.members.Add(mbr);
		    view_evt=new Event(Event.VIEW_CHANGE, 
				       gms.MakeView(gms.members.GetMembers(), gms.view_id));
		    gms.PassDown(view_evt);
		    gms.PassUp(view_evt);
		    gms.SetImpl(CoordGmsImpl.CreateInstance(gms));           // become coordinator
		    gms.PassUp(new Event(Event.BECOME_SERVER));
		    gms.PassDown(new Event(Event.BECOME_SERVER));
		    break;
		}
		
		coord=FindCoord(initial_mbrs);
		if(coord == null) {
		    System.err.println("ClientGmsImpl.Join(): could not determine coordinator " +
				       "from responses " + initial_mbrs);
		    continue;
		}
		
		try {
		    System.out.println("--> Sending async. JOIN request to " + coord + ", timeout=" +
				       gms.join_timeout);
		    gms.CallRemoteMethod(coord, "HandleJoin", mbr, GroupRequest.GET_NONE, 0);
		    view_installation_mutex.wait(gms.join_timeout);  // wait for view -> HandleView()
		}
		catch(Exception e) {
		    System.err.println("ClientGmsImpl.Join():" + e);
		    continue;
		}

		if(joined) {
		    System.out.println("******** JOIN successful ! *********");
		    return;  // --> SUCCESS
		}
		else {
		    System.out.println("JOIN failed, retrying");
		    Util.Sleep(gms.join_retry_timeout);
		}
	    } // end synchronized
	}  // end while
    }





    public void Leave(Address mbr) {
	WrongMethod("Leave");
    }



    public void Suspect(Address mbr) {
	// WrongMethod("Suspect");
    }



    public void Merge(Vector new_mbrs) {
	WrongMethod("Merge");
    }



    public boolean HandleJoin(Address mbr) {
	WrongMethod("HandleJoin");
	return false;
    }



    /** Returns false. Clients don't handle Leave() requests */
    public void HandleLeave(Address mbr, boolean suspected) {
	WrongMethod("HandleLeave");
    }



    /**
       Install the first view in which we are a member. This is essentially a confirmation
       of our JOIN request (see Join() above).
     */
    public void HandleViewChange(ViewId new_view, Vector mems) {


	System.out.println("********** Client.HandleViewChange() *****");

	if(gms.local_addr != null && mems != null && mems.contains(gms.local_addr)) {
	    synchronized(view_installation_mutex) {  // wait until JOIN is sent (above)
		joined=true;
		view_installation_mutex.notify();
		gms.InstallView(new_view, mems);
		gms.SetImpl(ParticipantGmsImpl.CreateInstance(gms));  // become participant
		gms.PassUp(new Event(Event.BECOME_SERVER));
		gms.PassDown(new Event(Event.BECOME_SERVER));
	    }
	}
    }




    /** Returns immediately. Clients don't handle Merge() requests */
    public void HandleMerge(Vector new_mems, Vector suspects, long other_ltime) {
	WrongMethod("HandleMerge");
	return;
    }



    /** Returns immediately. Clients don't handle Suspect() requests */
    public void HandleSuspect(Address mbr) {
	WrongMethod("HandleSuspect");
	return;
    }



    public boolean HandleUpEvent(Event evt) {
	Vector tmp;

	switch(evt.GetType()) {

	case Event.FIND_INITIAL_MBRS_OK:
	    tmp=(Vector)evt.GetArg();
	    synchronized(initial_mbrs) {
		if(tmp != null && tmp.size() > 0)
		    for(int i=0; i < tmp.size(); i++)
			initial_mbrs.addElement(tmp.elementAt(i));
		initial_mbrs.notify();
	    }
	    return false;  // don't pass up the stack
	}

	return true;
    }





    /* --------------------------- Private Methods ------------------------------------ */


    




    /**
       Pings initial members. Removes self before returning vector of initial members. 
       Uses IP multicast or gossiping, depending on parameters.
     */
    void FindInitialMembers() {
	PingRsp ping_rsp;

	synchronized(initial_mbrs) {
	    initial_mbrs.removeAllElements();
	    gms.PassDown(new Event(Event.FIND_INITIAL_MBRS));
	    try {initial_mbrs.wait(gms.initial_mbrs_timeout);}
	    catch(Exception e) {}

	    for(int i=0; i < initial_mbrs.size(); i++) {
		ping_rsp=(PingRsp)initial_mbrs.elementAt(i);
		if(ping_rsp.own_addr != null && gms.local_addr != null && 
		   ping_rsp.own_addr.equals(gms.local_addr)) {
		    initial_mbrs.removeElementAt(i);
		    break;
		}
	    }
	}
    }



    Object FindCoord(Vector mbrs) {
	PingRsp mbr;
	if(mbrs == null || mbrs.size() < 1)
	    return null;

	for(int i=0; i < mbrs.size(); i++) {
	    mbr=(PingRsp)mbrs.elementAt(i);
	    if(mbr.coord_addr != null)
		return mbr.coord_addr;
	}
	return null;
    }

    
}
