package JavaGroups.JavaStack;

import java.io.*;
import java.net.*;
import java.util.*;
import JavaGroups.*;



/**
 * Server that keeps track of (group) members. A member registers its address (oid) with the 
 * Gossip server periodically. Every n seconds, if the registration is not renewed, old entries
 * are purged in order to avoid stale references to members that have crashed.<p>
 * The GossipServer is mainly used to simulate IP multicast. It allows to find the initial
 * members of a group where IP multicasting is disabled.
 */
public class GossipServer implements Runnable {


    class Entry {
	Object         addr=null;
	long           timestamp=0;

	public Entry(Object addr) {
	    this.addr=addr; timestamp=System.currentTimeMillis();
	}
    }





    private final int       PURGE_TIMEOUT=10000;       // After timeout, stale entries are removed 
    private Hashtable       entries=new Hashtable();   // groupname - list of Entry's
    private ServerSocket    srv_sock=null;


    
    private int PurgeOldEntries() {
	long   current_time;
	Vector val;
	Entry  entry;
	int    num=0;

	synchronized(entries) {
	    current_time=System.currentTimeMillis();
	    for(Enumeration e=entries.keys(); e.hasMoreElements();) {
		val=(Vector)entries.get(e.nextElement());
		for(int i=0; i < val.size(); i++) {
		    entry=(Entry)val.elementAt(i);
		    if(current_time - entry.timestamp >= PURGE_TIMEOUT) {
			Util.Print("Purging entry " + entry.addr + "(age=" +
				   (current_time-entry.timestamp) + " msecs)");

			val.removeElement(entry); // Is the iteration afterwards guaranteed ?
			num++;
		    }
		}
	    }
	    return num;
	}
    }



    private void AddEntry(Vector v, Object my_addr) {
	Entry entry, e=new Entry(my_addr);
	
	for(int i=0; i < v.size(); i++) {
	    entry=(Entry)v.elementAt(i);
	    if(entry.addr != null && entry.addr.equals(my_addr)) {
		entry.timestamp=System.currentTimeMillis();
		return;
	    }	    
	}
	v.addElement(e);
    }


    /**
     * Registers an address with Gossip. If the address is already present, it will be 
     * overwritten by new entry. This prevents the server from removing stale entries (older
     * than 4 seconds).
     */
    private synchronized void Register(String groupname, Object my_addr) {
	Vector tmp;

	if(groupname == null || my_addr == null)
	    return;
	    

	synchronized(entries) {
	    tmp=(Vector)entries.get(groupname);
	    if(tmp == null) {
		tmp=new Vector();
		AddEntry(tmp, my_addr);
		entries.put(groupname, tmp);
	    }
	    else
		AddEntry(tmp, my_addr);
	}
    }



    private synchronized Vector Get(String groupname) {
	Vector retval=new Vector(), tmp;
	synchronized(entries) {
	    tmp=(Vector)entries.get(groupname);
	    if(tmp == null)
		return retval;
	    for(int i=0; i < tmp.size(); i++)
		retval.addElement(((Entry)tmp.elementAt(i)).addr);
	    return retval;
	}
    }


    private void CloseSocket(OutputStream o, Socket sock) {
	try {
	    if(o != null)
		o.flush();
	    if(sock != null)
		sock.close();
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }


    private void ProcessRequest(Socket sock) {
	OutputStream        o=null;
	ObjectOutputStream  out;
	ObjectInputStream   inp;
	Request             req=null;
	Vector              retval;

	try {
	    o=sock.getOutputStream();
	    out=new ObjectOutputStream(o);
	    inp=new ObjectInputStream(sock.getInputStream());
	    req=(Request)inp.readObject();

	    switch(req.type) {
	    case Request.REGISTER:

		System.out.println("REGISTER(" + req.groupname + ", " + req.addr + ")");

		if(req.groupname == null || req.addr == null)
		    System.err.println("GossipServer.ProcessRequest(): GET request did not " +
				       "specify a group name or address to be registered was " +
				       "null !");
		else
		    Register(req.groupname, req.addr);
		break;
	    case Request.GET:
		if(req.groupname == null) {
		    System.err.println("GossipServer.ProcessRequest(): GET request did not " +
				       "specify a group name !");
		    retval=new Vector();
		}
		else
		    retval=Get(req.groupname);

		System.out.println("GET(" + req.groupname + ")\n" +
				   "Rsp: " + retval);

		out.writeObject(retval);
		break;
	    default:
		System.err.println("GossipServer.ProcessRequest(): type " + req.type +
				   " is unknown ! ");
		return;
	    }	    	    
	}
	catch(ClassCastException cast_ex) {
	    System.err.println("GossipServer.ProcessRequest(): expecting type Request, but " +
			       "received type " + req.getClass().getName() + ": " + cast_ex);	    
	}
	catch(SocketException sock_ex) {
	    System.err.println("GossipServer.ProcessRequest()+++: " + sock_ex);
	}
	catch(Exception e) {
	    System.err.println("GossipServer.ProcessRequest(): " + e);
	}
	finally {
	    CloseSocket(o, sock);
	}
    }












    public GossipServer() {
	new Thread(this, "GossipServerThread").start();	
    }


    public void Start(int port) throws Exception {
	Socket sock;

	srv_sock=new ServerSocket(port, 50);

	while(true) {
	    sock=srv_sock.accept();
	    ProcessRequest(sock);
	}
    }


    /**
     * Thread periodically (every PURGE_TIMEOUT milliseconds) removes entries older than PURGE_TIMEOUT
     * milliseconds.
     */
    public void run() {
	int num;
	while(true) {
	    try {
		Thread.currentThread().sleep(PURGE_TIMEOUT);		
		num=PurgeOldEntries();
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	}
    }


    

    /** Start up as Gossip daemon */
    public static void main(String[] args) {
	String           arg, next_arg, reg_loc;
	int              port=12001;
	GossipServer     srv=null; 


	for(int i=0; i < args.length; i++) {
	    arg=args[i];
	    if(arg.equals("-help")) {
		System.out.println("GossipServer [-port <port> ]");
		return;
	    }
	    else if(arg.equals("-port")) {
		port=new Integer(args[++i]).intValue();
		continue;
	    }
	}
	try {
	    srv=new GossipServer();
	    System.out.println("Gossip server was created at " + new Date());
	    System.out.println("Listening on port " + port);
	    srv.Start(port);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }
    
}
