package JavaGroups.JavaStack;

import java.util.*;
import JavaGroups.Event;

/**
 * The task if this class is to setup and configure the protocol stack. A string describing
 * the desired setup, which is both the layering and the configuration of each layer, is
 * given to the configurator which creates and configures the protocol stack and returns
 * a reference to the top layer (Protocol).<p>
 * Future functionality will include the capability to dynamically modify the layering
 * of the protocol stack and the properties of each layer.
 */
public class Configurator {
    
    public class ProtocolConfiguration {
	private String        protocol_name=null;
	private String        properties_str=null;
	private Properties    properties=new Properties();
	private final String  protocol_prefix="JavaGroups.JavaStack.Protocols";



	private void SetContents(String config_str) {
	    int index=config_str.indexOf('(');  // e.g. "UDP(in_port=3333)"
	    int end_index=config_str.lastIndexOf(')');

	    if(index == -1) {
		protocol_name=config_str;
	    }
	    else {
		if(end_index == -1) {
		    System.err.println("Closing ')' not found in " + config_str);
		    System.err.println("Properties cannot be set !");
		}
		else {
		    properties_str=config_str.substring(index+1, end_index);
		    protocol_name=config_str.substring(0, index);
		}
	    }

	    /* "in_port=5555;out_port=6666" */
	    if(properties_str != null) {
		Vector  components=ParseComponentStrings(properties_str, ';');
		if(components.size() > 0) {
		    for(int i=0; i < components.size(); i++) {
			String  name, value, comp=(String)components.elementAt(i);
			index=comp.indexOf('=');
			if(index == -1) {
			    System.err.println("'=' not found in " + comp);
			    continue;
			}
			name=comp.substring(0, index);
			value=comp.substring(index+1, comp.length());
			properties.put(name, value);			
		    }
		}
	    }
	}



	public ProtocolConfiguration(String config_str) {
	    SetContents(config_str);
	}

	private String     GetProtocolName() {return protocol_name;}
	private Properties GetProperties()   {return properties;}


	private Protocol CreateLayer(ProtocolStack prot_stack) {
	    Protocol retval=null;
	    if(protocol_name == null)
		return null;
	    try {
		protocol_name=protocol_prefix + "." + protocol_name;
		retval=(Protocol)Class.forName(protocol_name).newInstance();
		if(retval == null)
		    System.err.println("Creation of protocol " + protocol_name  +"failed !");
		else
		    retval.SetProtocolStack(prot_stack);
	    }
	    catch(InstantiationException inst_ex) {
		System.err.println(inst_ex);
		System.err.println("An instance of " + protocol_name +
				   " could not be created. Please check that it implements" +
				   " interface Protocol and that is has a public empty" +
				   " constructor !");
	    }
	    catch(Exception e) {
		System.err.println(e);
		return null;
	    }
	    if(properties != null)
		if(!retval.SetProperties(properties))
		    return null;
	    return retval;
	}



	public String toString() {
	    StringBuffer retval=new StringBuffer();
	    retval.append("Protocol: ");
	    if(protocol_name == null)
		retval.append("<unknown>");
	    else
		retval.append(protocol_name);
	    if(properties != null)
		retval.append("(" + properties + ")");
	    return retval.toString();
	}
    }



    
    class ProtocolReq {
	Vector   up_reqs=null;
	Vector   down_reqs=null;
	Vector   up_provides=null;
	Vector   down_provides=null;
	String   name=null;

	ProtocolReq(String name) {this.name=name;}


	boolean ProvidesUpService(int evt_type) {
	    int type;

	    if(up_provides != null) {
		for(int i=0; i < up_provides.size(); i++) {
		    type=((Integer)up_provides.elementAt(i)).intValue();
		    if(type == evt_type)
			return true;
		}
	    }
	    return false;
	}

	boolean ProvidesDownService(int evt_type) {
	    int type;

	    if(down_provides != null) {
		for(int i=0; i < down_provides.size(); i++) {
		    type=((Integer)down_provides.elementAt(i)).intValue();
		    if(type == evt_type)
			return true;
		}
	    }
	    return false;
	}


	public String toString() {
	    StringBuffer ret=new StringBuffer();
	    ret.append("\n" + name + ":");
	    if(up_reqs != null)
		ret.append("\nRequires from above: " + PrintUpReqs());

	    if(down_reqs != null)
		ret.append("\nRequires from below: " + PrintDownReqs());

	    if(up_provides != null)
		ret.append("\nProvides to above: " + PrintUpProvides());

	    if(down_provides != null)
		ret.append("\nProvides to below: " + PrintDownProvides());
	    return ret.toString();
	}


	String PrintUpReqs() {
	    StringBuffer ret=new StringBuffer("[");
	    if(up_reqs != null) {
		for(int i=0; i < up_reqs.size(); i++) {
		    ret.append(Event.Type2String(((Integer)up_reqs.elementAt(i)).intValue()) + " ");
		}
	    }
	    return ret.toString() + "]";
	}
	
	String PrintDownReqs() {
	    StringBuffer ret=new StringBuffer("[");
	    if(down_reqs != null) {
		for(int i=0; i < down_reqs.size(); i++) {
		    ret.append(Event.Type2String(((Integer)down_reqs.elementAt(i)).intValue()) + " ");
		}
	    }
	    return ret.toString() + "]";
	}
	

	String PrintUpProvides() {
	    StringBuffer ret=new StringBuffer("[");
	    if(up_provides != null) {
		for(int i=0; i < up_provides.size(); i++) {
		    ret.append(Event.Type2String(((Integer)up_provides.elementAt(i)).intValue()) + " ");
		}
	    }
	    return ret.toString() + "]";
	}

	String PrintDownProvides() {
	    StringBuffer ret=new StringBuffer("[");
	    if(down_provides != null) {
		for(int i=0; i < down_provides.size(); i++)
		    ret.append(Event.Type2String(((Integer)down_provides.elementAt(i)).intValue()) + 
			       " ");
	    }
	    return ret.toString() + "]";
	}	    


    }




    /**
     * The configuration string has a number of entries, separated by a ':' (semi).
     * Each entry consists of the name of the protocol, followed by an optional configuration
     * of that protocol. The configuration is enclosed in parentheses, and contains entries
     * which are name/value pairs connected with an assignment sign (=) and separated by
     * a semicolon.
     * <pre>UDP(in_port=5555;out_port=4445):FRAG(frag_size=1024)</pre><p>
     * The <em>first</em> entry defines the <em>bottommost</em> layer, the string is parsed
     * left to right and the protocol stack constructed bottom up. Example: the string
     * "UDP(in_port=5555):FRAG(frag_size=32000):DEBUG" results is the following stack:<pre>
     *   
     *   -----------------------
     *  | DEBUG                 |
     *  |-----------------------|
     *  | FRAG frag_size=32000  |
     *  |-----------------------|
     *  | UDP in_port=32000     |
     *   -----------------------
     * </pre>
     */
    public Protocol SetupProtocolStack(String configuration, ProtocolStack st) throws Exception {
	Protocol                protocol_stack=null;
	Protocol                layer;
	Vector                  protocol_configs;
	ProtocolConfiguration   protocol_config;
	Vector                  protocols;

	protocol_configs=ParseConfigurations(configuration);
	protocols=CreateProtocols(protocol_configs, st);
	if(protocols == null)
	    return null;
	protocol_stack=ConnectProtocols(protocols);
	return protocol_stack;
    }

	
	
    /**
     * Get a string of the form "P1(config_str1):P2:P3(config_str3)" and return
     * ProtocolConfigurations for it. That means, parse "P1(config_str1)", "P2" and 
     * "P3(config_str3)"
     * @param config_str Configuration string
     * @return Vector of ProtocolConfigurations
     */
    private Vector ParseComponentStrings(String config_str, int delimiter) {
	Vector         retval=new Vector();
	StringBuffer   buf=new StringBuffer();
	int            index=0;
	int            ch;
	boolean        esc=false;

	for(int i=0; i < config_str.length(); i++) {
	    ch=config_str.charAt(i);

	    if(ch == '\\') { // esc character, copy the next char literally into string buffer
		i++;
		buf.append(config_str.charAt(i));
		continue;
	    }
	    if(ch == delimiter) {
		retval.addElement(buf.toString());		
		buf=new StringBuffer();
		continue;		
	    }
	    buf.append(config_str.charAt(i)); // catches rest
	}
	if(buf.length() > 0)
	    retval.addElement(buf.toString());
	return retval;
    }



    /**
     * Return a number of ProtocolConfigurations in a vector
     * @param configuration protocol-stack configuration string
     * @return Vector of ProtocolConfigurations
     */
    private Vector ParseConfigurations(String configuration) {
	Vector                 retval=new Vector();
	Vector                 component_strings=ParseComponentStrings(configuration, ':');
	String                 component_string;
	ProtocolConfiguration  protocol_config;

	if(component_strings == null)
	    return null;
	for(int i=0; i < component_strings.size(); i++) {
	    component_string=(String)component_strings.elementAt(i);
	    protocol_config=new ProtocolConfiguration(component_string);
	    retval.addElement(protocol_config);
	}
	return retval;
    }






    /**
     * Takes vector of ProtocolConfigurations, iterates through it, creates Protocol for
     * each ProtocolConfiguration and returns all Protocols in a vector.
     * @param configurations Vector of ProtocolConfigurations
     * @return Vector of Protocols
     */

    private Vector CreateProtocols(Vector protocol_configs, ProtocolStack stack) {
	Vector                 retval=new Vector();
	ProtocolConfiguration  protocol_config;
	Protocol               layer;

	for(int i=0; i < protocol_configs.size(); i++) {
	    protocol_config=(ProtocolConfiguration)protocol_configs.elementAt(i);
	    layer=protocol_config.CreateLayer(stack);
	    if(layer == null)
		return null;
	    retval.addElement(layer);
	}
	
	if(SanityCheck(retval) == false)
	    return null;
	return retval;
    }



    /**
       Return false if sanity check fails. Possible sanity check is uniqueness of all protocol
       names.
     */
    private boolean SanityCheck(Vector protocols) {
	Vector       names=new Vector();
	Protocol     prot;
	String       name;
	ProtocolReq  req, req2;
	Vector       req_list=new Vector();
	int          evt_type;
	boolean      retval=true;

	// Checks for unique names
	for(int i=0; i < protocols.size(); i++) {
	    prot=(Protocol)protocols.elementAt(i);
	    name=prot.GetName();
	    for(int j=0; j < names.size(); j++) {
		if(name.equals((String)names.elementAt(j))) {
		    System.err.println("Configurator.SanityCheck(): protocol name " + name +
				       " has been used more than once; protocol names have to be " +
				       "unique !");
		    return false;
		}
	    }
	    names.addElement(name);
	}


	// Checks whether all requirements of all layers are met	
	for(int i=0; i < protocols.size(); i++) {
	    prot=(Protocol)protocols.elementAt(i);
	    req=new ProtocolReq(prot.GetName());
	    req.up_reqs=prot.RequiredUpServices();
	    req.down_reqs=prot.RequiredDownServices();
	    req.up_provides=prot.ProvidedUpServices();
	    req.down_provides=prot.ProvidedDownServices();
	    req_list.addElement(req);
	    // System.out.println(req);
	}

	
	for(int i=0; i < req_list.size(); i++) {
	    req=(ProtocolReq)req_list.elementAt(i);
	    
	    // check whether layers above this one provide corresponding down services
	    // System.out.println("Checking up reqs for " + req.name);
	    if(req.up_reqs != null) {
		for(int j=0; j < req.up_reqs.size(); j++) {
		    evt_type=((Integer)req.up_reqs.elementAt(j)).intValue();

		    if(!ProvidesDownServices(i, req_list, evt_type)) {
			System.err.println("Configurator.SanityCheck(): event " +
					   Event.Type2String(evt_type) + " is required by " +
					   req.name + ", but not provided by any of the layers above");
			retval=false;
		    }
		}
	    }

	    // check whether layers below this one provide corresponding up services
	    // System.out.println("Checking down reqs for " + req.name + "\n");
	    if(req.down_reqs != null) {  // check whether layers above this one provide up_reqs
		for(int j=0; j < req.down_reqs.size(); j++) {
		    evt_type=((Integer)req.down_reqs.elementAt(j)).intValue();

		    if(!ProvidesUpServices(i, req_list, evt_type)) {
			System.err.println("Configurator.SanityCheck(): event " +
					   Event.Type2String(evt_type) + " is required by " +
					   req.name + ", but not provided by any of the layers below");
			retval=false;
		    }
		}
	    }
	    
	}

	return retval;
    }



    /** Check whether any of the protocol 'below' end_index provide evt_type */
    boolean ProvidesUpServices(int end_index, Vector req_list, int evt_type) {
	ProtocolReq req;

	for(int i=0; i < end_index; i++) {
	    req=(ProtocolReq)req_list.elementAt(i);
	    if(req.ProvidesUpService(evt_type))
		return true;
	}
	return false;
    }


    /** Checks whether any of the protocols 'above' start_index provide evt_type */
    boolean ProvidesDownServices(int start_index, Vector req_list, int evt_type) {
	ProtocolReq req;

	for(int i=start_index; i < req_list.size(); i++) {
	    req=(ProtocolReq)req_list.elementAt(i);
	    if(req.ProvidesDownService(evt_type))
		return true;
	}
	return false;
    }



    /**
     * Creates a protocol stack by iterating through the protocol list and connecting 
     * adjacent layers. The list starts with the topmost layer and has the bottommost
     * layer at the tail. When all layers are connected the algorithms traverses the list
     * once more to call StartInternal() and StartWork() on each layer.
     * @param protocol_list List of Protocol elements (from top to bottom)
     * @return Protocol stack
     */
    private Protocol ConnectProtocols(Vector protocol_list) {
	Protocol current_layer=null, next_layer=null;

	//System.out.println("Connecting " + protocol_list.size() + " protocols: ");
	for(int i=0; i < protocol_list.size(); i++) {
	    current_layer=(Protocol)protocol_list.elementAt(i);
	    if(i+1 >= protocol_list.size())
		break;
	    next_layer=(Protocol)protocol_list.elementAt(i+1);
	    current_layer.SetUpProtocol(next_layer);
	    next_layer.SetDownProtocol(current_layer);	    
	}
	return current_layer;
    }

    
    public void StartProtocolStack(Protocol bottom_prot) {
	while(bottom_prot != null) {
	    bottom_prot.StartDownHandler();
	    bottom_prot.StartUpHandler();
	    bottom_prot=bottom_prot.GetUpProtocol();
	}
    }




    public void StopProtocolStack(Protocol start_prot) {	
	while(start_prot != null) {
	    start_prot.StopInternal();
	    start_prot=start_prot.GetDownProtocol();
	}
    }






    public Protocol FindProtocol(Protocol prot_stack, String name) {
	String    s;
	Protocol  curr_prot=prot_stack;

	while(true) {
	    s=curr_prot.GetName();
	    if(s == null)
		continue;
	    if(s.equals(name))
		return curr_prot;
	    curr_prot=curr_prot.GetDownProtocol();
	    if(curr_prot == null)
		break;
	}
	return null;
    }


    Protocol GetBottommostProtocol(Protocol prot_stack) throws Exception {
	Protocol  tmp=null, curr_prot=prot_stack;

	while(true) {
	    if((tmp=curr_prot.GetDownProtocol()) == null)
		break;
	    curr_prot=tmp;
	}
	if(curr_prot == null)
	    throw new Exception("Configurator.GetBottommostProtocol(): bottommost protocol " +
				"is null !");
	return curr_prot;
    }




    public static void main(String args[]) {
	if(args.length != 1) {
	    System.err.println("Configurator <string>");
	    System.exit(0);
	}
	String         config_str=args[0];
	Configurator   conf=new Configurator();
	Vector         protocol_configs;
	Vector         protocols;
	Protocol       protocol_stack;


	protocol_configs=conf.ParseConfigurations(config_str);
	protocols=conf.CreateProtocols(protocol_configs, null);
	if(protocols == null)
	    return;
	protocol_stack=conf.ConnectProtocols(protocols);

	try {
	    Thread.currentThread().sleep(3000);
	    conf.StopProtocolStack(protocol_stack);
	    // conf.StopProtocolStackInternal(protocol_stack);
	}
	catch(Exception e) {
	    System.err.println(e);
	}

	System.err.println(protocols);
    }
    
   
}



