
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.io.*;


import JavaGroups.*;
import JavaGroups.Event;
import JavaGroups.List;



/**
   This sample program stores all messages received between views and dumps this information
   to file when exiting. Therefore, we can check whether messages received between subsequent
   views are really the same set (VSYNC properties).
 */



class UserMessage implements Serializable {
    Object sender=null;
    long   num=0;
    

    public UserMessage(Object sender, long num) {
	this.sender=sender;
	this.num=num;
    }

    public String toString() {
	return sender + " #" + num;
    }
}


class ViewGossip implements Serializable {
    ViewId     vid=null;
    Hashtable  senders=new Hashtable();  // sender - number of msgs received

    public ViewGossip(ViewId vid, Hashtable senders) {
	this.vid=vid;
	this.senders=senders;
    }
}



class SenderThread extends Thread {
    Channel      channel=null;
    UserMessage  user_msg;
    Message      msg;
    long         number=1;

    public SenderThread(Channel c, String n) {channel=c; setName(n);}

    public void run() {
	while(true) {
	    try {
		Util.SleepRandom(100);
		user_msg=new UserMessage(channel.GetLocalAddress(), number);
		msg=new Message(null, null, Util.ObjectToByteBuffer(user_msg));
		channel.Send(msg);
		number++;
	    }
	    catch(ChannelNotConnected not_conn) {
		System.out.println("SenderThread: the channel is not connected");
		break;
	    }
	    catch(ChannelClosed closed) {
		System.out.println("SenderThread: the channel has been closed");
		break;
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	}
    }
}




/**
   Check whether same set of messages is received by all members. Each member stores messages
   received under vid and sender. When a view change is received, all members gossip about all their
   vids/senders. For each sender's messages, a checksum is computed and boradcast. All checksums
   have to be the same. If this is not the case, the program exits, dumping all vid/sender/msg information
   to a file.
 */
public class CheckVs implements Runnable, WindowListener, ActionListener {
    private Graphics               graphics=null;
    private Frame                  mainFrame=null;
    private Panel                  panel=null, sub_panel=null;
    private ByteArrayOutputStream  out=new ByteArrayOutputStream();
    private DataOutputStream       outstream;
    private DataInputStream        instream;
    private Random                 random=new Random(System.currentTimeMillis());
    private Button                 start_button, stop_button;
    private final Font             default_font=new Font("Helvetica",Font.PLAIN,12);
    private String                 groupname="CheckVsGroup";
    private Channel                channel=null;
    private Thread                 sender=null;
    private Thread                 receiver=null;
    private int                    member_size=1;
    private int                    red=0, green=0, blue=0;
    private Color                  default_color=null;
    Hashtable                      vids=new Hashtable();  // vid - Hashtable < sender - msgs >
    ViewId                         vid=null;


 
    public CheckVs(ChannelFactory factory, String props) throws Exception {
	channel=factory.CreateChannel(props);
    }



	
    public void go() {
	try {
	    channel.Connect(groupname);

	    receiver=new Thread(this, "ReceiverThread");
	    receiver.setPriority(Thread.MAX_PRIORITY);
	    receiver.start();

	    mainFrame=new Frame();
	    panel=new Panel();
	    sub_panel=new Panel();
	    mainFrame.setSize(200,200);
	    mainFrame.add("Center", panel);
	    start_button=new Button("Start");
	    start_button.setFont(default_font);
	    start_button.addActionListener(this);
	    stop_button=new Button("Stop");
	    stop_button.setFont(default_font);
	    stop_button.addActionListener(this);
	    sub_panel.add("South", start_button);
	    sub_panel.add("South", stop_button);
	    mainFrame.add("South", sub_panel);
	    mainFrame.addWindowListener(this);
	    mainFrame.setVisible(true);
	    mainFrame.setTitle(member_size + " mbrs");
	    graphics=panel.getGraphics();
	    graphics.setColor(default_color);
	    mainFrame.setBackground(Color.white);
	    start_button.setForeground(Color.blue);
	    stop_button.setForeground(Color.blue);
	}
	catch(Exception e) {
	    System.err.println(e);
	    return;
	}
    }



    void DumpToFile(String filename) {
	FileOutputStream fstream;
	PrintWriter      writer;
	Hashtable        h;
	Object           sender;
	List             l;
	ViewId           key;
	UserMessage      user_msg;
	long             checksum=0, total_checksum=0;

	System.out.println("***** Dumping data to file " + filename);

	try {
	    fstream=new FileOutputStream(filename);
	    writer=new PrintWriter(fstream);
	    
	    for(Enumeration e=vids.keys(); e.hasMoreElements();) {
		key=(ViewId)e.nextElement();
		writer.println("\n\n ---------- " + key + " -----------\n");

		total_checksum=0;
		h=(Hashtable)vids.get(key);
		for(Enumeration e2=h.keys(); e2.hasMoreElements();) {
		    sender=e2.nextElement();
		    l=(List)h.get(sender);
		    writer.println("\nSender: " + sender);
		    writer.println("Msgs:");
		    checksum=0;
		    for(Enumeration e3=l.Elements(); e3.hasMoreElements();) {
			user_msg=(UserMessage)e3.nextElement();
			writer.println(user_msg);
			checksum+=user_msg.num;
		    }
		    writer.println("checksum=" + checksum + "\n");
		    total_checksum+=checksum;
		}
		writer.println("total checksum=" + total_checksum + "\n\n");
	    }

	    writer.close();
	    fstream.close();
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }





    long GetTotalChecksum(ViewId v) {
	Hashtable        h;
	Object           sender;
	List             l;
	UserMessage      user_msg;
	long             checksum=0;
	
	h=(Hashtable)vids.get(v);
	if(h == null) {
	    System.err.println("Could not get checksum for " + v + " !");
	    return -1;
	}
	for(Enumeration e2=h.keys(); e2.hasMoreElements();) {
	    sender=e2.nextElement();
	    l=(List)h.get(sender);
	    for(Enumeration e3=l.Elements(); e3.hasMoreElements();) {
		user_msg=(UserMessage)e3.nextElement();
		checksum+=user_msg.num;
	    }
	}
	return checksum;
    }
    



    
    /**
       Receive a message, add it to hashtable (vid/sender is key). When it is a View, start view
       gossip. When it is a view gossip, compare. If not the same, exit.
    */
    public void run() {
	Object       tmp, obj;
	Message      msg=null;
	boolean      fl=true;
	ViewGossip   view_gossip;
	Hashtable    h;
	List         msgs;
	ViewId       vid_copy=null;

	while(fl) {
	    try {
		tmp=channel.Receive(0);		
		if(tmp == null) continue;


		if(tmp instanceof View) {
		    View v=(View)tmp;
		    member_size=v.Size();

		    System.out.println("CheckVS: received VIEW " + v);

		    if(mainFrame != null)
			mainFrame.setTitle(member_size + " mbrs");

		    if(vid != null) {  // send view gossip
			vid_copy=vid.Copy();
			h=(Hashtable)vids.get(vid_copy);
			if(h != null) {
			    System.out.println("** Sending view gossip");
			    view_gossip=new ViewGossip(vid_copy, (Hashtable)h.clone());
			    channel.Send(new Message(null, null, Util.ObjectToByteBuffer(view_gossip)));
			}
		    }
		    else
			System.err.println("** previous vid was null");

		    vid=v.GetVid();
		    vids.put(vid, new Hashtable());
		    continue;
		}



		if(!(tmp instanceof Message))
		    continue;



		msg=(Message)tmp;
		obj=Util.ObjectFromByteBuffer(msg.GetBuffer());



		if(obj instanceof UserMessage) {
		    if(vid == null) {
			System.err.println("Received UserMessage, but vid is null; discarding msg !");
			continue;
		    }
		    h=(Hashtable)vids.get(vid); // get all senders
		    if(h == null) {
			h=new Hashtable();
			vids.put(vid, h);
		    }
		    msgs=(List)h.get(msg.GetSrc());
		    if(msgs == null) {
			msgs=new List();
			h.put(msg.GetSrc(), msgs);
		    }
		    msgs.Add(obj);
		}


		else if(obj instanceof ViewGossip) {
		    view_gossip=(ViewGossip)obj;
		    long    my_checksum=GetTotalChecksum(view_gossip.vid);
		    long    his_checksum=ComputeChecksum(view_gossip.senders);
		    Object  local_addr=channel.GetLocalAddress();

		    if(local_addr != null && local_addr.equals(msg.GetSrc()))
			continue;

		    System.out.println("Received view gossip from " + msg.GetSrc());
		    
		    if(my_checksum == -1 || his_checksum == -1) {
			System.err.println("View gossip: invalid checksum ! (my_checksum=" + my_checksum +
					   ", his_checksum=" + his_checksum + ")");
			continue;
		    }
		    if(my_checksum != his_checksum) {
			System.err.println("View gossip: checksums for VID " + view_gossip.vid + 
					   " are not equal ! (my_checksum=" + my_checksum +
					   ", his_checksum=" + his_checksum + ")");

			local_addr=channel.GetLocalAddress();
			StopSending();
			channel.Close();
			mainFrame.setVisible(false);
			DumpToFile(local_addr + ".data");
			System.exit(0);  // exit the dirty way ...
		    }
		    else {
			System.out.println("------> checksums for VID " + view_gossip.vid + 
					   " are equal (" + my_checksum + " == " + his_checksum + ")");
		    }
		}



		else {
		    // error
		}



	    }
	    catch(ChannelNotConnected not) {
		System.err.println("CheckVs: " + not);
		break;
	    }
	    catch(ChannelClosed closed) {
		System.err.println("CheckVs: " + closed);
		return;
	    }
	    catch(Exception e) {
		System.err.println(e);
		continue; // break;
	    }
	}
    }



    long ComputeChecksum(Hashtable h) {
	UserMessage  user_msg;
	List         l;
	long         checksum=0;

	for(Enumeration e=h.elements(); e.hasMoreElements();) {
	    l=(List)e.nextElement();
	    for(Enumeration e2=l.Elements(); e2.hasMoreElements();) {
		user_msg=(UserMessage)e2.nextElement();
		checksum+=user_msg.num;
	    }
	}
	return checksum;
    }



    public void StartSending() {
	if(sender == null) {
	    sender=new SenderThread(channel, "SenderThread");
	    sender.start();
	}
    }


    public void StopSending() {
	if(sender != null) {
	    sender.stop();
	    sender=null;
	}
    }



    public void windowActivated(WindowEvent e) {}
    public void windowClosed(WindowEvent e) {}
    public void windowClosing(WindowEvent e) {
	Object local_addr=channel.GetLocalAddress();
	StopSending();
	channel.Close();
	mainFrame.setVisible(false);
	DumpToFile(local_addr + ".data");
	System.exit(0);  // exit the dirty way ...
    }
    public void windowDeactivated(WindowEvent e) {}
    public void windowDeiconified(WindowEvent e) {}
    public void windowIconified(WindowEvent e) {}
    public void windowOpened(WindowEvent e) {}


    public void actionPerformed(ActionEvent e) {
	String     command=e.getActionCommand();	
	if(command.equals("Start"))
	    StartSending();
	else if(command.equals("Stop"))
	    StopSending();
	else
	    System.out.println("Unknown action");
    }







    public static void main(String[] args) {
	CheckVs             draw=null;
	ChannelFactory   factory=new JChannelFactory();
	String           arg, next_arg;
	String           logfile=null;
	String           props=null;
    

	try {
	    props="UDP:PING:FD(max_tries=5):STABLE:NAKACK:UNICAST:FRAG:FLUSH:GMS:"+
		"VIEW_ENFORCER:QUEUE";
	    
	    for(int i=0; i < args.length; i++) {
		arg=args[i];
		if(arg.equals("-type")) {
		    next_arg=args[++i];
		    if(next_arg.equals("ens")) {
			props=null;
			factory=new EnsChannelFactory();
			continue;
		    }
		    else if(next_arg.equals("ibus")) {
			factory=(ChannelFactory)Class.forName("JavaGroups.IbusChannelFactory").newInstance();
			continue;
		    }
		}
		else if(arg.equals("-help")) {
		    System.out.println("CheckVs [-type ens | ibus ]");
		    return;
		}
	    }
	}
	catch(Exception e) {
	    System.out.println("Creation of channel factory failed, reason is: " + e);
	    return;
	}
	try {
	    draw=new CheckVs(factory, props);
	    draw.go();
	}
	catch(Exception e) {
	    System.err.println(e);
	    System.exit(0);
	}
    }



}
