
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.io.*;
import JavaGroups.*;



/**
 * Application shows virtual synchrony. A thread in each process randomly chooses square
 * and computes a random color (red/green/blue tuple) for it. It then multicasts the
 * square's location (x,y) and new color to all members of the group. When the message is
 * received (method call SetColor), the corresponding square is set to the new color and
 * repainted.<p>
 * Note that a process must not update its local grid before or after multicasting the update as
 * this could lead to inconsistencies ! Therefore, the Grid demo requires loopback to be
 * enabled, i.e. multicasts to a groups always have to also be received by its sender !
 */

public class GridChannelMi extends Frame implements WindowListener, ActionListener, Runnable,
						      MessageListener {
    private Channel                channel;
    private Panel                  main_panel, button_panel, bottom_panel;
    private Button                 start_button, stop_button, check_button;
    private Label                  checksum_label=new Label("Checksum: ");
    private final Font             default_font=new Font("Helvetica",Font.PLAIN,12);
    private String                 groupname="GridChannelMiGroup";
    private boolean                started=false;
    private int                    NUM=4;
    private ColorComp[][]          grid;
    private Thread                 mythread;
    private Random                 random=new Random(System.currentTimeMillis());
    private int                    sleeptime=100;  // ms
    private OutputStream           log=null;
    private PullPushAdapter        ad;

    final int                      SET_COLOR_MSG=0;
    final int                      STOP_MSG=1;
    

 
    public GridChannelMi(ChannelFactory factory, String properties, int num, 
		       int sleeptime, String log) {
	NUM=num;
	this.sleeptime=sleeptime;
	grid=new ColorComp[NUM][NUM];    
	main_panel=new Panel();
	main_panel.setLayout(new GridLayout(NUM, NUM));
	button_panel=new Panel();
	bottom_panel=new Panel();
	bottom_panel.setLayout(new BorderLayout());
	start_button=new Button("Start");
	start_button.setFont(default_font);
	start_button.addActionListener(this);

	Profiler.SetFilename("GridChannelMi.out");

	try {
	    if(log != null)
		this.log=new FileOutputStream(log);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
	
	stop_button=new Button("Stop");
	stop_button.setFont(default_font);
	stop_button.addActionListener(this);
	
	check_button=new Button("Checksum");
	check_button.setFont(default_font);
	check_button.addActionListener(this);

	button_panel.add("South", start_button);
	button_panel.add("South", stop_button);
	button_panel.add("South", check_button);
	
	bottom_panel.add("Center", checksum_label);
	bottom_panel.add("South", button_panel);
	checksum_label.setFont(default_font);

	setLayout(new BorderLayout());
	add("Center", main_panel);
	add("South",  bottom_panel);
	SetStarted(false);
		
	try {
	    channel=factory.CreateChannel(groupname, properties);
	    channel.Connect(3000);
	    ad=new PullPushAdapter(channel, this);

	    for(int x=0; x < NUM; x++)
		for(int y=0; y < NUM; y++) {
		    grid[x][y]=new ColorComp(Color.blue);
		    main_panel.add(grid[x][y]);
		}
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }

    
    private void Log(String msg) {
	if(log != null) {
	    try {
		log.write(msg.getBytes());
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	}
    }




    private Color SelectColor() {
	int red=(Math.abs(random.nextInt()) % 255);
	int green=(Math.abs(random.nextInt()) % 255);
	int blue=(Math.abs(random.nextInt()) % 255);
	return new Color(red, green, blue);
    }



    public void run() {
	Color         new_color;
	int           x, y;
	MethodCall    method_call;

	while(started) {
	    x=Math.abs(random.nextInt() % NUM);
	    y=Math.abs(random.nextInt() % NUM);
	    new_color=SelectColor();

	    try {
		method_call=new MethodCall("SetColor", new Integer(x), new Integer(y), new_color);
		

		Profiler.Start("Cast");
		channel.Cast(Util.ObjectToByteBuffer(method_call));
		Profiler.Stop("Cast");
		
		Thread.currentThread().sleep(sleeptime);
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	}
    }



    public void Receive(Message msg) {
	try {
	    MethodCall method_call=(MethodCall)Util.ObjectFromByteBuffer(msg.GetBuffer());	
	    method_call.Invoke(this, null);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }


	
    public void go() {
	try {
	    setSize(300,350);
 	    addWindowListener(this);
	    setVisible(true);
	    setTitle("Color Grid");
	    setBackground(Color.white);
	}
	catch(Exception e) {
	    System.err.println(e);
	    return;
	}
    }




    public static void main(String[] args) {
	ChannelFactory  factory=new JChannelFactory();
	GridChannelMi            grid=null;
	String          properties="UDP(mcast=true;loopback=true)", arg, next_arg;
	int             num=4, sleeptime=100;
	String          logfile=null;

	try {
	    for(int i=0; i < args.length; i++) {
		arg=args[i];
		if(arg.equals("-type")) {
		    next_arg=args[++i];
		    if(next_arg.equals("ens")) {
			factory=new EnsChannelFactory();
			properties=null;
			continue;
		    }		
		}
		else if(arg.equals("-help")) {
		    System.out.println("GridChannelMi [-type ens] [-num <number>] "+
				       "[-sleep <time in ms>] [-log <filename>]");
		    return;
		}
		else if(arg.equals("-num")) {
		    num=new Integer(args[++i]).intValue();
		    continue;
		}		
		else if(arg.equals("-sleep")) {
		    sleeptime=new Integer(args[++i]).intValue();
		    continue;
		}
		else if(arg.equals("-log")) {
		    logfile=args[++i];
		    continue;
		}
	    }
	}
	catch(Exception e) {
	    System.out.println("GridChannelMi [-type ens] [-num <number>] [-sleep <time (in ms)>] " +
			       "[-log <filename>]");
	    return;
	}

	grid=new GridChannelMi(factory, properties, num, sleeptime, logfile);
	grid.go();
    }





    


    /* --------------- Callbacks --------------- */


    public void SetColor(int x, int y, Color new_color) {
	synchronized(grid) {
	    Log("SetColor  " + x + "," + y + " (" + new_color.getRed() + "|" + 
		new_color.getGreen() + "|" + new_color.getBlue() + ")\n");
	    grid[x][y].SetColor(new_color);
	}
    }


    public void Start() {
	if(started)
	    return;
	SetStarted(true);
	checksum_label.setText("Running");
	mythread=new Thread(this);	
	mythread.start();
    }


    void SetStarted(boolean flag) {
	started=flag;
	if(started) {
	    stop_button.setEnabled(true);
	    start_button.setEnabled(false);
	}
	else {
	    stop_button.setEnabled(false);
	    start_button.setEnabled(true);
	}
    }


    public void Stop() {
	MethodCall method_call=new MethodCall("Stop");
	byte[]      m;
	
	if(started) {
	    SetStarted(false);
	    if(mythread != null) {
		try {
		    mythread.join();
		}
		catch(Exception e) {
		    System.err.println(e);
		}
		mythread.stop();
		mythread=null;
	    }	    
	    try {
		channel.Cast(Util.ObjectToByteBuffer(method_call));
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	}
	/* Give other threads a chance to finish. If we compute the checksum and still
	   receive updates from other threads that haven't finished processing, our checksum
	   will not reflect the current grid ! A better implementation would use acks to
	   make sure all threads have ended ... */
	try {
	    Thread.currentThread().sleep(500);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
	ComputeChecksum();
    }



    public void ComputeChecksum() {
	long   checksum=0;
	Color  col;
	synchronized(grid) {
	    for(int x=0; x < NUM; x++)
		for(int y=0; y < NUM; y++)
		    checksum+=grid[x][y].GetChecksum();
	}
	checksum_label.setText("Checksum: " + checksum);
    }



    public void windowActivated(WindowEvent e) {}
    public void windowClosed(WindowEvent e) {}
    public void windowClosing(WindowEvent e) {
	setVisible(false);
	channel.Disconnect();

	Profiler.Dump();

	System.exit(0);
    }
    public void windowDeactivated(WindowEvent e) {}
    public void windowDeiconified(WindowEvent e) {}
    public void windowIconified(WindowEvent e) {}
    public void windowOpened(WindowEvent e) {}


    public void actionPerformed(ActionEvent e) {
	String     command=e.getActionCommand();
	if(command == "Start")
	    Start();
	else if(command == "Stop")
	    Stop();
	else if(command == "Checksum")
	    ComputeChecksum();
	else
	    System.out.println("Unknown action");
    }


}

