
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.io.*;
import JavaGroups.Ensemble.*;




/**
 * Application shows virtual synchrony. A thread in each process randomly chooses square
 * and computes a random color (red/green/blue tuple) for it. It then multicasts the
 * square's location (x,y) and new color to all members of the group. When the message is
 * received (method call SetColor), the corresponding square is set to the new color and
 * repainted.<p>
 * Note that a process must not update its local grid before or after multicasting the update as
 * this could lead to inconsistencies ! Therefore, the Grid demo requires loopback to be
 * enabled, i.e. multicasts to a groups always have to also be received by its sender !
 */

public class EnsGrid extends Frame implements WindowListener, ActionListener, 
					   Hot_Callbacks, Runnable {

    Hot_Ensemble                   ens;
    Thread                         ens_thread;
    Hot_JoinOps                    jops;
    Hot_GroupContext               gctx;

    final int                      SET_COLOR_MSG=0;
    final int                      STOP_MSG=1;

    private Panel                  main_panel, button_panel, bottom_panel;
    private Button                 start_button, stop_button, check_button;
    private Label                  checksum_label=new Label("Checksum: ");
    private final Font             default_font=new Font("Helvetica",Font.PLAIN,12);
    private String                 groupname="EnsGridGroup";
    private boolean                started=false;
    private int                    NUM=4;
    private ColorComp[][]          grid;
    private Thread                 mythread;
    private Random                 random=new Random(System.currentTimeMillis());
    private int                    sleeptime=100;  // ms
    private Vector                 my_view;
    private OutputStream           log=null;


    
    public EnsGrid(int num, int sleeptime, String log) {
	NUM=num;
	this.sleeptime=sleeptime;
	grid=new ColorComp[NUM][NUM];    
	main_panel=new Panel();
	main_panel.setLayout(new GridLayout(NUM, NUM));
	button_panel=new Panel();
	bottom_panel=new Panel();
	bottom_panel.setLayout(new BorderLayout());
	start_button=new Button("Start");
	start_button.setFont(default_font);
	start_button.addActionListener(this);

	try {
	    if(log != null)
		this.log=new FileOutputStream(log);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
	
	stop_button=new Button("Stop");
	stop_button.setFont(default_font);
	stop_button.addActionListener(this);
	
	check_button=new Button("Checksum");
	check_button.setFont(default_font);
	check_button.addActionListener(this);

	button_panel.add("South", start_button);
	button_panel.add("South", stop_button);
	button_panel.add("South", check_button);
	
	bottom_panel.add("Center", checksum_label);
	bottom_panel.add("South", button_panel);
	checksum_label.setFont(default_font);

	setLayout(new BorderLayout());
	add("Center", main_panel);
	add("South",  bottom_panel);
	SetStarted(false);
		
	try {
	    ens=new Hot_Ensemble();
	    ens.setDebug(false);
	    ens_thread=new Thread(ens);
	    ens_thread.start();

	    jops = new Hot_JoinOps();
	    jops.heartbeat_rate = 5000;
	    jops.transports = "UDP";

	    jops.group_name =groupname;
	    jops.params="suspect_max_idle=3:int;suspect_sweep=3.000:time";
	    jops.conf = this;
	    jops.use_properties = true;
	    jops.properties="Gmp:Sync:Heal:Frag:Suspect:Flow:Total";
	    Hot_GroupContext[] hgc = new Hot_GroupContext[1];
	    Hot_Error hotError = ens.Join(jops,hgc);
	    gctx = hgc[0];		
	    if (hotError != null) {
		System.out.println("Couldn't join group!!: " + hotError);
		Hot_Ensemble.Panic("HOSED!!!");
	    }
	    
	    for(int x=0; x < NUM; x++)
		for(int y=0; y < NUM; y++) {
		    grid[x][y]=new ColorComp(Color.blue);
		    main_panel.add(grid[x][y]);
		}
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }

    
    private void Log(String msg) {
	if(log != null) {
	    try {
		log.write(msg.getBytes());
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	}
    }


    public void ViewAccepted(Vector new_view) {
	System.out.println("my_view is: " + my_view);
	System.out.println("New view: " + new_view);
    }



    private Color SelectColor() {
	int red=(Math.abs(random.nextInt()) % 255);
	int green=(Math.abs(random.nextInt()) % 255);
	int blue=(Math.abs(random.nextInt()) % 255);
	return new Color(red, green, blue);
    }



    public void run() {
	Color               new_color;
	int                 x, y;
	Hot_ObjectMessage   msg;
	Hot_Error           rc;
	GridMessage         m;
	int[]               tmp=new int[1];

	while(started) {
	    x=Math.abs(random.nextInt() % NUM);
	    y=Math.abs(random.nextInt() % NUM);
	    new_color=SelectColor();

	    try {
		m=new GridMessage(SET_COLOR_MSG, x, y, new_color.getRed(), 
				  new_color.getGreen(), new_color.getBlue());
		msg=new Hot_ObjectMessage(m);

		rc=ens.Cast(gctx, msg, tmp);

		if (rc != null) {
		    System.out.println("Couldn't cast msg !: " + rc);
		    Hot_Ensemble.Panic("HOSED!!!");
		}

		Thread.currentThread().sleep(sleeptime);
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	}
    }



	
    public void go() {
	try {
	    setSize(300,350);
 	    addWindowListener(this);
	    setVisible(true);
	    setTitle("Color Grid");
	    setBackground(Color.white);
	}
	catch(Exception e) {
	    System.err.println(e);
	    return;
	}
    }




    public static void main(String[] args) {
	EnsGrid          grid=null;
	String           arg, next_arg;
	int              num=4, sleeptime=100;
	String           logfile=null;

	try {
	    for(int i=0; i < args.length; i++) {
		arg=args[i];
		if(arg.equals("-help")) {
		    System.out.println("EnsGrid [-num <number>] "+
				       "[-sleep <time in ms>] [-log <filename>]");
		    return;
		}
		else if(arg.equals("-num")) {
		    num=new Integer(args[++i]).intValue();
		    continue;
		}		
		else if(arg.equals("-sleep")) {
		    sleeptime=new Integer(args[++i]).intValue();
		    continue;
		}
		else if(arg.equals("-log")) {
		    logfile=args[++i];
		    continue;
		}
	    }
	}
	catch(Exception e) {
	    System.out.println("EnsGrid [-num <number>] [-sleep <time (in ms)>] " +
			       "[-log <filename>]");
	    return;
	}

	grid=new EnsGrid(num, sleeptime, logfile);
	grid.go();
    }





    /*--------------------- Hot Callbacks ---------------------*/


    void HandleMessage(GridMessage msg) {
	switch(msg.type) {
	case SET_COLOR_MSG:
	    SetColor(msg.x, msg.y, new Color(msg.red, msg.green, msg.blue));
	    break;
	case STOP_MSG:
	    Stop();
	    break;
	default:
	    System.out.println("HandleMessage: type " + msg.type + " not known !");
	    break;
	}
    }


    public void ReceiveCast(Hot_GroupContext gctx, Object env, 
			    Hot_Endpoint origin, Hot_Message msg) {
	Hot_ObjectMessage hom = new Hot_ObjectMessage(msg);
	GridMessage m = (GridMessage)hom.getObject();

	HandleMessage(m);
    }
    
    public void ReceiveSend(Hot_GroupContext gctx, Object env, 
			    Hot_Endpoint origin, Hot_Message msg) {
	System.out.println("Received SEND !");
    }

    public void AcceptedView(Hot_GroupContext gctx, Object env, 
			     Hot_ViewState viewState) {
	System.out.println("Accepted view " + viewState);
    }

    public void Heartbeat(Hot_GroupContext gctx, Object env, int rate) {
	
    }


    public void Block(Hot_GroupContext gctx, Object env) {
	
    }


    public void Exit(Hot_GroupContext gctx, Object env) {
	
    }

    /*------------------------------------------------------------*/






    


    /* --------------- Callbacks --------------- */


    public void SetColor(int x, int y, Color new_color) {
	synchronized(grid) {
	    Log("SetColor  " + x + "," + y + " (" + new_color.getRed() + "|" + 
		new_color.getGreen() + "|" + new_color.getBlue() + ")\n");
	    grid[x][y].SetColor(new_color);
	}
    }


    public void Start() {
	if(started)
	    return;
	SetStarted(true);
	checksum_label.setText("Running");
	mythread=new Thread(this);	
	mythread.start();
    }


    void SetStarted(boolean flag) {
	started=flag;
	if(started) {
	    stop_button.setEnabled(true);
	    start_button.setEnabled(false);
	}
	else {
	    stop_button.setEnabled(false);
	    start_button.setEnabled(true);
	}
    }


    public void Stop() {
	GridMessage        m;
	Hot_ObjectMessage  msg;
	Hot_Error          rc;
	int[]              tmp=new int[1];

	if(started) {
	    SetStarted(false);
	    if(mythread != null) {
		try {
		    mythread.join();
		}
		catch(Exception e) {
		    System.err.println(e);
		}
		mythread.stop();
		mythread=null;
	    }	    
	    try {
		// patcher.SendGetN(groupname, "Stop", null, 0, 0);
		m=new GridMessage(STOP_MSG);
		msg=new Hot_ObjectMessage(m);
		rc=ens.Cast(gctx, msg, tmp);
		if (rc != null) {
		    System.out.println("Couldn't cast msg !: " + rc);
		    Hot_Ensemble.Panic("HOSED!!!");
		}		
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	}
	/* Give other threads a chance to finish. If we compute the checksum and still
	   receive updates from other threads that haven't finished processing, our checksum
	   will not reflect the current grid ! A better implementation would use acks to
	   make sure all threads have ended ... */
	try {
	    Thread.currentThread().sleep(500);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
	ComputeChecksum();
    }



    public void ComputeChecksum() {
	long   checksum=0;
	Color  col;
	synchronized(grid) {
	    for(int x=0; x < NUM; x++)
		for(int y=0; y < NUM; y++)
		    checksum+=grid[x][y].GetChecksum();
	}
	checksum_label.setText("Checksum: " + checksum);
    }



    public void windowActivated(WindowEvent e) {}
    public void windowClosed(WindowEvent e) {}
    public void windowClosing(WindowEvent e) {
	Hot_Error rc;

	setVisible(false);
	// 	dispatcher.Leave(groupname, this);
	rc=ens.Leave(gctx);
	if (rc != null) {
	    System.out.println("Couldn't leave group, rc= " + rc);
	    Hot_Ensemble.Panic("HOSED!!!");
	} 
	System.exit(0);
    }
    public void windowDeactivated(WindowEvent e) {}
    public void windowDeiconified(WindowEvent e) {}
    public void windowIconified(WindowEvent e) {}
    public void windowOpened(WindowEvent e) {}


    public void actionPerformed(ActionEvent e) {
	String     command=e.getActionCommand();	
	if(command == "Start")
	    Start();
	else if(command == "Stop")
	    Stop();
	else if(command == "Checksum")
	    ComputeChecksum();
	else
	    System.out.println("Unknown action");
    }


}

