package JavaGroups;

import java.io.*;
import java.util.*;


/**
 * Allows to send method invocations to all (or a subset) of the members of a group. Uses
 * <code>SyncCall</code> to wait for the first or the first N responses. Converts response
 * back to an object or, in the case of an exception, throws it.<p>
 * The example code below shows how a client can invoke a remote method call on all members
 * of a group and print the first result returned:<p>
 * <pre>
 * import java.util.*;
 * import JavaGroups.channel.*;
 * 
 * public class RemoteMethodCallTest {<br>
 * 
 *    public static void main(String args[]) {
 *        Channel           channel;
 *        RemoteMethodCall  m;
 *        Object            ret;
 *
 *        try {
 *            channel=new EnsChannel("MethodInvokerTest", null);
 *            channel.Connect(5000);
 *            <b>m=new RemoteMethodCall(channel, "Divide", new Integer(3), new Integer(4));</b>
 *            <b>ret=m.SendGetFirst(5000);</b>  // wait for maximum 5 seconds
 *            System.out.println( ret != null ? ret : "Response is null");
 *        }
 *        catch(Exception e) {
 *            System.err.println(e);
 *        }
 *        finally {
 *            channel.Disconnect();
 *            channel.Destroy();
 *        }
 *    }<br>
 * }
 * </pre><p>
 * The code first creates a channel and connects to it. Then (in bold letters) a 
 * <code>RemoteMethodCall</code> object is created which uses the channel to send messages 
 * and receive responses. (Actually, a <code>SyncCall</code> object is used to send a message
 * <em>synchronously</em> and wait for the first (or the first N) responses). Method
 * <code>SendGetFirst</code> sends the method call (as a <code>Message</code>) to all members 
 * of the group, converts the first reponse message to an object and returns it to the caller.
 * Method <code>SendGetN</code> does essentially the same, but returns N results (e.g. waits for
 * all groups members to respond). Note that N=0 does not wait for a response.<p>
 * In the example a method called <code>"Divide"</code> will be called with two arguments.<p>
 * The corresponding code for the server side (dispatching of message to method and returning
 * of result(s) can written using an instance of 
 * <a href=JavaGroups.channel.MethodInvoker.html>MethodInvoker</a>.<p>
 * If an <em>asynchronous</em> (oneway) remote method call is wanted, method <code>
 * SendGetN</code> can be used by setting <code>expected_responses</code> to 0. In this case,
 * no responses will be returned (even if the method itself does return a result).
 * @see MethodInvoker
 * @see Dispatcher
 */

public class RemoteMethodCall extends MethodCall {
    protected SyncCall       sync_call=null;
    protected Transportable  transport=null;


    protected void Init(Transportable t) {
	transport=t;
	sync_call=new SyncCall(transport);
    }

    public RemoteMethodCall(Transportable t) {
	super();
	Init(t);
    }
    
    public RemoteMethodCall(Transportable t, String name) {
	super(name);
	Init(t);
    }
    public RemoteMethodCall(Transportable t, String name, String funclet_name) {
	super(name, funclet_name);
	Init(t);
    }

    public RemoteMethodCall(Transportable t, String name, Object arg1) {
	super(name, arg1);
	Init(t);
    }
    public RemoteMethodCall(Transportable t, String name, String funclet_name, Object arg1) {
	super(name, funclet_name, arg1);
	Init(t);
    }

    public RemoteMethodCall(Transportable t, String name, Object arg1, Object arg2) {
	super(name, arg1, arg2);
	Init(t);
    }
    public RemoteMethodCall(Transportable t, String name, String funclet_name,
			    Object arg1, Object arg2) {
	super(name, funclet_name, arg1, arg2);
	Init(t);
    }

    public RemoteMethodCall(Transportable t, String name, Object arg1, Object arg2, Object arg3) {
	super(name, arg1, arg2, arg3);
	Init(t);
    }
    public RemoteMethodCall(Transportable t, String name, String funclet_name,
			    Object arg1, Object arg2, Object arg3) {
	super(name, funclet_name, arg1, arg2, arg3);
	Init(t);
    }



    protected void Check() throws Exception {
	if(method_name == null || method_name.length() < 1)
	    throw new Exception("RemoteMethodCall: method name is not valid");
    }


    public Object Send(Object dest, boolean oneway, long timeout) throws Exception {
	Message tmp;
	Object  retval=null;
	byte[]  buf;

	Check();

	buf=Util.ObjectToByteBuffer(this);
	tmp=(Message)sync_call.Send(dest, buf, oneway, timeout);
	if(tmp != null && (buf=tmp.GetBuffer()) != null) {
	    try {
		retval=(Object)Util.ObjectFromByteBuffer(buf);
		if(retval instanceof Exception)
		    throw (Exception)retval;
		return retval;
	    }
	    catch(ClassCastException cast_ex) {
		System.err.println(cast_ex);
		return null;
	    }
	}
	return null;	
    }

    
    public Object SendGetFirst(long timeout) throws Exception {
	Object   retval=null;
	byte     buf[]=null;
	Message  m;

	Check();

	buf=Util.ObjectToByteBuffer(this);

	m=sync_call.SendGetFirst(buf, timeout);

	// Convert Message.buf to Object (or Exception)

	if(m == null || (m.GetBuffer() == null))
	    return null;

	retval=Util.ObjectFromByteBuffer(m.GetBuffer());
	if(retval instanceof Exception)
	    throw (Exception)retval;
	return retval;
    }


    /**
     * Invokes a method on all members of the group and returns more than one response. (Note
     * that N should not be greater than the size of the group, or at least the timeout should be
     * sety to greater than 0, otherwise the call waits forever). All responses are converted to
     * objects and returned in a vector. If any method call threw an exception, the vector will
     * contain it instead of a normal object. It is the caller's duty to check whether there are
     * any exceptions present in the vector.<p>
     * <b>SendGetN(0, 0) can be used to send a method call to all members without expecting 
     * any responses</b>
     */
    public Vector SendGetN(int expected_responses, long timeout) throws Exception {
	Vector  responses, retval=new Vector();
	byte    buf[], tmpbuf[];
	Object  obj=null;
	Message tmp;

	Check();

	buf=Util.ObjectToByteBuffer(this);
	responses=sync_call.SendGetN(buf, expected_responses, timeout);
	if(responses == null)
	    return null;
	
	for(int i=0; i < responses.size(); i++) {
	    tmp=(Message)responses.elementAt(i);
	    tmpbuf=tmp.GetBuffer();
	    if(tmpbuf == null || tmpbuf.length < 1)
		continue;
	    try {
		obj=Util.ObjectFromByteBuffer(tmpbuf);
	    }
	    catch(Exception e) {
		System.err.println(e);
		continue;
	    }
	    retval.addElement(obj);
	}	
	return retval;
    }


    
    private void writeObject(java.io.ObjectOutputStream out) throws IOException {}
    
    private void readObject(java.io.ObjectInputStream in)
	throws IOException, ClassNotFoundException {
    }

    

    public static void main(String args[]) {
	FileOutputStream    file;
	FileInputStream     infile;
	ObjectOutputStream  out;
	ObjectInputStream   in;
	RemoteMethodCall    m;
	RemoteMethodCall    new_call;

	try {
	    m=new RemoteMethodCall(null, "Foo", new Integer(22), new Integer(44));
	    file=new FileOutputStream("data.dat");
	    out=new ObjectOutputStream(file);
	    out.writeObject(m);
	    file.close();

	    infile=new FileInputStream("data.dat");
	    in=new ObjectInputStream(infile);
	    new_call=(RemoteMethodCall)in.readObject();
	    System.out.println("Read object: " + new_call);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
	

    }
    

    
}
