package JavaGroups.JavaStack.Protocols;

import java.io.*;
import java.net.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;


/**
 * Algorithm for this class:<p>
 * <li>Remove Message from send_queue
 * <li>Send out a UDP packet.<p>
 * Messages sent by this thread flow from the top to the bottom of the stack,
 * that is, towards the network interface.
 */

public class UdpSenderThread extends Thread {    
    private UDP              udp_prot;
    private DatagramPacket   packet;
    private DatagramSocket   sockets[]=null;
    private int              num_socks=1;
    private int              curr_sock=0;
    private boolean          is_linux=false;
    private Queue            send_queue=null;


    // Most OSs support more, this has to be made configurable in FRAG
    // Linux (2.0) supports 65535 - size of IP header
    private final int        UDP_MAX_PACKET_SIZE=8192;
    

    private void CheckForLinux() {
	String os=System.getProperty("os.name");
	if(os != null && os.equals("Linux")) {
	    // System.out.println("OS is Linux --> using the SENDTO() workaround " +
	    // "(see ./design/sendto.txt)");
	    is_linux=true;
	}
    }


    public UdpSenderThread(UDP udp, int num) {
	setName("UdpSenderThread");
	CheckForLinux();
	udp_prot=udp;
	if(num <= 0) {
	    System.err.println("UdpSenderThread: number of sockets is corrected to 1");
	    num_socks=1;
	}
	else
	    num_socks=num;

	try {
	    sockets=new DatagramSocket[num_socks];
	    for(int i=0; i < num_socks; i++)
		sockets[i]=new DatagramSocket();
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }


    private void SendPendingPackets() {
	int      len=send_queue.Size();
	Message  msg;

	while((len=send_queue.Size()) > 0) {
	    try {
		msg=(Message)send_queue.Remove(100); // blocks until a Message is available
		if(msg == null)
		    return;

		SendPacket(msg);
	    }
	    catch(TimeoutException tex) {
		break;
	    }
	    catch(QueueClosed closed_ex) {
		// Util.Print("UDP: " + send_queue + " messages to send");
		break;
	    }
	    catch(Exception e) {
		System.err.println(e);
		break;
	    }	    
	}
    }

    public void Stop() {
	if(send_queue != null) {
	    try {
		SendPendingPackets();
		send_queue.Close();
		join(2000);
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	}
	for(int i=0; i < num_socks; i++) {
	    if(sockets[i] != null) {
		sockets[i].close();
		sockets[i]=null;
	    }
	}
    }




    protected synchronized void SendPacket(Message msg) throws Exception  {
	InternetAddress         dest, src;
	boolean                 done=false;
	ByteArrayOutputStream   out_stream;
	ObjectOutputStream      out;
	byte                    buf[];

	dest=(InternetAddress)((Oid)msg.GetDest()).GetAddress();
	if(dest == null) {
	    System.err.println("UdpSenderThread.SendPacket(): destination address is null !");
	    return;
	}

	if(!(dest instanceof InternetAddress)) {
	    System.err.println("UdpSenderThread.SendPacket(): destination address is not of " +
			       "type InternetAddress !");
	    return;
	}

	SetSourceAddress(msg);

	out_stream=new ByteArrayOutputStream(256);
	out=new ObjectOutputStream(out_stream);	
	out.writeObject(msg);
	buf=out_stream.toByteArray();

	packet=new DatagramPacket(buf, buf.length, dest.GetIpAddress(), dest.GetPort());

	if(packet.getLength() > UDP_MAX_PACKET_SIZE)
	    System.err.println("UdpSenderThread.Send(): packet size exceeds maximum " +
			       "UDP packet size: " + packet.getLength());

	while(!done) {	
	    try {
		sockets[curr_sock].send(packet);
		if(is_linux) {
		    sockets[curr_sock].close();
		    sockets[curr_sock]=new DatagramSocket();
		}		
		curr_sock=(curr_sock+1)%num_socks;
		done=true;
	    }
	    catch(Exception e) {
		System.err.println("UdpSenderThread: " + e);
		curr_sock=(curr_sock+1)%num_socks;
	    }
	}
    }



    private void SetSourceAddress(Message msg) {
	Oid              src=(Oid)msg.GetSrc();
	InternetAddress  inet_addr=(InternetAddress)udp_prot.GetLocalAddress();

	if(src == null) {
	    msg.SetSrc(new Oid(inet_addr, udp_prot.GetProtocolStack().GetChannelName()));
	    return;
	}
	if(src.GetAddress() == null)
	    src.SetAddress(inet_addr);
    }


    /**
     * Remove messages from send_queue of UDP and send them to the address as specified in
     * ((Oid)(Message.GetDest())).GetAddress()  --> InternetAddress
     */
    public void run() {
	Message                msg;

	send_queue=udp_prot.GetSendQueue();

	while(true) {
	    try {
		msg=(Me,ssage)send_queue.Remove(); // blocks until a Message is available
		if(msg == null)
		    return;

		SendPacket(msg);
	    }
	    catch(QueueClosed closed_ex) {
		// Util.Print("UDP: " + send_queue + " messages to send");
		break;
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	}
    }
    
    

}
