package JavaGroups.JavaStack.Protocols;

import java.io.Serializable;
import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;


class MackHeader implements Serializable {
    public long      seq_no=0;        // sequence number (requests)
    public long      ack_no=0;        // ACK number      (responses)
    public boolean   handle=true;     // handle or pass up ?

    public MackHeader(long seq_no, long ack_no, boolean handle) {
	this.seq_no=seq_no;
	this.ack_no=ack_no;
	this.handle=handle;
    }

    public String toString() {
	StringBuffer ret=new StringBuffer();
	ret.append("[MACK: ");
	ret.append("seq_no=" + seq_no + ", ack_no=" + ack_no + ", handle=" + handle);
	ret.append("]");
	return ret.toString();
    }
}


class AckTable {
    Hashtable acks=new Hashtable();

    AckTable() {}

    AckTable(Vector initial_view) {
	Set(initial_view);
    }

    int Size() {
	synchronized(acks) {
	    return acks.size();
	}
    }

    void Set(Vector view) {
	synchronized(acks) {
	    acks.clear();
	    for(int i=0; i < view.size(); i++)
		acks.put(view.elementAt(i), new Integer(0));
	}
    }

    /** Remove all items from acks that are not present in new_view */
    void Update(Vector new_view) {
	Object key;

	synchronized(acks) {
	    for(Enumeration e=acks.keys(); e.hasMoreElements();) {
		key=e.nextElement();
		if(!new_view.contains(key)) {
		    acks.remove(key);
		}
	    }
	    acks.notify();
	}
    }

    void AddAck(Object sender) {
	synchronized(acks) {
	    acks.remove(sender);
	    acks.notify();
	}
    }

    void Increment(Object sender) {
	Integer val;

	synchronized(acks) {
	    val=(Integer)acks.get(sender);
	    if(val != null)
		acks.put(sender, new Integer(val.intValue() + 1));
	}
    }

    void Reset() {
	synchronized(acks) {
	    acks.clear();
	    acks.notify();
	}
    }


    /** Get all keys that have a count of smaller than or equal to num. Increment their count. */
    Vector GetSmallerThanOrEqual(int num) {
	Object    key;
	Integer   val;
	int       n;
	Vector    retval=new Vector();

	synchronized(acks) {
	    for(Enumeration e=acks.keys(); e.hasMoreElements();) {
		key=e.nextElement();
		val=(Integer)acks.get(key);
		n=val.intValue();
		if(n <= num) {
		    retval.addElement(key);
		    acks.put(key, new Integer(n+1));
		}
	    }
	}
	return retval;
    }

    /** Get all keys that have a count of greater than num. Remove them from acks */
    Vector GetGreaterThan(int num) {
	Object    key;
	Integer   val;
	int       n;
	Vector    retval=new Vector();
	
	synchronized(acks) {
	    for(Enumeration e=acks.keys(); e.hasMoreElements();) {
		key=e.nextElement();
		val=(Integer)acks.get(key);
		n=val.intValue();
		if(n > num) {
		    retval.addElement(key);
		    acks.remove(key);
		}
	    }
	}
	return retval;
    }



    boolean WaitUntilAllAcksReceived(long timeout) {
	boolean  retval=false;
	long     start_time=System.currentTimeMillis(), current_time, wait_time=timeout;
	    
	synchronized(acks) {
	
	    while(acks.size() > 0) {
		try {
		    System.out.println("Waiting for " + acks.size() + " acks: " + wait_time + " ms");
		    acks.wait(wait_time);  // will be notified when acks is modified (addition, deletion)
		}
		catch(InterruptedException e) {}
		current_time=System.currentTimeMillis();
		if(acks.size() > 0) {
		    wait_time=timeout - (current_time - start_time);
		    if(wait_time <=0)
			return false;
		}
		else
		    break;
	    }
	    return true;    
	}
    }



}



/**
   <b>M</b>ulticast <b>ACK</b>nowledgement protocol layer. For each message sent, we expect 
   ACKs from all group members. When an ACK from a member has not been received for a certain time,
   the message will be resent to that member. If, after n retries, no ACK has been received, 
   the member is suspected.
   If the view changes while waiting for ACKs, the list of members from which ACKs are expected is 
   changed correspondingly.<p>
   Currently implemented without sliding window; all ACKs for a request have to be received before
   a new request can be sent, which is not very efficient ! TODO: implement sliding window, this allows
   multiple send requests to be processed at once (depending on the window size).<p>
   Use this protocol when
   <ul>
   <li>size of the group is relatively small
   <li>not too many messages are sent
   </ul>
   The properties guaranteed by the protocol are
   <ul>
   <li>no message is lost
   <li>if a receiver does not respond, it will be excluded from the group
   <li>the messages from  a sender to receiver are received in FIFO order
   </ul>
 */

public class MACK extends Protocol {
    long      my_seq_no=1;
    Vector    members=null;
    AckTable  ack_table=new AckTable();
    boolean   trace=false;
    long      timeout=2000; // Default of 2 secs before message is resent
    int       retries=2;
    Message   current_msg=null;



    public String  GetName() {return "MACK";}


    public void SetProperties(Properties props) {
	String     str;

	this.props=props;
	str=props.getProperty("trace");
	if(str != null)
	    trace=new Boolean(str).booleanValue();

	str=props.getProperty("timeout");
	if(str != null)
	    timeout=new Long(str).longValue();

	str=props.getProperty("retries");
	if(str != null)
	    retries=new Integer(str).intValue();
    }



    public void Up(Event evt) {
	Message    msg;
	MackHeader hdr;
	boolean    contains;

	if(trace)
	    System.out.println("Up: evt is " + evt);

	if(evt.GetType() == Event.MSG) {
	    msg=(Message)evt.GetArg();
	    hdr=(MackHeader)msg.RemoveHeader();
	    if(!hdr.handle) {
		PassUp(evt);
		return;
	    }

	    if(hdr.seq_no > 0) {                                         // received request 

		//System.out.println("MACK.Up(): received request (from " + msg.GetSrc() + "):" + hdr);

		Message ack_msg=msg.MakeReply();
		ack_msg.SetSrc(stack.GetLocalAddress());
		ack_msg.AddHeader(new MackHeader(0, hdr.seq_no, true));	 // send ACK to sender of msg
		
		if(trace)
		    System.out.println("Sending ACK for " + hdr.seq_no);

		//	System.out.println("MACK.Up(): sending ACK (to " + ack_msg.GetDest() + "):" + 
		//	   new MackHeader(0, hdr.seq_no, true));
		PassDown(new Event(Event.MSG, ack_msg));

		PassUp(evt);
		return;
	    }
	    else {                                                        // received response (ack)
		//System.out.println("MACK.Up(): received ACK (from " + msg.GetSrc() + "): " + hdr);
		ack_table.AddAck(msg.GetSrc());
		return;
	    }

	}
	
	PassUp(evt); // Pass up to the layer above us
    }





    void ReceiveAcksFromAllMembers() {
	boolean  all_acks;
	Object   dest;

	System.out.println("ReceiveAllAcks:");

	while(true) {

	    // wait until all acks received or timeout. if timeout, retval is false, else true
	    all_acks=ack_table.WaitUntilAllAcksReceived(timeout);
	    if(all_acks)
		break;

	    System.out.println("Resending msg");

	    // send acks
	    Vector missing=ack_table.GetSmallerThanOrEqual(retries);
	    Vector suspects=ack_table.GetGreaterThan(retries);
	    
	    for(int i=0; i < missing.size(); i++) {
		dest=missing.elementAt(i);
		current_msg.SetDest(dest);
		PassDown(new Event(Event.MSG, current_msg));
	    }

	    for(int i=0; i < suspects.size(); i++) {		
		dest=suspects.elementAt(i);
		System.out.println("MACK: suspecting " + dest);
		members.removeElement(dest);
		PassUp(new Event(Event.SUSPECT, dest));		
	    }
	}
	ack_table.Reset();
	
	System.out.println("DONE");
    }



    
    public void Down(Event evt) {
	Message     msg;
	MackHeader  hdr;
	Vector      tmp;

	if(trace)
	    System.out.println("Down: evt is " + evt);

	if(evt.GetType() != Event.MSG) {
	    if(evt.GetType() == Event.VIEW_CHANGE) {

		
		System.out.println("MACK: view change is " + evt.GetArg());

		if((tmp=(Vector)evt.GetArg()) != null) {
		    members=tmp;
		    ack_table.Update(members);
		}
	    }
	    PassDown(evt);
	    return;
	}

	msg=(Message)evt.GetArg();
	if(msg.GetDest() != null) {                         // ucast message (handled by NAK below)
	    hdr=new MackHeader(0, 0, false);
	    msg.AddHeader(hdr);
	    PassDown(evt);
	}
	else {	                                            // mcast request
	    hdr=new MackHeader(my_seq_no++, 0, true);
	    msg.AddHeader(hdr);
	    ack_table.Set(members);
	    
	    System.out.println("MACK.Down(): sending request (to " + msg.GetDest() + "): " + hdr);
	    
	    PassDown(evt);

	    current_msg=msg;
	    
	    /* Now receive ACKs for all members */
	    ReceiveAcksFromAllMembers();
	}
    }


}
