package JavaGroups.JavaStack;

import java.util.*;
import JavaGroups.*;

/**
 * A ProtocolStack manages a number of protocols layered above each other. It creates all
 * protocol classes, initializes them and, when ready, starts all of them, beginning with the 
 * bottom most protocol. It also dispatches messages received from the stack to registered
 * objects (e.g. channel, GMP) and sends messages sent by those objects down the stack.<p>
 * The ProtocolStack makes use of the Configurator to setup and initialize stacks, and to
 * detroy them again when not needed anymore.<p>
 * <b>The bottommost instance has to implement interface <code>LowLevelCommunication</code></b>.
 * This is necessary to be able to retrieve one's own (transport specific) address, join multicast
 * addresses etc.
 */
public class ProtocolStack extends Protocol implements LowLevelCommunication {
    private Protocol                top_prot=null;
    private LowLevelCommunication   bottom_prot;
    private Configurator            conf=new Configurator();
    private String                  setup_string;
    private JChannel                channel=null;
    private Hashtable               objects=new Hashtable();
    private String                  channel_name;
    
    public ProtocolStack(JChannel channel, String setup_string) {
	this.setup_string=setup_string;
	this.channel=channel;
	this.channel_name=channel.GetName();
    }
    
    
    /**
     * Create protocol stack (using Configurator). Put MessageDispatcher on top of stack and
     * redirect all messages to it.
     */
    public void Start() throws Exception {
	if(top_prot == null) {
	    top_prot=conf.SetupProtocolStack(setup_string, this);
	    top_prot.SetUpProtocol(this);
	    bottom_prot=conf.GetBottommostProtocol(top_prot);
	    conf.StartProtocolStack(top_prot);
	}

	//top_prot.PassDown(new Event(Event.GET_LOCAL_ADDRESS));
    }

    public void Stop() {
	if(top_prot != null) {	    
	    conf.StopProtocolStack(top_prot);
	    //conf.StopProtocolStackInternal(top_prot);
	    top_prot=null;
	}
    }

    
    /**
     * Since a protocol stack is always associated with only 1 channel, it always know the
     * channel name.
     */
    public String GetChannelName() {return channel_name;}


    /*--------------------------- Registry functionality -----------------------------*/

    public void Register(String name, Object obj) {
	objects.put(name, obj);
    }

    public void UnRegister(String name) {
	objects.remove(name);
    }

    public Object Lookup(String name) {
	return objects.get(name);
    }
    /*-----------------------  End of Registry functionality --------------------------*/






    /*--------------------------- Protocol functionality ------------------------------*/
    public String GetName()  {return "ProtocolStack";}


    public void Up(Event evt) {
	if(channel != null)
	    channel.Up(evt);
	else
	    System.err.println("ProtocolStack.Up(): channel is null !");
    }


    public void Down(Event evt) {
	if(top_prot != null)
	    top_prot.ReceiveDownEvent(evt);
	else
	    System.err.println("ProtocolStack.Down() at " + ": no down protocol available !");
    }



    protected void ReceiveUpEvent(Event evt) {
	Up(evt);
    }
    
    /** Just override with null functionality: we don't need any threads to be started ! */
    public void StartWork() {}

    /** Just override with null functionality: we don't need any threads to be started ! */
    public void StopWork()  {}


    /*----------------------- End of Protocol functionality ---------------------------*/








    /*---------------------- LowLevelCommunication interface ------------------------- */

    public Address GetLocalAddress() {
	if(bottom_prot != null)
	    return bottom_prot.GetLocalAddress();
	return null;
    }

    public Address GetNewMulticastAddress() {
	if(bottom_prot != null)
	    return bottom_prot.GetNewMulticastAddress();
	return null;
    }

    public void JoinMulticastAddress(Address mcast_address) {
	if(bottom_prot != null)
	    bottom_prot.JoinMulticastAddress(mcast_address);
    }

    public void LeaveMulticastAddress(Address mcast_address) {
	if(bottom_prot != null)
	    bottom_prot.LeaveMulticastAddress(mcast_address);
    }

    /*----------------- End of LowLevelCommunication interface ----------------------- */
    
}
