package JavaGroups;

import java.util.*;
import JavaGroups.JavaStack.*;


/**
 * JChannel is a pure Java implementation of Channel.
 */
public class JChannel extends Channel {

    class QueueItem {
	public QueueItem(Message msg)      {this.msg=msg;}
	public QueueItem(Vector new_view)  {this.new_view=new_view;}

	Message           msg=null;       // non-null means message
	Vector            new_view=null;  // non-null means view

	public String toString() {
	    if(msg != null) 
		return new String("QueueItem[msg=" + msg + "]");
	    return new String("QueueItem[view=" + new_view + "]");
	}
    }



    private String               channel_name=null;
    private String               channel_properties="UDP:GMS";
    private Queue                mq=new Queue();
    private MembershipListener   membership_listener;
    private Object               my_addr=null;
    private Vector               membership=new Vector();
    private int                  num_views_accepted=0;
    private ProtocolStack        prot_stack=null;
    private boolean              connected=false;

    private Integer              leave_mutex=new Integer(0);
    private Integer              flush_mutex=new Integer(0);
    private Integer              view_mutex=new Integer(0);
    

    private void CheckConnection() throws NotConnected {
	if(!connected)
	    throw new NotConnected();
    }


    public JChannel(String name, String properties) {
	channel_name=name;
	if(properties != null)
	    channel_properties=properties;

	prot_stack=new ProtocolStack(this, channel_properties);
	try {
	    prot_stack.Start();
	    my_addr=prot_stack.GetLocalAddress();
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }


    public void finalize() {
	if(connected)
	    Disconnect();
	Destroy();
    }


    public void Connect(long timeout) throws Exception {
	long start_time;
	if(!connected) {
	    mq.Reset();

	    Event join_event=new Event(Event.JOIN);
	    Down(join_event);

	    /* Wait for timeout or 2 members (whichever happens first): */
	    start_time=System.currentTimeMillis();
	    while((timeout-=(System.currentTimeMillis() - start_time)) > 0 && membership.size() < 2) {
		synchronized(view_mutex) {
		    view_mutex.wait(timeout);  // will be notified when new view has been received
		}
	    }
	    connected=true;
	}
    }


    public void Disconnect() {
	if(connected) {
	    
	    /* Send down a LEAVE and a FLUSH event. The LEAVE event travels down to the GMS, where
	       a LEAVE_OK response is generated and sent up the stack. The FLUSH travels down to
	       the bottom-most layer which flushes outstanding messages and then sends a FLUSH_OK
	       up the stack.
	       Each layer has to send outstanding messages before passing the FLUSH event further down. 
	       After that, the layer is not supposed to send more messages.
	       JChannel blocks until a LEAVE_OK and then a FLUSH_OK event have been received, or 
	       until timeouts have elapsed.
	       The protocol stack architecture guarantees that LEAVE and FLUSH events, and LEAVE_OK and
	       FLUSH_OK responses are sent down and up the stack in the same order, i.e. they never
	       'pass' each other. */

	    Event leave_event=new Event(Event.LEAVE, my_addr), flush_event=new Event(Event.FLUSH);


	    synchronized(leave_mutex) {
		try {
		    Down(leave_event);  // LEAVE will always be handled by each layer before FLUSH
		    leave_mutex.wait(5000);  // wait for LEAVE_OK event
		}
		catch(Exception e) {
		    System.err.println("JChannel.Disconnect(): " + e);
		}
	    }
	    
	    synchronized(flush_mutex) {
		try {
		    Down(flush_event);  // FLUSH will always be handled by each layer after LEAVE
		    flush_mutex.wait(10000); // wait for FLUSH_OK event
		}
		catch(Exception e) {
		    System.err.println("JChannel.Disconnect(): " + e);
		}
	    }

	    try {
	        mq.Close();
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }

	    connected=false;
	}
    }



    public void Destroy() {
	if(prot_stack != null) {
	    prot_stack.Stop();
	    prot_stack=null;
	}
    }




    public void Cast(byte[] msg) throws Exception {
	CheckConnection();
	Down(new Event(Event.MSG, new Message(null, my_addr, msg)));
    }


    public void Send(Object dest_addr, byte[] msg) throws Exception  {
	CheckConnection();

	if(dest_addr == null) {
	    System.err.println("JChannel.Send(): destination is null");
	    return;
	}
	Down(new Event(Event.MSG, new Message(dest_addr, my_addr, msg)));
    }



    /** This is almost certainly WRONG ! It will not send the SAME message to
	the members of the list, but a DIFFERENT one every time Send() is called !! */

    /** Fix: send a multicast to the whole group. The message sent contains the vector
     *  of destination addresses. When a member receives a cast, it checks whether it
     *  is member of the destination vector. If not, it discards the message.<p>
     *  This solution is not efficient, but avoids running into message ordering
     *  problems when certain members receive a message, and others don't
     */

    public void Send(Vector dests, byte[] msg) throws NotConnected  {	
	CheckConnection();
	System.err.println("JChannel.Send(Vector): not yet implemented !");
	return;

	// Message m=new Message(dests, my_addr, msg);
	// Send(m);
    }



    public void Send(Message msg) throws Exception  {
	CheckConnection();
	Down(new Event(Event.MSG, msg));
    }

	
	
    public Message Receive(long timeout) throws NotConnected, TimeoutException {
	Message    retval=null;
	QueueItem  item;
	Vector     mem_copy=null;

	while(true) {
	    CheckConnection();
	    
	    if(mq == null)
		throw new NotConnected();
	    
	    try {
		item=(timeout <= 0)? (QueueItem)mq.Remove() : (QueueItem)mq.Remove(timeout);
		if(item == null)
		    return null; // correct ?

		if(item.msg != null)
		    return item.msg;

		else if(item.new_view != null) {
		    synchronized(membership) {
			membership.removeAllElements();
			if(item.new_view.size() > 0)
			    for(int i=0; i < item.new_view.size(); i++)
				membership.addElement(item.new_view.elementAt(i));
			mem_copy=(Vector)membership.clone();
		    }		    		    
		    if(membership_listener != null)
			membership_listener.ViewAccepted(mem_copy);
		}

		else {
		    System.err.println("JChannel.Receive(): item is neither message nor view !");
		    continue;
		}
	    }
	    catch(TimeoutException tex) {
		throw tex;
	    }
	    catch(QueueClosed queue_closed) {
		System.err.println("JChannel.Receive(): " + queue_closed);
		throw new NotConnected();
	    }
	    catch(Exception e) {
		System.err.println(e);
		return null;
	    }
	}
    }




    public Vector GetMembers() {
	return membership;
    }

    public int GetNumMembers() {
	return membership.size();
    }

    public void SetMembershipListener(MembershipListener listener) {
	membership_listener=listener;
    }



    public Object GetAddress() {return my_addr;}
    public String GetName()    {return channel_name;}





    private void HandleUpEvent(Event evt) {
	switch(evt.GetType()) {
	case Event.LEAVE_OK:
	    synchronized(leave_mutex) {
		leave_mutex.notify();
	    }
	    break;
	case Event.FLUSH_OK:
	    synchronized(flush_mutex) {
		flush_mutex.notify();
	    }
	    break;
	default:
	    break;
	}
    }




    /**
     * Called by the ProtocolStack when a message is received. It will be added to the message
     * queue from which subsequent <code>Receive</code>s will dequeue it.
     */
    public void Up(Event evt) {
	QueueItem  item=null;
	int        type=evt.GetType();

	if(type != Event.MSG && type != Event.VIEW_CHANGE) {  // received normal Event
	    HandleUpEvent(evt);
	    return;
	}

	if(mq == null) {
	    System.err.println("JChannel.Up(): message queue is null");
	    return;
	}

	switch(type) {
	case Event.MSG:
	    item=new QueueItem((Message)evt.GetArg());
	    break;
	case Event.VIEW_CHANGE:
	    item=new QueueItem((Vector)evt.GetArg());
	    if(connected == false) {
		synchronized(membership) {
		    membership.removeAllElements();
		    if(item.new_view.size() > 0)
			for(int i=0; i < item.new_view.size(); i++)
			    membership.addElement(item.new_view.elementAt(i));
		}
		
		synchronized(view_mutex) {
		    view_mutex.notify();
		}
	    }
	    break;

	default:
	    System.err.println("JChannel.Up(): event type " + type + " not known");
	    return;
	}
	
	try {
	    mq.Add(item);
	}
	catch(Exception e) {
	    //System.err.println("JChannel.Up(): " + e);
	}
    }



    public void Down(Event evt) {
	if(prot_stack != null)
	    prot_stack.Down(evt);
	else
	    System.err.println("JChannel.Down(): no protocol stack available !");
    }







}
