package JavaGroups;

import java.io.*;
import java.util.*;
import iBus.*;

/**
 * Implementation of a channel using iBus. <a href=http://www.softwired.ch/ibus/>iBus</a>
 * is a pure Java-based publish/subscribe bus. An entity subscribed to a certain address
 * receives all messages published by any publisher under the same address (or topic).
 */
public class IbusChannel extends Channel implements iBus.Receiver, Membership {


    class QueueItem {
	int               type=-1;
	Message           msg;
	View              view_info=null;

	public QueueItem(int type) {this.type=type;}
	public QueueItem(int type, Message msg) {
	    this.type=type; this.msg=msg;
	}
	public QueueItem(int type, View view) {
	    this.type=type; view_info=view;
	}
    }


    private iBusURL             url;
    private String              properties="Reliable"; // replace by Vsync when available !
    private iBus.Stack          stack=null;
    private Queue               mq=new Queue();
    private Vector              members=new Vector();
    private MembershipListener  membership_listener=null;
    private final int           CAST_MSG=0;
    private final int           VIEW_MSG=1;
    private boolean             connecting=false;

    
    private void CheckConnection() throws NotConnected {
	if(stack == null)
	    throw new NotConnected();
    }


    public IbusChannel(String channel_name, String props) {
	try {
	    url=new iBusURL("ibus://226.0.0.1/tmp/" + channel_name);
	    if(props != null) 
		properties=props;
	    stack=new iBus.Stack(properties);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }

    public void Connect(long timeout) throws Exception {
	stack.registerMonitor(url, this);
	stack.subscribe(url, this);
	stack.registerTalker(url);

	if(timeout > 0) {
	    synchronized(this) {
		connecting=true;
		wait(timeout); // will be notified by AcceptedView()
		connecting=false;
	    }
	}
    }

    public void Disconnect() {
	try {
	    stack.unregisterTalker(url);
	    stack.unsubscribe(url, this);
	    stack.unregisterMonitor(url, this);
	}
	catch(Exception e) {
	    System.err.println("IbusChannel.Disconnect(): " + e);
	}
    }

    public void Destroy() {
	if(stack != null) {
	    stack.readyToExit();	    
	    stack=null;
	}
    }


    public void Cast(byte[] msg) throws Exception {
	CheckConnection();
	Send(url, msg);
    }


    public void Send(Object dest_addr, byte[] msg) throws Exception {
	CheckConnection();

	iBusURL  dest_url=(iBusURL)dest_addr;
	Posting  p=new Posting();
	Message  m=new Message(dest_url, stack.getStackURL(), msg);

	p.setLength(1);
	p.setObject(0, m);
	stack.push(dest_url, p);
    }



    public void Send(Vector dests, byte[] msg) throws Exception {
	throw new Exception("IbusChannel.Send(Vector): not implemented !");
    }


    public void Send(Message msg) throws Exception {
	byte buf[];

	CheckConnection();

	if(msg.GetDest() == null)
	    msg.SetDest(url);
	msg.SetSrc(stack.getStackURL());

	Posting p=new Posting();
	p.setLength(1);
	p.setObject(0, msg);
	stack.push(url, p);
    }




 public Message Receive(long timeout) throws NotConnected, TimeoutException {
	Message    retval=null;
	QueueItem  item;
	View       new_view;

	while(true) {
	    CheckConnection();
	    
	    if(mq == null)
		throw new NotConnected();
	    
	    try {
		item=(timeout <= 0)? (QueueItem)mq.Remove() : (QueueItem)mq.Remove(timeout);
		if(item == null)
		    return null;
		
		if(item.type == CAST_MSG)
		    return item.msg;
		else if(item.type == VIEW_MSG) { 
		    synchronized(members) {
			members.removeAllElements();
			ChannelMember[] mems=item.view_info.getMembers();
			for(int i=0; i < mems.length; i++)
			    members.addElement(mems[i].getURL());
		    }
		    if(membership_listener != null)
			membership_listener.ViewAccepted(members);
		}
		else {
		    System.err.println("IbusChannel.Receive(): error, type is " + item.type);
		    continue;
		}
	    }
	    catch(TimeoutException tex) {
		throw tex;
	    }
	    catch(QueueClosed queue_closed) {
		System.err.println("IbusChannel.Receive(): " + queue_closed);
		throw new NotConnected();
	    }
	    catch(Exception e) {
		System.err.println(e);
		return null;
	    }
	}
    }



    public synchronized Vector GetMembers() {
	return members;
    }


    public synchronized int GetNumMembers() {
	return members.size();
    }

    public void SetMembershipListener(MembershipListener listener) {
	membership_listener=listener;
    }

    public Object GetAddress() {
	return stack != null ? stack.getStackURL() : null;
    }

    public String GetName() {return url.toString();}


    /*-------------------- iBus callbacks ---------------------------- */

    public Posting dispatchPull(iBusURL channel, Posting request) {
	System.err.println("IbusChannel.dispatchPull(): not implemented !");
	return null;
    }

    public void dispatchPush(iBusURL source, Posting p) {
	Message    msg=(Message)p.getObject(0);
	QueueItem  item;

	try {
	    if(msg.GetSrc() == null)
		msg.SetSrc(p.getSender());

	    if(!msg.GetDest().equals(stack.getStackURL()) && !msg.GetDest().equals(url))
		return;

	    mq.Add(new QueueItem(CAST_MSG, msg));
	}
	catch(Exception e) {
	    System.err.println("IbusChannel.dispatchPush(): " + e);
	}
    }
    
    public void error (iBusURL url, String details) {
	System.err.println ("IbusChannel.error(): " + details);
	System.exit (1); // ??
    }


    public Serializable getState(iBusURL channel) {return null;}

    public void setState(iBusURL channel, Serializable state) {}

    public synchronized void viewChange(View newView) {
	int len=newView.getNumMembers();
	if(connecting == true && len > 1) {
	    synchronized(this) {
		members.removeAllElements();
		ChannelMember[] mems=newView.getMembers();
		for(int i=0; i < len; i++)
		    members.addElement(mems[i].getURL());		
		notify();            // wakes up Connect()
	    }
	}
	try {
	    mq.Add(new QueueItem(VIEW_MSG, newView));
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }

}
