package JavaGroups;

import java.util.*;
import JavaGroups.Ensemble.*;

/**
 * EnsChannel is an implementation of <em>Channel</em> based on 
 * <a href=http://simon.cs.cornell.edu/Info/Projects/Ensemble/index.html>Ensemble</a>. It 
 * maps a process group to a channel. Requirements are the presence of an executable called
 * <em>outboard</em> (which is Ensemble) and <em>gossip</em> running.
 */
public class EnsChannel extends Channel implements Hot_Callbacks {


    class QueueItem {
	int               type=-1;
	Hot_GroupContext  gctx=null;
	Hot_Endpoint      sender=null;
	Hot_Message       msg=null;
	Hot_ViewState     view_state=null;

	public QueueItem(int type) {this.type=type;}
	public QueueItem(int type, Hot_GroupContext gctx, Hot_Message msg) {
	    this(type);
	    this.gctx=gctx;
	    this.msg=msg;
	}
    }



    private Hot_Ensemble        ensemble=null;
    private Thread              ens_thread=null;
    private Hot_GroupContext    channel_id=null;
    private String              channel_name=null;
    private Hot_JoinOps         options=new Hot_JoinOps();
    private Queue               mq=null;
    private MembershipListener  mbrship_listener=null;
    private Object              my_addr=null;
    private Vector              membership=new Vector();
    private final String        default_properties="Gmp:Sync:Heal:Frag:Suspect:Flow:Total";
    private boolean             connecting=false;

    private final int         CAST_MSG=0;
    private final int         SEND_MSG=1;
    private final int         VIEW_MSG=2;
    private final int         EXIT_MSG=3;
    

    private void CheckConnection() throws NotConnected {
	if(channel_id == null)
	    throw new NotConnected();
    }



    /**
       Creates a new EnsChannel, which spawns an outboard process and connects to it.
       @param name Channel name. All channels with the same name form a group.
       @param properties Ensemble properties (cf. Ensemble reference manual). 
                         A value of <code>null</code> uses the default properties.
     */
    public EnsChannel(String name, String properties) {
	channel_name=name;
	options.heartbeat_rate=5000;
	options.transports="UDP";
	options.group_name=channel_name;
	options.properties=properties == null? default_properties : properties;	
	options.params="suspect_max_idle=3:int;suspect_sweep=3.000:time";
	options.conf=this;
	options.use_properties=true;
	ensemble=new Hot_Ensemble();
	ens_thread=new Thread(ensemble, "EnsembleThread");
	ens_thread.start();
    }

    /**
       Creates a new EnsChannel. Instead of spawning a new outboard process, it connects to an
       already running outboard process (on the same machine) using <code>outboard_port</code>.
       This allows multiple EnsChannels to share a copy of outboard. If the port is 0, outboard
       <em>will</em> be spawned. Parameter <code>transport_props</code> defines the type of transport
       to be used (UDP, ATM, IP MCAST etc).
       @param name Channel name. All channels with the same name form a group.
       @param properties Ensemble properties (cf. Ensemble reference manual). 
                         A value of <code>null</code> uses the default properties.
       @param transport_props Transport parameters. <code>Null</code> means use default (UDP).
                              Example: <code>"UDP:DEERING"</code> uses IP multicast (gossip is not
			      needed in this case).
       @param outboard_port Port on which the local outboard process is listening. The outboard
                            process has to be started before. Value of 0 means spawn outboard
			    nevertheless.
    */
    public EnsChannel(String name, String properties, String transport_props, int outboard_port) {
	channel_name=name;
	options.heartbeat_rate=5000;
	options.transports="UDP";
	if(transport_props != null)
	    options.transports=transport_props;
	options.group_name=channel_name;
	options.properties=properties == null? default_properties : properties;	
	options.params="suspect_max_idle=3:int;suspect_sweep=3.000:time";
	options.conf=this;
	options.use_properties=true;
	ensemble=outboard_port == 0 ? new Hot_Ensemble() : new Hot_Ensemble(outboard_port);
	ens_thread=new Thread(ensemble, "EnsembleThread");
	ens_thread.start();	
    }
    

    public void finalize() {
	if(channel_id != null)
	    Disconnect();
	Destroy();
    }

    public void Destroy() {
	if(ensemble != null) {
	    if(ens_thread != null) {
		ens_thread.stop();
		ens_thread=null;
	    }
	    try {
		Thread.currentThread().sleep(500);
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	    ensemble.destroyOutboard();
	    ensemble=null;
	}
    }


    public void Connect(long timeout) throws Exception {
	Hot_Error         rc;
	Hot_GroupContext  tmp[]=new Hot_GroupContext[1];
	
	if(channel_id != null)
	    throw new Exception("Already connected");
	if(ensemble == null || ens_thread == null)
	    throw new Exception("Ensemble has not been started");

	mq=new Queue();
	rc=ensemble.Join(options, tmp);
	if(rc != null)
	    throw new Exception(rc.toString());
	channel_id=tmp[0];
	if(timeout > 0)
	    synchronized(this) {
		connecting=true;
		wait(timeout); // will be notified by AcceptedView()
		connecting=false;
	    }
    }


    public void Disconnect() {
	Hot_Error rc;
	if(channel_id == null) {
	    System.err.println("Cannot disconnect as channel id is null");
	    return;
	}
	rc=ensemble.Leave(channel_id);	
	if(rc != null)
	    System.err.println("EnsChannel.Disconnect(): " + rc);
	channel_id=null;
	try {
	    mq.Close();
	}
	catch(Exception e) {
	    System.err.println(e);
	}
	mq=null;
    }



    public void Cast(byte[] msg) throws NotConnected {
	Hot_ObjectMessage  m;
	Hot_Error          rc;
	int                tmp[]=new int[1];

	CheckConnection();

	m=new Hot_ObjectMessage(new Message(null, null, msg));

	rc=ensemble.Cast(channel_id, m, tmp);
	if(rc != null)
	    System.err.println("EnsChannel.Cast(): " + rc);
    }


    public void Send(Object dest_addr, byte[] msg) throws NotConnected  {
	Hot_Endpoint        dest;
	Hot_Error           rc;
	Hot_ObjectMessage   m;
	int                 tmp[]=new int[1];

	CheckConnection();

	if(dest_addr == null) {
	    System.err.println("EnsChannel.Send(): destination is null");
	    return;
	}
	dest=(Hot_Endpoint)dest_addr;
	if(dest == null) {
	    System.err.println("EnsChannel.Send(): destination object is not of type " +
			       "Hot_Endpoint, but of type " + dest_addr.getClass().getName());
	    return;
	}

	m=new Hot_ObjectMessage(new Message(dest, null, msg));

	rc=ensemble.Send(channel_id, dest, m, tmp);
	if(rc != null)
	    System.err.println("EnsChannel.Send(): " + rc);
    }



    /** This is almost certainly WRONG ! It will not send the SAME message to
	the members of the list, but a DIFFERENT one every time Send() is called !! */

    /** Fix: send a multicast to the whole group. The message sent contains the vector
     *  of destination addresses. When a member receives a cast, it checks whether it
     *  is member of the destination vector. If not, it discards the message.<p>
     *  This solution is not efficient, but avoids running into message ordering
     *  problems when certain members receive a message, and others don't
     */

    public void Send(Vector dests, byte[] msg) throws NotConnected  {
	CheckConnection();
	Message m=new Message(dests, null, msg);
	Send(m);
    }



    public void Send(Message msg) throws NotConnected  {
	Object             dest=msg.GetDest();
	Hot_ObjectMessage  m=new Hot_ObjectMessage(msg);
	Hot_Error          rc=null;
	int                tmp[]=new int[1];

	CheckConnection();
	
	if(dest == null || dest instanceof String || dest instanceof Vector)
	    rc=ensemble.Cast(channel_id, m, tmp);
	else if(dest instanceof Hot_Endpoint)
	    rc=ensemble.Send(channel_id, (Hot_Endpoint)dest, m, tmp);
	else {
	    System.err.println("EnsChannel.Send(): dest address is " + 
			       "wrong (" + dest + ")");
	    return;
	}

	if(rc != null)
	    System.err.println("EnsChannel.Send(): rc is " + rc);
    }

	
    
    public Message Receive(long timeout) throws NotConnected, TimeoutException {
	Message    retval;
	QueueItem  item;

	while(true) {
	    CheckConnection();
	    
	    if(mq == null)
		throw new NotConnected();
	    
	    try {
		item=(timeout <= 0)? (QueueItem)mq.Remove() : (QueueItem)mq.Remove(timeout);
		if(item == null)
		    return null;
		if((retval=HandleQueueItem(item)) != null)
		    return retval;
	    }
	    catch(TimeoutException tex) {
		throw tex;
	    }
	    catch(QueueClosed queue_closed) {
		System.err.println("EnsChannel.Receive(): " + queue_closed);
		throw new NotConnected();
	    }
	    catch(Exception e) {
		System.err.println(e);
		return null;
	    }
	}
    }



    Message HandleCast(Hot_GroupContext gctx, Hot_Endpoint origin, Hot_Message msg) {
	Hot_ObjectMessage  tmp=new Hot_ObjectMessage(msg);
	Message            m=(Message)tmp.getObject();
	Object             dest;
	if(m == null) {
	    System.out.println("EnsChannel.ReceiveCast(): received message that is " +
			       "not of type Message. Discarding.");
	    return null;
	}

	/* If destination of message is a vector, make sure that we are in the target set.
	   When we are not in the target set, discard the message.
	   This is used for sending messages to a subset of the group; a multicast is
	   sent to all members, and those that are not member of the target set, just discard
	   the message. Thus, we need not worry about ordering of message when delivering
	   messages to only certain members of the group */

	if(m.GetSrc() == null)
	    m.SetSrc(origin);

	dest=m.GetDest();
	if(dest instanceof Vector) {
	    boolean       is_member=false;
	    Vector        v=(Vector)dest;
	    Hot_Endpoint  endp;

	    if(my_addr == null) {
		System.err.println("EnsChannel.ReceiveCast(): my_addr is null");
		return null;
	    }
	    for(int i=0; i < v.size(); i++) {
		endp=(Hot_Endpoint)v.elementAt(i);
		if(my_addr.equals(endp)) {
		    is_member=true;
		    break;
		}
	    }
	    if(!is_member) {
		System.err.println("EnsChannel.ReceiveCast(): discarded message #" +
				   m.GetId() + " sent to subset of group because endpoint " +
				   my_addr + " is not member of subset");
		return null;
	    }
	}
	
	return m;
    }


    void HandleView(Hot_GroupContext gctx, Hot_Endpoint origin, Hot_ViewState viewState) {
	Hot_Endpoint[]    members=viewState.members, tmp;
	Vector            v=new Vector();

	if(members.length > 0) {
	    membership.removeAllElements();
	    for(int i=0; i < members.length; i++)
		membership.addElement(members[i]);
	}
	
	if(mbrship_listener != null && members.length > 0) {       
	    for(int i=0; i < members.length; i++)     // Make a copy of the membership
		v.addElement(members[i]);	    
	    mbrship_listener.ViewAccepted(v);
	}
    }



    Message HandleQueueItem(QueueItem item) {
	Message retval=null;

	switch(item.type) {
	case CAST_MSG:	    
	case SEND_MSG:
	    retval=HandleCast(item.gctx, item.sender, item.msg);
	    break;
	case VIEW_MSG:
	    HandleView(item.gctx, item.sender, item.view_state);
	    break;
	case EXIT_MSG:
	    System.out.println("Received EXIT call !");
	    System.exit(0); // gotta be kidding !!
	    break;
	default:
	    System.out.println("EnsChannel.HandleQueueItem(): type " + item.type + " not known");
	    break;
	}
	return retval;
    }



    public Vector  GetMembers()                                 {return membership;}
    public int     GetNumMembers()                              {return membership.size();}
    public Object  GetAddress()                                 {return my_addr;}
    public String  GetName()                                    {return channel_name;}
    public void    SetMembershipListener(MembershipListener l)  {mbrship_listener=l;}




    /* --------------------- Ensemble Callbacks ------------------------- */

    public void ReceiveCast(Hot_GroupContext gctx, Object env, 
			    Hot_Endpoint origin, Hot_Message msg) {
	QueueItem item=new QueueItem(CAST_MSG, gctx, msg);
	item.sender=origin;
	try {
	    mq.Add(item);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }

    
    public void ReceiveSend(Hot_GroupContext gctx, Object env, 
			    Hot_Endpoint origin, Hot_Message msg) {
	ReceiveCast(gctx, env, origin, msg);
    }


    public void AcceptedView(Hot_GroupContext gctx, Object env, Hot_ViewState viewState) {
	QueueItem item=new QueueItem(VIEW_MSG);
	item.gctx=gctx; item.view_state=viewState;

	if(connecting == true && viewState.nmembers > 1) {
	    synchronized(this) {
		if(viewState.members.length > 0) {
		    membership.removeAllElements();
		    for(int i=0; i < viewState.members.length; i++)
			membership.addElement(viewState.members[i]);
		}
		notify();            // wakes up Connect()
	    }
	}

	if(my_addr == null && viewState.members != null && viewState.nmembers == 1) {
	    my_addr=viewState.members[0];
	    System.out.println("My address is " + my_addr);
	}

	try {
	    mq.Add(item);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }



    public void Heartbeat(Hot_GroupContext gctx, Object env, int rate) {
	// System.out.println("----> HEARTBEAT");
    }


    public void Block(Hot_GroupContext gctx, Object env) {}


    public void Exit(Hot_GroupContext gctx, Object env) {
	try {
	    mq.Add(new QueueItem(EXIT_MSG));
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }


    /* ------------------------------------------------------------------ */
}
