package JavaGroups;

import java.util.*;
import JavaGroups.algorithms.*;



/**
 * Provides the abstraction of a java.util.Hashtable that is replicated at several
 * locations. Any change to the hashtable (clear, put, remove etc) will transparently be
 * propagated to all replicas in the group. All read-only methods will always access the
 * local replica.<p>
 * Both keys and values added to the hashtable <em>must be serializable</em>, the reason
 * being that they will be sent across the network to all replicas of the group. Having said
 * this, it is now for example possible to add RMI remote objects to the hashtable as they
 * are derived from <code>java.rmi.server.RemoteObject</code> which in turn is serializable.
 * This allows to lookup shared distributed objects by their name and invoke methods on them,
 * regardless of one's onw location. A <code>DistributedHashtable</code> thus allows to
 * implement a distributed naming service in just a couple of lines.<p>
 * An instance of this class will contact an existing member of the group to fetch its
 * initial state (using the state exchange funclet <code>StateExchangeFunclet</code>.
 */
public class DistributedHashtable extends Hashtable implements StateExchange, Cloneable {

    public interface Notification {
	public void EntrySet(Object key, Object value);
	public void EntryRemoved(Object key);
    }


    private transient Dispatcher            disp=null;
    private transient String                groupname=null;
    private transient Hashtable             copy=new Hashtable();
    private transient StateExchangeFunclet  funclet=null;
    private transient final String          funclet_name="StateExchangeFunclet";
    private transient Vector                notifs=new Vector();


    public DistributedHashtable(String groupname, ChannelFactory factory) {
	this.groupname=groupname;
	disp=new Dispatcher(factory, null);
	disp.SetMethodLookup(new MethodLookupClos());
	disp.Join(groupname, this);
	funclet=new StateExchangeFunclet(disp, groupname, funclet_name, this);
	disp.AddFunclet(groupname, this, funclet_name, funclet);
	funclet.Start();
    }

    public void finalize() throws Throwable {
	disp.Leave(groupname, this);
	disp.Destroy();
    }



    public void AddNotifier(Notification n) {
	if(!notifs.contains(n))
	    notifs.addElement(n);
    }

    
    public synchronized Object put(Object key, Object value) {
	try {
	    return disp.SendGetFirst(groupname, "_put", key, value, 500);
	}
	catch(Exception e) {
	    return null;
	}
    }

    public synchronized void clear() {
	try {
	    disp.SendGetN(groupname, "_clear", 0, 0);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }

    public synchronized Object remove(Object key) {
	try {
	    return disp.SendGetFirst(groupname, "_remove", key, 500);
	}
	catch(Exception e) {
	    return null;
	}
    }



    /*------------------------ Callbacks -----------------------*/

    public synchronized Object _put(Object key, Object value) {
	Object retval=super.put(key, value);
	for(int i=0; i < notifs.size(); i++)
	    ((Notification)notifs.elementAt(i)).EntrySet(key, value);
	return retval;
    }


    public synchronized void _clear() {super.clear();}


    public synchronized Object _remove(Object key) {
	Object retval=super.remove(key);
	for(int i=0; i < notifs.size(); i++)
	    ((Notification)notifs.elementAt(i)).EntryRemoved(key);
	return retval;
    }

    /*----------------------------------------------------------*/



    /*-------------------- State Exchange ----------------------*/

    public synchronized void SaveState() {
	Object key, val;
	copy.clear();
	for(Enumeration e=keys(); e.hasMoreElements();) {
	    key=e.nextElement();
	    val=get(key);
	    copy.put(key, val);
	}
    }


    public Object GetState() {return copy; }


    public synchronized void SetState(Object new_state) {
	Hashtable new_copy=(Hashtable)new_state;
	Object    key;
	if(new_copy == null)
	    return;

	_clear(); // remove all elements	
	for(Enumeration e=new_copy.keys(); e.hasMoreElements();) {
	    key=e.nextElement();
	    _put(key, new_copy.get(key));
	}
    }




}
