package JavaGroups;

import java.net.*;
import java.util.*;


/**
 * A Dispatcher maintains a number of channels and allows clients to join one or more of those
 * channels, and send and receive messages to/from channels. When an object joins a channel
 * (given the channel name), it will be dispatched all messages received on that channel in
 * the form of method invocations. Note that in order to receive all messages, an object should
 * implement the methods required by the group (application-specific). It may itself invoke
 * methods on all members of the channel (a channel is nothing else than a <em>group</em> !).<p>
 * <b>Note that an object that has not previously called Join() is nevertheless able to send
 * messages to the group members and receive responses, but requests dispatched to the group 
 * members will not be dispatched to it.</b><p>
 * The <code>Dispatcher</code> class is a replacement for classes <code>RemoteMethodCall</code> 
 * (client side) and <code>MethodInvoker</code> (server side). Instead of using 2 classes,
 * client-server applications can more conveniently be written using the dispatcher. Its main
 * advantage is that, instead of assuming 1 client and 1 server, it allows multiple clients
 * to issue requests and register to get their methods invoked. In one line, the dispatcher
 * is a more sophisticated class offering the combined interfaces of both 
 * <code>RemotemethodCall</code> and <code>MethodInvoker</code> and allows multiple objects
 * to be registered.<p>
 * The example code below shows an application that acts both in the client role by sending
 * multicasts to all members, and in the server role by registering its object for methods
 * to be invoked on it:
 * <pre>
 *
 *
 *import java.util.*;
 *import JavaGroups.channel.*;<br>
 *
 *public class DispatcherTestChannel {<br>
 *    
 *    public Date <b>GetDate</b>() {return new Date();}<br>
 *    
 *    public static void main(String args[]) {
 *	DispatcherTestChannel obj1;
 *	Dispatcher            disp=new Dispatcher();
 *	Object                result;<br>
 *
 *	try {
 *	    obj1=new DispatcherTestChannel();
 *	    <b>disp.Join("GroupA", obj1);</b>
 * 	    while(true) {
 *  		<b>result=(Date)disp.SendGetFirst("GroupA", "GetDate", null, 3000);</b>
 *  		if(result != null)
 *  		    System.out.println("Received response: " + result);
 *  		Thread.currentThread().sleep(2000);
 *  	    }
 *	}
 *	Catch(Exception E) {
 *	    System.Err.println(e);
 *	}
 *    }<br>
 *}
 *</pre><p>
 * @see RemoteMethodCall
 * @see MethodInvoker
 */

public class Dispatcher {
    protected Hashtable       channels=new Hashtable();  // channel name -- ChannelEntry
    protected ChannelFactory  channel_factory;
    protected String          default_properties;
    protected MethodLookup    method_lookup=null;

    public Dispatcher(ChannelFactory f, String default_properties) {
	channel_factory=f;
	this.default_properties=default_properties;
    }

    public void SetMethodLookup(MethodLookup lookup) {method_lookup=lookup;}

    public void HoldRequests(String channel_name) {
	ChannelEntry entry=(ChannelEntry)channels.get(channel_name);
	if(entry == null) {
	    System.err.println("Dispatcher.HoldRequests(): channel " + 
			       channel_name + " not found");
	    return;
	}
	entry.HoldRequests();
    }

    public void ReleaseRequests(String channel_name) {
	ChannelEntry entry=(ChannelEntry)channels.get(channel_name);
	if(entry == null) {
	    System.err.println("Dispatcher.ReleaseRequests(): channel " +
			       channel_name + " not found");
	    return;
	}
	entry.ReleaseRequests();
    }

    public void Destroy() {
	ChannelEntry entry;
	for(Enumeration e=channels.elements(); e.hasMoreElements();) {
	    entry=(ChannelEntry)e.nextElement();
	    entry.Destroy();
	}
	channels=new Hashtable();
    }

    public int GetNumMembers(String channel_name) {
	ChannelEntry entry=(ChannelEntry)channels.get(channel_name);
	if(entry == null) {
	    System.err.println("Channel " + channel_name + " does not exist");
	    return 0;
	}
	return entry.GetChannel().GetNumMembers();
    }

    public Vector GetMembers(String channel_name) {
	ChannelEntry entry=(ChannelEntry)channels.get(channel_name);
	if(entry == null) {
	    System.err.println("Channel " + channel_name + " does not exist");
	    return null;
	}
	return entry.GetChannel().GetMembers();
    }


    public void AddFunclet(String channel_name, Object target, 
			   String funclet_name, Object funclet) {
	try {
	    ChannelEntry e=LocateChannelEntry(channel_name);
	    e.AddFunclet(target, funclet_name, funclet);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }


    /**
     * By default, all channels are created only when needed, using the default properties
     * specified when creating the Dispatcher. This method can be used to create a channel 
     * beforehand, and giving it special properties different from the default one.
     */ 	
    public void CreateChannel(String channel_name, String properties) throws Exception {
	ChannelEntry entry=(ChannelEntry)channels.get(channel_name);
	Channel      channel;
	if(entry != null)
	    throw new Exception("Channel " + channel_name + " already exists");

	channel=channel_factory.CreateChannel(channel_name, properties);
	entry=new ChannelEntry(channel, method_lookup);
	channels.put(channel_name, entry);
    }


    /**
     * Creates a new ChannelEntry (if not yet present) and increments its use count. When the
     * ChannelEntry is not yet present a new ChannelEntry will be created with an accompanying
     * channel which is then connected to.<p>
     * Adds the object to the ChannelEntry's object list. Every message received by the channel, 
     * that is a <em>request</em>, will be dispatched to all objects in the list.
     */
    public synchronized void Join(String channel_name, Object joiner) {
	try {
	    ChannelEntry entry=LocateChannelEntry(channel_name);
	    entry.Add(joiner);
	}
	catch(Exception e) {
	    System.err.println("Dispatcher.Join(): failed finding (or creating) channel " +
			       channel_name + ", exception is: " + e);
	}
    }

    /**
     * Removes the object from the corresponding ChannelEntry's object list and decrements
     * the use count. When the use count is 0, the channel in the ChannelEntry will be 
     * disconnected and destroyed and the ChannelEntry removed.
     */
    public synchronized void Leave(String channel_name, Object leaver) {
	ChannelEntry entry=(ChannelEntry)channels.get(channel_name);
	if(entry == null) {
	    System.err.println("Dispatcher.Leave(): no channel with name " + channel_name +
			       " found");
	    return;
	}
	entry.Remove(leaver);
	if(entry.GetUseCount() <= 0) {
	    entry.Destroy();
	    channels.remove(channel_name);
	}
    }


    public void SetMembershipListener(String channel_name, MembershipListener l) {
	ChannelEntry entry;
	try {
	    entry=LocateChannelEntry(channel_name);
	    entry.GetChannel().SetMembershipListener(l);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }

    public Object GetAddress(String channel_name) {
	try {
	    return LocateChannelEntry(channel_name).GetAddress();
	}
	catch(Exception e) {
	    System.err.println(e);
	    return null;
	}
    }


    protected synchronized ChannelEntry LocateChannelEntry(String channel_name) throws Exception {
	ChannelEntry entry=(ChannelEntry)channels.get(channel_name);
	Channel      channel;
	if(entry != null)
	    return entry;

	channel=channel_factory.CreateChannel(channel_name, default_properties);
	entry=new ChannelEntry(channel, method_lookup);
	channels.put(channel_name, entry);
	return entry;
    }

    /**
     * Send message, add req id in message correlation table and send request
     */
    public Object SendGetFirst(String channel_name, MethodCall method_call, 
			       long timeout) throws Exception {
	Message            msg=new Message(null, null, Util.ObjectToByteBuffer(method_call));
	ChannelEntry       entry=LocateChannelEntry(channel_name);
	Channel            channel=entry.GetChannel();
	MessageCorrelator  corr=entry.GetMessageCorrelator();
       
	corr.AddRequest(msg);
	channel.Send(msg);
	return GetResultAsObject(corr, msg.GetId(), timeout);
    }


    public Vector SendGetN(String channel_name, MethodCall method_call, 
			   int expected_responses, long timeout) throws Exception {
	Message            msg=new Message(null, null, Util.ObjectToByteBuffer(method_call));
	ChannelEntry       entry=LocateChannelEntry(channel_name);
	Channel            channel=entry.GetChannel();
	MessageCorrelator  corr=entry.GetMessageCorrelator();

	if(expected_responses < 1) {  // no response expected
	    msg.SetOneway();
	    channel.Send(msg);
	    return null;
	}
	corr.AddRequest(msg);
	channel.Send(msg);
	return GetResultsAsObject(corr, msg.GetId(), expected_responses, timeout);
    }


    
    protected Object GetResultAsObject(MessageCorrelator corr, long req_id, 
				       long timeout) throws Exception {
	Object  retval=null;
	byte    buf[]=null;
	Message rsp=corr.GetResponse(req_id, timeout);
	if(rsp == null)
	    return null;
	
	buf=rsp.GetBuffer();
	if(buf == null || buf.length < 1)
	    return null;

	retval=Util.ObjectFromByteBuffer(buf);
	if(retval != null && retval instanceof Exception)
	    throw (Exception)retval;

	return retval;
    }

    protected Vector GetResultsAsObject(MessageCorrelator corr, long req_id, 
					int expected_responses, 
					long timeout) throws Exception {
	Vector   retval=new Vector();
	Object   tmp;
	Message  rsp;
	byte     buf[]=null;
	Vector  responses=corr.GetResponses(req_id, expected_responses, timeout);
	if(responses == null)
	    return null;
	
	for(int i=0; i < responses.size(); i++) {
	    rsp=(Message)responses.elementAt(i);
	    buf=rsp.GetBuffer();
	    if(buf == null || buf.length < 1)
		continue;
	    tmp=Util.ObjectFromByteBuffer(buf);
	    retval.addElement(tmp);
	}
	return retval;
    }


    /**
     * Sends a synchronous method call (in the form of a message) to 'dest' and returns 
     * the response. Timeout of 0 means block forever.
     */
    public Object Send(String channel_name, Object dest, MethodCall method_call, 
			boolean oneway, long timeout) throws Exception {
	Message            msg=new Message(dest, null, Util.ObjectToByteBuffer(method_call));
	ChannelEntry       entry=LocateChannelEntry(channel_name);
	Channel            channel=entry.GetChannel();
	MessageCorrelator  corr=entry.GetMessageCorrelator();

	if(oneway) {
	    msg.SetOneway();
	    channel.Send(msg);
	    return null;
	}      
	corr.AddRequest(msg);
	channel.Send(msg);
	return GetResultAsObject(corr, msg.GetId(), timeout);	
    }





    public Object Send(String channel_name, Object dest, String method_name, 
		       boolean oneway, long timeout) throws Exception {
	return Send(channel_name, dest, new MethodCall(method_name), oneway, timeout);
    }

    public Object Send(String channel_name, Object dest, String method_name, 
		       Object arg1, 
		       boolean oneway, long timeout) throws Exception {
	return Send(channel_name, dest, new MethodCall(method_name, arg1), oneway, timeout);
    }

    public Object Send(String channel_name, Object dest, String method_name, 
		       Object arg1, Object arg2,
		       boolean oneway, long timeout) throws Exception {
	return Send(channel_name, dest, new MethodCall(method_name, arg1, arg2), oneway, timeout);
    }

    public Object Send(String channel_name, Object dest, String method_name, 
		       Object arg1, Object arg2, Object arg3,
		       boolean oneway, long timeout) throws Exception {
	return Send(channel_name, dest, new MethodCall(method_name, arg1, arg2, arg3), 
		     oneway, timeout);
    }












    /**
     * Sends the message via the corresponding ChannelEntry and adds it to the MessageCorrelator
     * table. Blocks until response is received. Converts response to Object and returns it.<p>
     */
    public Object SendGetFirst(String channel_name,
			       String method_name,
			       long timeout) throws Exception {
	// 1. Create request (Message) from method_name and args
	// 2. Locate ChannelEntry according to name (create new one if not existant)
	// 3. Add request to ChannelEntry's message correlator
	// 4. Use ChannelEntry's channel to send message
	// 5. Call ChannelEntry's message correlator's GetResponse(). This will block until
	//    the first response is received

	return SendGetFirst(channel_name, new MethodCall(method_name), timeout);
    }


    public Object SendGetFirst(String channel_name,
			       String method_name, Object arg1,
			       long timeout) throws Exception {
	// 1. Create request (Message) from method_name and args
	// 2. Locate ChannelEntry according to name (create new one if not existant)
	// 3. Add request to ChannelEntry's message correlator
	// 4. Use ChannelEntry's channel to send message
	// 5. Call ChannelEntry's message correlator's GetResponse(). This will block until
	//    the first response is received

	return SendGetFirst(channel_name, new MethodCall(method_name, arg1), timeout);
    }


    public Object SendGetFirst(String channel_name,
			       String method_name, Object arg1, Object arg2,
			       long timeout) throws Exception {
	return SendGetFirst(channel_name, new MethodCall(method_name, arg1, arg2), timeout);
    }

    public Object SendGetFirst(String channel_name,
			       String method_name, Object arg1, Object arg2, Object arg3, 
			       long timeout) throws Exception {
	return SendGetFirst(channel_name, 
			     new MethodCall(method_name, arg1, arg2, arg3), timeout);
    }










    public Vector SendGetN(String channel_name,
			   String method_name,
			   int expected_responses, long timeout) throws Exception {
	return SendGetN(channel_name, 
			 new MethodCall(method_name), expected_responses,
			 timeout);
    }


    public Vector SendGetN(String channel_name,
			   String method_name, Object arg1,
			   int expected_responses, long timeout) throws Exception {
	return SendGetN(channel_name, 
			 new MethodCall(method_name, arg1), expected_responses,
			 timeout);
    }

    public Vector SendGetN(String channel_name,
			   String method_name, Object arg1, Object arg2,
			   int expected_responses, long timeout) throws Exception {
	return SendGetN(channel_name, 
			 new MethodCall(method_name, arg1, arg2), expected_responses,
			 timeout);
    }

    public Vector SendGetN(String channel_name,
			   String method_name, Object arg1, Object arg2, Object arg3, 
			   int expected_responses, long timeout) throws Exception {
	return SendGetN(channel_name, 
			 new MethodCall(method_name, arg1, arg2, arg3), expected_responses,
			 timeout);
    }



    
    
}
