package JavaGroups;

import java.util.*;

/**
 * A channel represents one member of a specific group in a Java VM. Only a single object 
 * at a time is allowed to connect to a channel, but there can be more than one channel in
 * an application (for different or the same groups). Using <b>Group</b> objects, multiple
 * objects can connect to a channel.<p>
 * Messages can be multicast to all objects connected to all channels with the same name 
 * using the <em>Cast</em> or <em>Send</em> methods and messages can be received using the
 * <em>Receive</em> method.<p>
 * Various degrees of sophistication in message exchange can be achieved using building
 * blocks on top of channels, e.g. light-weight groups, synchronous message invocation, or
 * remote method calls. Channels are on the same abstraction level as sockets, and should really
 * be simple to use. Higher-level abstractions are all built on top of channels.
 */

public abstract class Channel implements Transportable {

    /** Start receiving (and storing) msgs
     *  @param timeout Wait <em>timeout</em> milliseconds until other members are found
     *                 (a new view is installed). 0 means don't wait. The call returns when 
     *                 some members are found or timeout ms have elasped, whichever comes first
        @exception     Exception When the connection cannot be established
     */
    abstract public void    Connect(long timeout) throws Exception;

    /** Stop receiving msgs */
    abstract public void    Disconnect();

    /** Deletes the channel */
    abstract public void    Destroy();

    /** Send to all members 
     */
    abstract public void    Cast(byte[] msg) throws Exception;

    /** Send to 1 member */
    abstract public void    Send(Object dest_addr, byte[] msg) throws Exception;

    /** Send to a subset of the members */
    abstract public void    Send(Vector dests, byte[] msg) throws Exception;

    /** Send a message */
    abstract public void    Send(Message msg) throws Exception;

    /** Receive a message (must be connected)
     *  @param timeout Value in milliseconds. Return value is null when timeout is
     *         reached. Value of 0 means wait forever */
    abstract public Message Receive(long timeout) throws Exception;

    /** Get all members */
    abstract public Vector  GetMembers();

    /** Get number of members connected to channel <b>including</b> caller */
    abstract public int     GetNumMembers();

    abstract public void    SetMembershipListener(MembershipListener listener);

    abstract public Object  GetAddress();

    abstract public String  GetName();
}
