/*********************************************************************
 Copyright 2000-2001, Princeton University.  All rights reserved. 
 By using this software the USER indicates that he or she has read, 
 understood and will comply with the following:

 --- Princeton University hereby grants USER nonexclusive permission 
 to use, copy and/or modify this software for internal, noncommercial,
 research purposes only. Any distribution, including commercial sale 
 or license, of this software, copies of the software, its associated 
 documentation and/or modifications of either is strictly prohibited 
 without the prior consent of Princeton University.  Title to copyright
 to this software and its associated documentation shall at all times 
 remain with Princeton University.  Appropriate copyright notice shall 
 be placed on all software copies, and a complete copy of this notice 
 shall be included in all copies of the associated documentation.  
 No right is  granted to use in advertising, publicity or otherwise 
 any trademark,  service mark, or the name of Princeton University. 


 --- This software and any associated documentation is provided "as is" 

 PRINCETON UNIVERSITY MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS 
 OR IMPLIED, INCLUDING THOSE OF MERCHANTABILITY OR FITNESS FOR A 
 PARTICULAR PURPOSE, OR THAT  USE OF THE SOFTWARE, MODIFICATIONS, OR 
 ASSOCIATED DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, 
 TRADEMARKS OR OTHER INTELLECTUAL PROPERTY RIGHTS OF A THIRD PARTY.  

 Princeton University shall not be liable under any circumstances for 
 any direct, indirect, special, incidental, or consequential damages 
 with respect to any claim by USER or any third party on account of 
 or arising from the use, or inability to use, this software or its 
 associated documentation, even if Princeton University has been advised
 of the possibility of those damages.
*********************************************************************/

/*********************************************************************
 ADDENDUM: Modifications and additions to the original software,
 zChaff version 2003.11.04, have been made by Ashish Sabharwal (the
 "author") to produce this new software, SymChaff version 2005.03.

 --- This new software and any associated documentation is also
     provided "as is"

 THE AUTHOR OR THE UNIVERSITY OF WASHINGTON MAKE NO REPRESENTATIONS OR
 WARRANTIES, EXPRESS OR IMPLIED, INCLUDING THOSE OF MERCHANTABILITY OR
 FITNESS FOR A PARTICULAR PURPOSE, OR THAT USE OF THE SOFTWARE,
 MODIFICATIONS, OR ASSOCIATED DOCUMENTATION WILL NOT INFRINGE ANY
 PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER INTELLECTUAL PROPERTY RIGHTS
 OF A THIRD PARTY.

 The author or the University of Washington shall not be liable under
 any circumstances for any direct, indirect, special, incidental, or
 consequential damages with respect to any claim by USER or any third
 party on account of or arising from the use of, or inability to use,
 this software or its associated documentation, even if the author or
 the University of Washington have been advised of the possibility of
 those damages.
*********************************************************************/



#ifndef __SAT_SOLVER__
#define __SAT_SOLVER__

#include <set>
#include <stack>

using namespace std;

#include "zchaff_version.h"
#include "zchaff_dbase.h"

#ifndef _SAT_STATUS_
#define _SAT_STATUS_
enum SAT_StatusT {
    UNDETERMINED,
    UNSATISFIABLE,
    SATISFIABLE,
    TIME_OUT,
    MEM_OUT,
    ABORTED
};
#endif

enum SAT_DeductionT {
    CONFLICT,
    NO_CONFLICT,
    TRIV_CONFLICT          // Ashish
};

// Ashish
enum SAT_BranchT {
  SUCCESS,
  SYMBRANCH_OVER_WITH_IMPLICATION,
  SYMBRANCH_OVER_WITHOUT_IMPLICATION,
  FAIL_UNSAT,
  FAIL_SAT,
  FAIL_MEM_OUT
};

class CSolver;

typedef void(*HookFunPtrT)(void *) ;
typedef void(*OutsideConstraintHookPtrT)(CSolver * solver);
typedef bool(*SatHookPtrT)(CSolver * solver);

/**Struct**********************************************************************

  Synopsis    [Sat solver parameters ]

  Description []

  SeeAlso     []

******************************************************************************/
struct CSolverParameters {
  float time_limit;
  int   verbosity;

  struct {
    int strategy;
    int	restart_randomness;	
    int	base_randomness;
    int	bubble_init_step;
    int	decay_period;
  } decision;

  struct {
    bool     enable;
    unsigned interval;
    unsigned max_unrelevance;
    unsigned min_num_lits;
    unsigned max_conf_cls_size;
  } cls_deletion;

  struct {
    bool enable;
    int  first_restart;
    int  backtrack_incr;		
    int  backtrack_incr_incr;
  } restart;

  // Ashish
  struct {
    bool enable;
  } learning;

  struct {
    bool enable;
  } pb;

  struct {
    bool enable;
  } symmetry;

  struct {
    bool enable;
    unsigned  varclass_priority_incr;
  } order;

  struct {
    bool enable;
  } fastbackjumping;

};
/**Struct**********************************************************************

  Synopsis    [Sat solver statistics ]

  Description []

  SeeAlso     []

******************************************************************************/
struct CSolverStats {
    bool	been_reset;		//when delete clause in incremental solving, must reset.
    bool 	is_solver_started;	
    int 	outcome;
    bool	is_mem_out;		//this flag will be set if memory out

    double 	start_cpu_time;    	
    double 	finish_cpu_time;

    int		current_randomness;

    int		next_restart;
    int		restart_incr;
    int		next_cls_deletion;
    int		next_var_score_decay;
    int 	num_free_variables;
    int		num_free_branch_vars;

    long64 	total_bubble_move;
    int 	num_decisions;
    int 	num_symdecisions;      // Ashish
    int		num_backtracks;
    int 	max_dlevel;
    long64 	num_implications;
    int 	num_restarts;
};
/**Class**********************************************************************

  Synopsis    [Sat Solver]

  Description [This class contains the process and datastructrues to solve
               the Sat problem.]

  SeeAlso     []

******************************************************************************/
inline bool cmp_var_stat(const pair<CVariable *,int> & v1, 
			    const pair<CVariable *,int> & v2) 
{	
  // Ashish: >= changed to > so as branching order is not the
  // reverse of the original variable order when scores are the same
  return (v1.second > v2.second);
};

struct cmp_var_assgn_pos {
    bool operator () (CVariable * v1, CVariable * v2)
	{
	    if (v1->dlevel() > v2->dlevel())
		return true;
	    else if (v1->dlevel() < v2->dlevel())
		return false;
	    else if (v1->assgn_stack_pos() > v2->assgn_stack_pos()) 
		return true;
	    return false;
	}
};

//  struct CImplication
//  {
//      long64 id;
//      int lit;
//      int dlevel;
//      int antecedent;
//  };

//  struct cmp_implication {
//      bool operator() (const CImplication & i1, const CImplication & i2) {
//  	if (i1.dlevel > i2.dlevel) 
//  	    return true;
//  	else if (i1.dlevel < i2.dlevel)
//  	    return false;
//  	else if (i1.id > i2.id)
//  	    return true;
//  	return false;
//      }
//  };

//  struct ImplicationQueue: priority_queue<CImplication, vector<CImplication>, cmp_implication> 
//  {
//      CImplication front(void) {
//  	return top();
//      }
//      void dump(ostream & os) {
//  	priority_queue<CImplication, vector<CImplication>, cmp_implication> temp(&this);
//  	os << "Implication Queue Previous: " ;
//  	while(!temp.empty()) {
//  	    CImplication a = temp.front();
//  	    os << "(" << ((a.lit&1)?"-":"+") << (a.lit>>1) 
//  	       << "@" << a.dlevel << ":" << a.antecedent << ")  ";
//  	    temp.pop();
//  	}
//  	os << endl;
//      }
//  };

struct CImplication
{
  int lit;
  int antecedent;
  int dlevel;

  // Ashish: method for priority queue of CImplications
  // Note: g++ 3.3 requires "operator >" to be defined, 
  //       while 2.96 requires "operator <"
  bool operator > (const CImplication & b) const {
    return dlevel > b.dlevel;
  }
};

struct  ImplicationQueue:queue<CImplication> 
{
    void dump(ostream & os) {
	queue<CImplication> temp(*this); //have to make a copy, since queue can only be accessed destructively
	os << "Implication Queue Previous: " ;
	while(!temp.empty()) {
	    CImplication a = temp.front();
	    os << "(" << ((a.lit&1)?"-":"+") << (a.lit>>1) 
	       << "@" << a.dlevel << ":" << a.antecedent << ")  ";
	    temp.pop();
	}
    }
};

// Ashish: the priority of the pqueue is such that low dlevel vars are
// produced first
struct ImplicationPQueue:priority_queue<CImplication,vector<CImplication>,greater<CImplication> > 
{
  void dump(ostream & os) {
    priority_queue<CImplication,vector<CImplication>,greater<CImplication> > temp(*this); //have to make a copy, since queue can only be accessed destructively
    os << "Implication PQueue Previous: " ;
    while(!temp.empty()) {
      CImplication a = temp.top();
      os << "(" << ((a.lit&1)?"-":"+") << (a.lit>>1) 
	 << "@" << a.dlevel << ":" << a.antecedent << ")  ";
      temp.pop();
    }
  }
};

class CSolver:public CDatabase
{
protected:
    int				_id;			//the id of the solver, in case we need to distinguish
    bool			_force_terminate;	//forced to time out by outside caller
    CSolverParameters 		_params;		//parameters for the solver 
    CSolverStats     		_stats;			//statistics and states of the current run

    int 			_dlevel;		//current decision elvel
    vector<vector<int> *> 	_assignment_stack;
    long64			_implication_id;
    ImplicationQueue		_implication_queue;
    ImplicationPQueue		_implication_pqueue;     // Ashish

    //////////////////////////////////////////////
    // Ashish: add structure to keep track of branches for normal DPLL type
    //         flipping using symmetry info
    //         _branchinfo_vector stores detailed info for each sym branch
    //         _branchlevel_stack stores dlevels of multiway sym branches for backtrack
    vector<CBranchInfo>         _branchinfo_vector;
    stack<int>                  _branchlevel_stack;
    int                         _branchlevel_last;
    int                         _triv_conflict_dl;
    ///////////////////////////////////////////////
    
    vector<pair<int,pair< HookFunPtrT, int> > > _hooks;	//hook function run after certain number of decisions
    OutsideConstraintHookPtrT	_outside_constraint_hook;
    SatHookPtrT	_sat_hook; //hook function run after a satisfiable solution found, return true to continue solving and false to terminate as satisfiable
//these are for decision making
    int				_max_score_pos;		//index the unassigned var with max score
    vector<pair<CVariable*,int> >_ordered_vars;		//pair's first pointing to the var, second is the score. 

//these are for conflict analysis
    int 		_num_marked;		//used when constructing learned clauses
    int 		_num_in_new_cl;		//used when constructing learned clauses
    vector<ConsIdx> 	_conflicts;		//the conflicting constraints
    vector<int> 	_conflict_lits; 	//used when constructing learned clause
    vector<int> 	_resolvents;

protected:
    void re_init_stats(void);
    void init_stats(void);
    void init_parameters(void);
    void init_solve(void);			
    void real_solve(void);
    void restart (void);
    int preprocess(void);
    SAT_DeductionT deduce(void);  // Ashish: changed return type
    void run_periodic_functions (void);

    // Ashish: symmetry related
    // These methods and back_track_sym are the key additions for SymChaff
    SAT_DeductionT deduce_sym(void);
    void real_solve_sym(void);
    SAT_BranchT branch_sym(void);
    int branchlevel_next(void) {
      if (_branchlevel_stack.empty()) return 1;
      return _branchlevel_stack.top() + 
        _branchinfo_vector[_branchlevel_stack.top()].branch_size();
    }

//for decision making
    bool decide_next_branch(void);
    void decay_variable_score(void) ;
    void adjust_variable_order(vector<int> & lits);
    void update_var_score(void);

//for preprocessing
    int simplify(void);
    int add_probe_clause(vector<int> lits);
    int do_deduction(void);
    void make_equivalent(vector<pair<int, int> > & equiv);
    void get_rl_candidate(vector<vector<ConsIdx> > * candidate);
    int probe(vector<pair<int,int> > & equiv_svar_pair, bool & modified);
    int probe_one_variable_assignment( vector<ConsIdx>& cls);
    int justify_one_clause(ConsIdx cls_idx) ;

//for conflict analysis
    ConsIdx add_conflict_clause (vector<int> & lits, int gflag);
    ConsIdx add_symconflict_clause (vector<int> & lits); // Ashish
    int analyze_conflicts(void);
    void analyze_trivial_conflict(void);    // Ashish
    ConsIdx finish_add_conf_clause(int gflag);
    int conflict_analysis_grasp (void);
    int conflict_analysis_firstUIP (void);
/*     int conflict_analysis_lastUIP(void); */
/*     int conflict_analysis_allUIP (void); */
/*     int conflict_analysis_mincut (void); */
    int conflict_analysis_decisions_only (void);
    void mark_vars_at_level(ConsIdx cons, int var_idx, int dl);
    void mark_vars_of_dl(vector<int> & lits, int dl);
    void mark_vars_at_current_dlevel(ConsIdx idx, int var_idx) {
	mark_vars_at_level(idx, var_idx, dlevel());
    }
    int find_max_cons_dlevel(ConsIdx cl);	//the max dlevel of all the assigned lits in this constraints
    void back_track(int level);

    // Ashish
    void back_track_sym(bool complete_backtrack = false,
                        bool discard_own_symconf_lits = false,
                        bool discard_last_symconf_lits = false);

//for another way of doing conflict analysis
/*     void set_up_resolve(ConsIdx conf_cons, set<CVariable *, cmp_var_assgn_pos> & conf_lits); */
/*     unsigned resolve_one_lit(set<CVariable *, cmp_var_assgn_pos> & conf_lits); */
/*     bool is_uip_reached(set<CVariable *, cmp_var_assgn_pos> & conf_lits); */
/*     int conflict_analysis_firstUIP_resolve_based(void); */

//for bcp 
    void set_var_value(int var, int value, ConsIdx ante, int dl);
    void set_var_value_not_current_dl(int v, int value);
    void set_var_value_current_dl(int v, int value);
    void unset_var_value(int var);

//misc functions
    bool time_out(void);
    void delete_unrelevant_clauses(void);
    ConsIdx add_clause_with_gid (vector<int> & lits, int gid = 0);

public:
//constructors and destructors
    CSolver();
    ~CSolver();
//member access function
    void set_time_limit(int t);   // Ashish: changed from float to int
    void set_mem_limit(int s);
    void set_decision_strategy(int s) ;
    void set_preprocess_strategy(int s); 
    void enable_cls_deletion(bool allow); 
    void set_cls_del_interval(int n);
    void set_max_unrelevance(int n );
    void set_min_num_clause_lits_for_delete(int n) ;
    void set_max_conflict_clause_length(int l) ;
    void set_allow_multiple_conflict( bool b) ;
    void set_allow_multiple_conflict_clause( bool b) ;
    void set_randomness(int n) ;
    void set_random_seed(int seed);

    // Ashish: small things to have convenient command line behavior
    void set_allow_restart(bool allow);
    void set_allow_learning(bool allow);
    void set_allow_pb(bool allow);
    void set_allow_symmetry(bool allow);
    void set_allow_order(bool allow);
    void set_allow_fastbackjumping(bool allow);

    void set_variable_number(int n);
    int add_variable(void) ;
    void mark_var_branchable(int vid);
    void mark_var_unbranchable(int vid);

    int & dlevel() 		{ return _dlevel; }
    int outcome () 		{ return _stats.outcome; }
    int num_decisions() 	{ return _stats.num_decisions; }
    int num_symdecisions()      { return _stats.num_symdecisions; }
    int & num_free_variables() 	{ return _stats.num_free_variables; }
    int max_dlevel() 		{ return _stats.max_dlevel; }
    long64 num_implications()	{ return _stats.num_implications; }
    long64 total_bubble_move() 	{ return _stats.total_bubble_move; }
    int num_restarts()  	{ return _stats.num_restarts; }

    char * version(void)	{ return __SYMCHAFF_VERSION__; }
    
    float elapsed_cpu_time();
    float cpu_run_time() ;
    int estimate_mem_usage() {	return CDatabase::estimate_mem_usage(); }
    int mem_usage(void); 

//      void queue_implication (int lit, ClauseIdx ante_clause, int dlevel) {
//  	DBG1 (cout << "\t\t\t\t\t\tQueueing " << (lit&0x01?" -":" +") << (lit>>1) ;
//  	      cout << "at " << dlevel << " because of  " << ante_clause << endl; );
//  	CImplication i;
//  	i.lit = lit;
//  	i.id = ++_implication_id;
//  	i.antecedent = ante_clause;
//  	i.dlevel = dlevel;
//  	_implication_queue.push(i);
//      }

    void queue_implication (int lit, ConsIdx ante_cons, int dlevel) {
	DBG1 (cout << "\t\t\t\t\t\tQueueing " << (lit&0x01?"-":"+") << (lit>>1) ;
	      cout << " at " << dlevel << " because of "
              << ((ante_cons == -1) ? "" : ((ante_cons & 0x01) ? "PB " : "CL "))
              << (ante_cons >> 1) << endl; );
	CImplication i;
	i.lit = lit;
	i.antecedent = ante_cons;
  	i.dlevel = dlevel;

	// Ashish: use priority queue instead when symmetry is enabled
	if (_params.symmetry.enable)
	  _implication_pqueue.push(i);
	else
	  _implication_queue.push(i);
    }

//top level function
    int id(void) {return _id; }
    void set_id(int i) { _id = i;}
    void force_terminate (void) { _force_terminate = true;}
    void unset_force_terminate (void) { _force_terminate = false;}

//for incremental SAT
    int add_clause_incr(vector<int> & lits, int gid = 0);
    void make_decision(int lit) {
	++ dlevel();
	queue_implication(lit, NULL_CONS, dlevel());
    }

    void add_hook( HookFunPtrT fun, int interval);
    void add_outside_constraint_hook (OutsideConstraintHookPtrT fun) {_outside_constraint_hook = fun;}
    void add_sat_hook (SatHookPtrT fun) {_sat_hook = fun;}

    void verify_integrity(void);
    void delete_cons_group(int gid);
    void reset (void);
    int	solve(void);
    ConsIdx add_orig_clause (vector<int> & lits, int gid = 0);
    void clean_up_dbase(void);
    void dump_assignment_stack(ostream & os = cout);
    void dump_implication_queue(ostream & os = cout);
    void dump_implication_pqueue(ostream & os = cout);   // Ashish

    void print_cls(ostream & os= cout);
    void dump(ostream & os = cout ) {
	CDatabase::dump(os);
	dump_assignment_stack(os);
    }
    void add_outside_clauses(void);

    // Ashish
    ConsIdx add_orig_pbcons (vector<int> & lits, vector<int> & coeffs, int rhs);
};
#endif
