/*********************************************************************
 Copyright 2000-2003, Princeton University.  All rights reserved. 
 By using this software the USER indicates that he or she has read, 
 understood and will comply with the following:

 --- Princeton University hereby grants USER nonexclusive permission 
 to use, copy and/or modify this software for internal, noncommercial,
 research purposes only. Any distribution, including commercial sale 
 or license, of this software, copies of the software, its associated 
 documentation and/or modifications of either is strictly prohibited 
 without the prior consent of Princeton University.  Title to copyright
 to this software and its associated documentation shall at all times 
 remain with Princeton University.  Appropriate copyright notice shall 
 be placed on all software copies, and a complete copy of this notice 
 shall be included in all copies of the associated documentation.  
 No right is  granted to use in advertising, publicity or otherwise 
 any trademark, service mark, or the name of Princeton University. 


 --- This software and any associated documentation is provided "as is" 

 PRINCETON UNIVERSITY MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS 
 OR IMPLIED, INCLUDING THOSE OF MERCHANTABILITY OR FITNESS FOR A 
 PARTICULAR PURPOSE, OR THAT  USE OF THE SOFTWARE, MODIFICATIONS, OR 
 ASSOCIATED DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, 
 TRADEMARKS OR OTHER INTELLECTUAL PROPERTY RIGHTS OF A THIRD PARTY.  

 Princeton University shall not be liable under any circumstances for 
 any direct, indirect, special, incidental, or consequential damages 
 with respect to any claim by USER or any third party on account of 
 or arising from the use, or inability to use, this software or its 
 associated documentation, even if Princeton University has been advised
 of the possibility of those damages.
*********************************************************************/

/*********************************************************************
 ADDENDUM: Modifications and additions to the original software,
 zChaff version 2003.11.04, have been made by Ashish Sabharwal (the
 "author") to produce this new software, SymChaff version 2005.03.

 --- This new software and any associated documentation is also
     provided "as is"

 THE AUTHOR OR THE UNIVERSITY OF WASHINGTON MAKE NO REPRESENTATIONS OR
 WARRANTIES, EXPRESS OR IMPLIED, INCLUDING THOSE OF MERCHANTABILITY OR
 FITNESS FOR A PARTICULAR PURPOSE, OR THAT USE OF THE SOFTWARE,
 MODIFICATIONS, OR ASSOCIATED DOCUMENTATION WILL NOT INFRINGE ANY
 PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER INTELLECTUAL PROPERTY RIGHTS
 OF A THIRD PARTY.

 The author or the University of Washington shall not be liable under
 any circumstances for any direct, indirect, special, incidental, or
 consequential damages with respect to any claim by USER or any third
 party on account of or arising from the use of, or inability to use,
 this software or its associated documentation, even if the author or
 the University of Washington have been advised of the possibility of
 those damages.
*********************************************************************/

#include <cstdio>
#include <algorithm>
#include <iterator>
#include <fstream>
#include "zchaff_solver.h"
//typedef double DOUBLE;
//  extern "C" DOUBLE PLED_FindMaxFlow(int nvtxs, int nedges,
//  				   int *edge_from, int *edge_to, 
//  				   DOUBLE *ewts, int *part);

//#define VERIFY_ON

#ifdef VERIFY_ON
ofstream verify_out("resolve_trace");
#endif

void CSolver::re_init_stats(void)
{
    _stats.is_mem_out		= false;
    _stats.outcome		= UNDETERMINED;

    _stats.next_restart 	= _params.restart.first_restart;
    _stats.restart_incr 	= _params.restart.backtrack_incr;
    _stats.next_cls_deletion 	= _params.cls_deletion.interval;
    _stats.next_var_score_decay = _params.decision.decay_period;

    _stats.current_randomness 	= _params.decision.base_randomness;

    _stats.total_bubble_move 	= 0;
    _stats.num_decisions 	= 0;
    _stats.num_symdecisions     = 0;       // Ashish
    _stats.num_backtracks	= 0;
    _stats.max_dlevel 		= 0;
    _stats.num_implications   	= 0;
    _stats.num_restarts   	= 0;

    _stats.start_cpu_time	= get_cpu_time();
    _stats.finish_cpu_time	= 0;
}

void CSolver::init_stats(void) 
{
    re_init_stats();

    _stats.been_reset		= true;
    _stats.outcome 		= UNDETERMINED;

    _stats.num_free_variables 	= 0;
    _stats.num_free_branch_vars = 0;
}

void CSolver::init_parameters(void)
{
    _params.verbosity				= 0;
    _params.time_limit				= 3600 * 6; // Ashish: 6 hours

    _params.decision.strategy 			= 0;
    _params.decision.restart_randomness		= 0;	
    _params.decision.base_randomness		= 0;
    _params.decision.decay_period		= 0x100;
    _params.decision.bubble_init_step 		= 0x400;

    _params.cls_deletion.enable 		= true ;
    _params.cls_deletion.interval		= 5000;
    _params.cls_deletion.max_unrelevance	= 20;
    _params.cls_deletion.min_num_lits		= 100;
    _params.cls_deletion.max_conf_cls_size	= 5000;

    _params.restart.enable			= false;	
    _params.restart.first_restart		= 5000;
    _params.restart.backtrack_incr		= 1500;
    _params.restart.backtrack_incr_incr 	= 200;

    // Ashish
    _params.learning.enable                     = true;
    _params.pb.enable                           = false;
    _params.symmetry.enable                     = false;
    _params.fastbackjumping.enable              = true;
    _params.order.enable                        = false;
    _params.order.varclass_priority_incr        = 1<<22;
       // Enough for up to 2^(32-22) = 1024 varclasses
       // Allows variable scores up to 2^22 - 1
    int max_varclass_priority_groups = 1<<(32-_params.order.varclass_priority_incr);
    if (num_varclass_priority_groups() > max_varclass_priority_groups) {
      // This should happen EXTREMELY RARELY!
      cerr << "ERROR 12: Number of varclass priority groups exceeds the maximum\n"
           << "limit of " << max_varclass_priority_groups << ".\n";
      exit(12);
    }
}

CSolver::CSolver(void) {
    init_parameters();
    init_stats();
    _dlevel			= 0;		
    _force_terminate		= false;
    _implication_id		= 0;
    _num_marked			= 0;
    _num_in_new_cl		= 0;
    _outside_constraint_hook	= NULL;
    _sat_hook	                = NULL;
    // Ashish
    _branchlevel_last           = -1;
    _triv_conflict_dl           = -1;
}

CSolver::~CSolver()
{
   while (!_assignment_stack.empty()) {
     delete _assignment_stack.back();
     _assignment_stack.pop_back();
   }
}

void CSolver::set_time_limit(int t) 
{ 
    _params.time_limit = t; 
}

float CSolver::elapsed_cpu_time(void) 
{
    return get_cpu_time() - _stats.start_cpu_time;
}

float CSolver::cpu_run_time() 
{ 
    return (_stats.finish_cpu_time - _stats.start_cpu_time);
}

void CSolver::set_variable_number(int n) 
{ 	
  assert (num_variables() == 0);
  CDatabase::set_variable_number(n);
  _stats.num_free_variables	= num_variables();
  while (_assignment_stack.size() <= num_variables())
    _assignment_stack.push_back(new vector<int>);
  assert (_assignment_stack.size() == num_variables() + 1);

  // Ashish: update size of branchinfo_vector as well
  if (_params.symmetry.enable)
    _branchinfo_vector.resize(num_variables() + 2);   // first and last empty
}

int CSolver::add_variable(void) 
{
  int num = CDatabase::add_variable();
  ++_stats.num_free_variables;
  while (_assignment_stack.size() <= num_variables())
    _assignment_stack.push_back(new vector<int>);
  assert (_assignment_stack.size() == num_variables() + 1);

  // Ashish: update size of branchinfo_vector as well
  if (_params.symmetry.enable)
    _branchinfo_vector.resize(num_variables() + 2);   // first and last empty

  return num;
}

void CSolver::set_mem_limit(int s)
{ 
    CDatabase::set_mem_limit(s); 
}

void CSolver::set_decision_strategy(int s) 
{
    _params.decision.strategy = s; 
}

void CSolver::enable_cls_deletion(bool allow) 
{
    _params.cls_deletion.enable = allow; 
}

void CSolver::set_cls_del_interval(int n)
{
    _params.cls_deletion.interval = n; 
}

void CSolver::set_max_unrelevance(int n ) 
{
    _params.cls_deletion.max_unrelevance = n; 
}

void CSolver::set_min_num_clause_lits_for_delete(int n) 
{
    _params.cls_deletion.min_num_lits = n; 
}

void CSolver::set_max_conflict_clause_length(int l) 
{
    _params.cls_deletion.max_conf_cls_size = l; 
}

void CSolver::set_randomness(int n) 
{
    _params.decision.base_randomness = n; 
}

void CSolver::set_random_seed(int seed) {
    srand (seed);
}

// Ashish

void CSolver::set_allow_restart(bool allow) {
  _params.restart.enable = allow;
}

void CSolver::set_allow_learning(bool allow) {
  _params.learning.enable = allow;
}

void CSolver::set_allow_pb(bool allow) {
  _params.pb.enable = allow;
}

void CSolver::set_allow_symmetry(bool allow) {
  _params.symmetry.enable = allow;
}

void CSolver::set_allow_order(bool allow) {
  _params.order.enable = allow;
}

void CSolver::set_allow_fastbackjumping(bool allow) {
  _params.fastbackjumping.enable = allow;
}



void CSolver::add_hook( HookFunPtrT fun, int interval) 
{
    pair<HookFunPtrT, int> a(fun, interval);
    _hooks.push_back(pair<int, pair<HookFunPtrT, int> > (0, a));
}

void CSolver::run_periodic_functions(void)
{
    //a. clause deletion
    if ( _params.cls_deletion.enable &&
	 _stats.num_backtracks > _stats.next_cls_deletion) {
	_stats.next_cls_deletion = _stats.num_backtracks + _params.cls_deletion.interval;
	delete_unrelevant_clauses(); 
    }
    //b. restart
    if (_params.restart.enable && 
	_stats.num_backtracks > _stats.next_restart) {
	_stats.next_restart = _stats.num_backtracks + _stats.restart_incr;
	_stats.restart_incr += _params.restart.backtrack_incr_incr;
	restart();
    }    
    //c. update var stats for decision
    if (_stats.num_decisions + _stats.num_symdecisions > _stats.next_var_score_decay) {
	_stats.next_var_score_decay =
          _stats.num_decisions
          + _stats.num_symdecisions
          + _params.decision.decay_period;
	decay_variable_score();
    }
    //d. run hook functions
    for (unsigned i=0; i< _hooks.size(); ++i) {
	pair<int,pair<HookFunPtrT, int> > & hook = _hooks[i];
	if (_stats.num_decisions + _stats.num_symdecisions >= hook.first) {
	    hook.first += hook.second.second;
	    hook.second.first((void *) this);
	}
    }
} 


void CSolver::init_solve(void)
{
  CDatabase::init_stats();
  re_init_stats();
  _stats.been_reset 		= false;

  assert (_conflicts.empty());
  assert (_conflict_lits.empty());
  assert (_num_marked == 0);
  assert (_num_in_new_cl == 0);
  assert (_dlevel == 0);
	
  for (unsigned i=0, sz = variables().size(); i< sz; ++i) {
    variable(i).score(0) = variable(i).lits_count(0);
    variable(i).score(1) = variable(i).lits_count(1);
  }
  // Ashish: adjust scores according to varclass_priority_groups if specified
  if (_params.order.enable) {
    int n_classes = num_varclass_priority_groups();
    // compute number of varclasses in priority_groups
    int m = 0;
    for (int i=0; i<n_classes; i++)
      m += varclass_priority_group(i).size();
    if (m == num_varclasses()+1)
      --n_classes;      // all varclasses covered; need not weigh the lowest class
    for (int i=0; i<n_classes; i++) {
      vector<int> & priority_group = varclass_priority_group(i);
      for (vector<int>::iterator varclass_itr=priority_group.begin(); varclass_itr!=priority_group.end(); ++varclass_itr) {
        assert(*varclass_itr >= 1 && *varclass_itr <= num_varclasses()+1);
        if (*varclass_itr == num_varclasses()+1) {   // variables without symmetry
          for (int j=0; j<num_variables(); j++) {
            if (!var_findsymindex(j)) {
              variable(j).score(0) += _params.order.varclass_priority_incr * (n_classes-i);
              variable(j).score(1) += _params.order.varclass_priority_incr * (n_classes-i);
            }
          }
          continue;
        }
        // variables with symmetry
        vector<int> & varmap = varclass(*varclass_itr).varmap;
        for (vector<int>::iterator var_itr=varmap.begin(); var_itr!=varmap.end(); ++var_itr) {
          variable(*var_itr).score(0) += _params.order.varclass_priority_incr * (n_classes-i);
          variable(*var_itr).score(1) += _params.order.varclass_priority_incr * (n_classes-i);
        }
      }
    }
  }

  _ordered_vars.resize( num_variables());
  update_var_score();

  DBG2(dump());
}

void CSolver::set_var_value(int v, int value, ConsIdx ante, int dl)
{
    assert (value == 0 || value == 1);
    DBG2(dump());
    CHECK(verify_integrity());
    DBG1 (cout << "Setting\t" << (value>0?"+":"-") << v 
	  //	  << " at " << dlevel() << " because " << ante<< endl;);
	  << " at " << dl << " because of " 
          << ((ante == -1) ? "" : ((ante & 0x01) ? "PB " : "CL "))
          << (ante >> 1) << endl;);  // Ashish
    CVariable & var = _variables[v];
    assert (var.value() == UNKNOWN);
    var.set_dlevel(dl);
    var.set_value(value);
    var.antecedent() = ante;

    if (dl == dlevel())
	set_var_value_current_dl(v, value);
    else 
	set_var_value_not_current_dl(v, value);
	
    ++_stats.num_implications ;
    if (var.is_branchable()) 
	--num_free_variables();
}

void CSolver::set_var_value_current_dl(int v, int value)
{
  vector<CLitPoolElement *> & watchs = variable(v).watched(value);
  for (vector <CLitPoolElement *>::iterator itr = watchs.begin(); itr != watchs.end(); ++itr) {
    ConsIdx cl_idx;
    CLitPoolElement * other_watched;
    CLitPoolElement * watched = *itr;
    int dir = watched->direction();
    CLitPoolElement * ptr = watched;
    while(1) {
      ptr += dir;
      if (ptr->val() <= 0) { //reached one end of the clause
	if (dir == 1) 	//reached the right end, i.e. spacing element is the cl_id
	  cl_idx = ptr->get_cons_index();
	if (dir == watched->direction()) { //we haven't go both directions.
	  ptr = watched;	
	  dir = -dir;			//change direction, go the other way
	  continue;
	}
	//otherwise, we have already go through the whole clause
	int the_value = literal_value (*other_watched);
	if (the_value == 0) //a conflict
	  _conflicts.push_back(cl_idx);
	else if ( the_value != 1) //i.e. unknown
	  queue_implication (other_watched->s_var(), cl_idx, dlevel());
	break;
      }
      if (ptr->is_watched()) {	//literal is the other watched lit, skip it.
	other_watched = ptr;
	continue;
      }
      if (literal_value(*ptr) == 0) // literal value is 0, keep going
	continue;
      //now the literal's value is either 1 or unknown, watch it instead
      int v1 = ptr->var_index();
      int sign = ptr->var_sign();
      variable(v1).watched(sign).push_back(ptr);
      ptr->set_watch(dir);
      //remove the original watched literal from watched list
      watched->unwatch();
      *itr = watchs.back(); //copy the last element in it's place
      watchs.pop_back();	//remove the last element
      --itr;		//do this so we don't skip one during traversal
      break;
    }
  }

  // Ashish: fix PB watched literals
  vector<CLitPoolElement *> & pbwatchs = variable(v).pbwatched(value);
  for (vector <CLitPoolElement *>::iterator itr = pbwatchs.begin(); itr != pbwatchs.end(); ++itr) {
    CLitPoolElement * lit = *itr;
    ConsIdx pb_idx = lit->find_cons_index();
    CPBcons & pb = pbcons(pb_idx);

    // stop watching lit and update watch_slack
    lit->unwatch();
    pb.incr_watch_slack(- pb.coeff(lit));

    // fix watch_max if needed
    if (pb.coeff(lit) == pb.watch_max()) {
      CLitPoolElement * ptr = pb.literals();
      while(!ptr->is_watched()) {
	++ptr; 
      }
      pb.set_watch_max(pb.coeff(ptr));   // the first one must be the max (sorted)
    }

    // search for new watches to replace lit
    bool watch_max_is_set = false;
    CLitPoolElement * ptr = pb.literals();
    while (ptr->is_literal() && pb.watch_slack()<0) {
      if (!ptr->is_watched() && literal_value(*ptr) != 0) {
	int v_idx = ptr->var_index();
	int v_sign = ptr->var_sign();
	CVariable & v = variable(v_idx);
	v.pbwatched(v_sign).push_back(ptr);
	ptr->set_watch();
	pb.incr_watch_slack(pb.coeff(ptr));
      }
      if (!watch_max_is_set && ptr->is_watched()) {   // the first one must be the max (sorted)
	pb.set_watch_max(pb.coeff(ptr));
	watch_max_is_set = true;
      }
      ++ptr;
    }

    if (pb.watch_slack() < 0) {
      // All non-false lits in the pb must be getting watched
      // Any lit with coeff more than real slack is unit-implied
      int real_slack = pb.watch_slack() + pb.watch_max();
      ptr = pb.literals();
      while (ptr->is_literal()) {
	if (ptr->is_watched() && pb.coeff(ptr) > real_slack) {
	  int the_value = literal_value(*ptr);
	  if (the_value == 0) // a conflict
	    _conflicts.push_back(pb_idx);
	  else if (the_value != 1) // must be unknown
	    queue_implication(ptr->s_var(), pb_idx, dlevel());
	}
	++ptr;
      }

      // Add the original watch lit back so as to make watch_slack >= 0
      // Note that this is the literal with max dlevel.
      // The resulting watch set may be weaker (larger) than what we
      //  started with, but that's OK
      lit->set_watch();
      int coeff = pb.coeff(lit);
      pb.incr_watch_slack(coeff);
      if (coeff > pb.watch_max())
	pb.set_watch_max(coeff);
      continue;   // avoid removing lit from the list of watches
    }

    // remove itr from the list of watches
    *itr = pbwatchs.back();   // copy the last element in its place
    pbwatchs.pop_back();
    --itr;
  }
}

void CSolver::set_var_value_not_current_dl(int v, int value)
{
  //the only difference between this one and the last function
  //is that because the assignment var is not at current_dl, 
  //(that means, < current_dl), it's possible some variable 
  //have a dlevel higher than the assigned one. So, we need 
  //to make sure that watched literal has the highest dlevel if
  //we can't find other unassigned or value 1 literals.
  //also, it's possible we need to readjust decision levels of 
  //some variables implied.

  ConsIdx cl_idx;
  CLitPoolElement * watched, * other_watched, * ptr, * max_ptr = NULL;
  int dir,max_dl;
  vector<CLitPoolElement *> & watchs = variable(v).watched(value);
  for (vector <CLitPoolElement *>::iterator itr = watchs.begin(); 
       itr != watchs.end(); ++itr) {
    max_dl = -1;
    watched = *itr;
    dir = watched->direction();
    ptr = watched;
    while(1) {
      ptr += dir;
      if (ptr->val() <= 0) {
	if (dir == 1) 
	  cl_idx = ptr->get_cons_index();
	if (dir == watched->direction()) {
	  ptr = watched;
	  dir = -dir;
	  continue;
	}
	// otherwise we have gone through the whole clause
	CLitPoolElement * current_watched = watched;
	if (variable(watched->var_index()).dlevel() < max_dl) {
	  int v1 = max_ptr->var_index();
	  int sign = max_ptr->var_sign();
	  variable(v1).watched(sign).push_back(max_ptr);
	  max_ptr->set_watch(dir);
	  watched->unwatch();
	  *itr = watchs.back();
	  watchs.pop_back();
	  --itr;
	  current_watched = max_ptr;
	}
	int max = variable(current_watched->var_index()).dlevel();
	int the_value = literal_value (*other_watched);
	if (the_value == 0) {
	  //it's a conflict, but we will not put into _conflicts
	  //instead, make it a implication so it will be resolved 
	  //in deduce()
	  assert (variable(other_watched->var_index()).dlevel() >= max);
	  //  		    cout << "Queueing Conflict for " << other_watched->var_index() << " orig DL: " 
	  //  			 << variable(other_watched->var_index()).dlevel() << " current DL: " 
	  //  			 << max << endl;
	  queue_implication(other_watched->s_var(), cl_idx, max);
	}
	else if ( the_value == 1) {
	  int v1 = other_watched->var_index();
	  if (variable(v1).dlevel() > max)
	    queue_implication(other_watched->s_var(), cl_idx, max);
	}
	else 
	  queue_implication (other_watched->s_var(), cl_idx, max);
	break;
      }
      if (ptr->is_watched()) {
	other_watched = ptr;
	continue;
      }
      if (literal_value(*ptr) == 0) {
	//keep track of the 0 literal with max decision level
	int ptr_dl = variable(ptr->var_index()).dlevel();
	if ( ptr_dl > max_dl) {
	  max_dl = ptr_dl;
	  max_ptr = ptr;
	}
	continue;
      }
      //now it's value is either 1 or unknown
      int v1 = ptr->var_index();
      int sign = ptr->var_sign();
      variable(v1).watched(sign).push_back(ptr);
      watched->unwatch();
      ptr->set_watch(dir);

      *itr = watchs.back();
      watchs.pop_back();
      --itr;
      break;
    }
  }

  // Ashish: fix PB watched literals
  // Again, use max_dl literals if necessary
  vector<CLitPoolElement *> & pbwatchs = variable(v).pbwatched(value);
  for (vector <CLitPoolElement *>::iterator itr = pbwatchs.begin(); itr != pbwatchs.end(); ++itr) {
    CLitPoolElement * lit = *itr;
    ConsIdx pb_idx = lit->find_cons_index();
    CPBcons & pb = pbcons(pb_idx);
    int max_dl = -1;

    // stop watching lit and remove itr from the list of watches
    lit->unwatch();
    *itr = pbwatchs.back();   // copy the last element in its place
    pbwatchs.pop_back();
    --itr;

    // update watch_slack and fix watch_max if needed
    pb.incr_watch_slack(- pb.coeff(lit));
    if (pb.coeff(lit) == pb.watch_max()) {
      CLitPoolElement * ptr = pb.literals();
      while(!ptr->is_watched()) {
	++ptr; 
      }
      pb.set_watch_max(pb.coeff(ptr));   // the first one must be the max (sorted)
    }

    // search for new watches to replace lit
    // also keep track of false lits at max_dl
    bool watch_max_is_set = false;
    CLitPoolElement * ptr = pb.literals();
    while (ptr->is_literal() && pb.watch_slack()<0) {
      if (!ptr->is_watched()) {
	if (literal_value(*ptr) != 0) {
	  int v_idx = ptr->var_index();
	  int v_sign = ptr->var_sign();
	  CVariable & v = variable(v_idx);
	  v.pbwatched(v_sign).push_back(ptr);
	  ptr->set_watch();
	  pb.incr_watch_slack(pb.coeff(ptr));
	}
	else {
	  int new_dl = variable(ptr->var_index()).dlevel();
	  if (new_dl > max_dl)
	    max_dl = new_dl;
	}
      }
      if (!watch_max_is_set && ptr->is_watched()) {   // the first one must be the max (sorted)
	pb.set_watch_max(pb.coeff(ptr));
	watch_max_is_set = true;
      }
      ++ptr;
    }

    if (pb.watch_slack() < 0) {
      // All non-false lits in the pb must be getting watched
      // Any lit with coeff more than real slack is unit-implied
      int real_slack = pb.watch_slack() + pb.watch_max();
      ptr = pb.literals();
      while (ptr->is_literal()) {
	if (ptr->is_watched() && pb.coeff(ptr) > real_slack) {
	  int the_value = literal_value(*ptr);
	  static bool first_time = true;
	  if (first_time) {
	    cerr << "WARNING: queueing pb-implication at max_dl could be wrong!" << endl;
	    first_time = false;
	  }
	  if (the_value == 0) // a conflict normally, but in this case
	    // an implication (because its dlevel is higher) to be
	    // handled later in deduce()
	    queue_implication(ptr->s_var(), pb_idx, max_dl);
	  else if (the_value == 1) {
	    if (variable(ptr->var_index()).dlevel() > max_dl)
	      queue_implication(ptr->s_var(), pb_idx, max_dl);
	  }
	  else // must be unknown
	    queue_implication(ptr->s_var(), pb_idx, max_dl);
	}
	++ptr;
      }

      // Add enough max_dl lits to watch list so as to make watch_slack >= 0
      // The following loop will eventually stop because the constraint is
      //  not inherently unsatisfiable
      while (pb.watch_slack() < 0) {
	watch_max_is_set = false;
	int next_max_dl = 0;
	CLitPoolElement * ptr = pb.literals();
	while (ptr->is_literal() && pb.watch_slack()<0) {
	  if (ptr->is_watched()) {
	    watch_max_is_set = true;
	    ++ptr;
	    continue;
	  }
	  // because watch_slack < 0, any unwatched lits are false
	  int ptr_dl = variable(ptr->var_index()).dlevel();
	  if (ptr_dl == max_dl) {
	    int v_idx = ptr->var_index();
	    int v_sign = ptr->var_sign();
	    CVariable & v = variable(v_idx);
	    v.pbwatched(v_sign).push_back(ptr);
	    if (v_idx == lit->var_index())
	      ++itr;   // don't re-process if this is the same as lit (*itr)
	    ptr->set_watch();
	    pb.incr_watch_slack(pb.coeff(ptr));
	    if (!watch_max_is_set) {
	      pb.set_watch_max(pb.coeff(ptr));
	      watch_max_is_set = true;
	    }
	  }
	  else if (ptr_dl < max_dl && ptr_dl > next_max_dl)
	    next_max_dl = ptr_dl;
	  ++ptr;
	}
	max_dl = next_max_dl;
      }
    }
  }
}

void CSolver::unset_var_value(int v)
{
    if (v == 0) return;
    DBG1(cout <<"Unset var " << (variable(v).value()?"+":"-") << v << endl;);
    CVariable & var = variable(v);
    var.set_value(UNKNOWN);
    var.set_antecedent(NULL_CONS);
    var.set_dlevel( -1);
    var.assgn_stack_pos() = -1;

    if (var.is_branchable()) {
	++num_free_variables();
	if (var.var_score_pos() < _max_score_pos)
	    _max_score_pos = var.var_score_pos();
    }
}

int CSolver::find_max_cons_dlevel(ConsIdx idx)
{
//if cons has single literal, then the dlevel for it should be 0
//thus, we need to set initial max_level = 0 instead
//of -1 because if clause has single literal, then the
//loop will found no literal with value UNKNOWN
    int max_level = 0;
    if (idx == NULL_CONS) 
	return dlevel();
    CConstraint & cons = constraint(idx); 
    for (unsigned i=0, sz= cons.num_lits(); i<sz;  ++i) {
	int var_idx = cons.literal(i).var_index();
	if (_variables[var_idx].value() != UNKNOWN) {
	    if (max_level < _variables[var_idx].dlevel())
		max_level =  _variables[var_idx].dlevel();
	}
    }
    return max_level; 
}

void CSolver::dump_assignment_stack(ostream & os ) {
    cout << "Assignment Stack:  ";
    for (int i=0; i<= dlevel(); ++i) {
	cout << "(" <<i << ":";
	for (unsigned j=0; j<(*_assignment_stack[i]).size(); ++j )
	    cout << ((*_assignment_stack[i])[j]&0x1?"-":"+")
		 << ((*_assignment_stack[i])[j] >> 1) << " ";
	cout << ") " << endl;
    }
    cout << endl;
}

void CSolver::dump_implication_queue(ostream & os) 
{
    _implication_queue.dump(os);
}

// Ashish
void CSolver::dump_implication_pqueue(ostream & os) 
{
    _implication_pqueue.dump(os);
}

void CSolver::delete_cons_group (int gid)
{
  assert( is_gid_allocated(gid) );
  if (_stats.been_reset==false)
    reset(); //if delete some clause, then implication queue are invalidated
  for (vector<CClause>::iterator itr1 = clauses().begin(); 
       itr1 != clauses().end(); ++itr1) {
    CClause & cl = * itr1;
    if (cl.status() != DELETED_CONS) {
      if (cl.gid(gid) == true) {
        mark_cons_deleted(cl);
      }
    }
  }
  // Ashish
  for (vector<CPBcons>::iterator itr2 = pbcons().begin(); 
       itr2 != pbcons().end(); ++itr2) {
    CPBcons & pb = * itr2;
    if (pb.status() != DELETED_CONS) {
      if (pb.gid(gid) == true) {
        mark_cons_deleted(pb);
      }
    }
  }
  //delete the index from variables
  for (vector<CVariable>::iterator itr = variables().begin(); 
       itr != variables().end(); ++ itr) {
    for (unsigned i=0; i<2; ++i) { //for each phase
      //delete the lit index from the vars
#ifdef KEEP_LIT_CONS
      vector<ConsIdx> & lit_cons = (*itr).lit_cons(i);
      for (vector<ConsIdx>::iterator itr1 = lit_cons.begin();
           itr1 != lit_cons.end(); ++ itr1) 
        if ( cons(*itr1).status() == DELETED_CONS ) {
          *itr1 = lit_cons.back();
          lit_cons.pop_back();
          -- itr1;
        }
#endif
      //delete the watched index from the vars
      vector<CLitPoolElement *> & watched = (*itr).watched(i);
      for (vector<CLitPoolElement *>::iterator itr1 = watched.begin(); 
           itr1 != watched.end(); ++itr1) 
        if ( (*itr1)->val() <= 0) {
          *itr1 = watched.back();
          watched.pop_back();
          --itr1;
        }
    }
  }
  free_gid(gid);
}

void CSolver::reset(void)
{
    if (_stats.been_reset)
    	return;
    if (num_variables()==0) return;
    back_track(0);
    _conflicts.clear();
    while (!_implication_queue.empty())
	_implication_queue.pop();
    // Ashish
    while (!_implication_pqueue.empty())
	_implication_pqueue.pop();

    _stats.is_solver_started 	= false;
    _stats.outcome		= UNDETERMINED;
    _stats.been_reset 		= true;
}

void CSolver::delete_unrelevant_clauses(void)
{
    if (CDatabase::_stats.mem_used_up) {
	CDatabase::_stats.mem_used_up = false;
	if (++CDatabase::_stats.mem_used_up_counts < 5) {
	    _params.cls_deletion.max_unrelevance = (int) (_params.cls_deletion.max_unrelevance * 0.8);
	    if (_params.cls_deletion.max_unrelevance < 4)
		_params.cls_deletion.max_unrelevance = 4;
	    _params.cls_deletion.min_num_lits = (int) (_params.cls_deletion.min_num_lits* 0.8);
	    if (_params.cls_deletion.min_num_lits < 10)
		_params.cls_deletion.min_num_lits = 10;
	    _params.cls_deletion.max_conf_cls_size = (int) (_params.cls_deletion.max_conf_cls_size*0.8);
	    if (_params.cls_deletion.max_conf_cls_size < 50 )
		_params.cls_deletion.max_conf_cls_size = 50;
	    DBG1(
		cout << "Forced to be more aggressive in clause deletion. " << endl;
		cout <<"MaxUnrel: " << _params.cls_deletion.max_unrelevance 
		<< "  MinLenDel: " << _params.cls_deletion.min_num_lits
		<< "  MaxLenCL : " << _params.cls_deletion.max_conf_cls_size << endl;
		);
	}
    }
    unsigned original_del_cls = num_deleted_clauses();
    int original_del_lits = num_deleted_literals();
    DBG2 (dump());
    for (vector<CClause>::iterator itr1 = clauses().begin(); 
	 itr1 != clauses().end(); ++itr1) {

	CClause & cl = * itr1;

	if (cl.status()!=CONFLICT_CONS || cl.num_lits() < _params.cls_deletion.min_num_lits ) continue;

	int val_0_lits = 0, val_1_lits = 0, unknown_lits = 0;
	for (unsigned i=0; i< cl.num_lits(); ++i) {
	    int lit_value = literal_value (cl.literal(i));

	    if (lit_value == 0 ) 
		++val_0_lits;
	    else if (lit_value == 1) 
		++val_1_lits;
	    else 
		++unknown_lits;

	    if (unknown_lits + val_1_lits > (int)_params.cls_deletion.max_unrelevance) {
		mark_cons_deleted(cl);
		DBG1(cout << "Deleting Unrelevant clause: " << cl << endl;);
		break;
	    }
	    if (cl.num_lits() > _params.cls_deletion.max_conf_cls_size && 
		(unknown_lits+val_1_lits > 1) ) { //to make sure it's not generating an implication
				//and it's not an antecedent for other var assignment
		mark_cons_deleted(cl);
		DBG1 (cout << "Deleting Large clause: " << cl << endl;);
		break;
	    }
	}
    }
    if (original_del_cls == num_deleted_clauses()) return;
    //delete the index from variables
    for (vector<CVariable>::iterator itr = variables().begin(); 
	 itr != variables().end(); ++ itr) {
	for (unsigned i=0; i<2; ++i) { //for each phase
	    //delete the lit index from the vars
#ifdef KEEP_LIT_CLAUSES
	    vector<ConsIdx> & lit_cons = (*itr).lit_cons(i);
	    for (vector<ConsIdx>::iterator itr1 = lit_cons.begin();
		 itr1 != lit_cons.end(); ++ itr1) 
		if ( cons(*itr1).status() == DELETED_CONS ) {
		    *itr1 = lit_cons.back();
		    lit_cons.pop_back();
		    -- itr1;
		}
#endif
	    //delete the watched index from the vars
	    vector<CLitPoolElement *> & watched = (*itr).watched(i);
	    for (vector<CLitPoolElement *>::iterator itr1 = watched.begin(); 
		 itr1 != watched.end(); ++itr1) {
		if ( (*itr1)->val() <= 0) {
		    *itr1 = watched.back();
		    watched.pop_back();
		    --itr1;
		}
	    }
	    // Ashish: delete the pbwatched index from the vars
	    vector<CLitPoolElement *> & pbwatched = (*itr).pbwatched(i);
	    for (vector<CLitPoolElement *>::iterator itr1 = pbwatched.begin(); 
		 itr1 != pbwatched.end(); ++itr1) {
		if ( (*itr1)->val() <= 0) {
		    *itr1 = pbwatched.back();
		    pbwatched.pop_back();
		    --itr1;
		}
	    }
	}
    }
    update_var_score();
    DBG1(cout << "Deleting " << num_deleted_clauses() - original_del_cls << " Clause ";
	 cout << " and " << num_deleted_literals() - original_del_lits << " Literals " << endl;);
    DBG2 (dump());
}
//============================================================================================
bool CSolver::time_out(void) 
{
    return (get_cpu_time() - _stats.start_cpu_time> _params.time_limit);
}

void CSolver::adjust_variable_order(vector<int> & lits) 
  // note lits are signed vars, not CLitPoolElements
{
    int n_lits = lits.size();
    for (int i=0; i<n_lits; ++i) {
	int var_idx = lits[i]>>1;
	int var_sign = lits[i]&0x1;
	CVariable & var = variable(var_idx);
	assert (var.value() != UNKNOWN);
	int orig_score = var.score();
	++ var.score(var_sign);
	int new_score = var.score();
	if (orig_score == new_score) 
	    continue;
	int pos = var.var_score_pos();
	int orig_pos = pos;
	assert (_ordered_vars[pos].first == & var);
	assert (_ordered_vars[pos].second == orig_score);
	//next we bubble up the position, because the score can 
	//increase by at most 1, this is valid.

/* this is a linear search */
//  	while (pos > 0) {
//  	    if (_ordered_vars[pos-1].second >= new_score)
//  		break;
//  	    else {
//  		_ordered_vars[pos].second = _ordered_vars[pos-1].second;
//  		_ordered_vars[pos].first = _ordered_vars[pos-1].first;
//  		_ordered_vars[pos].first->set_var_score_pos(pos);
//  		--pos;
//  	    }
//  	}
//  	_stats.total_bubble_move += orig_pos - pos;
//  	_ordered_vars[pos].first = & var;
//  	_ordered_vars[pos].second = new_score;
//  	_ordered_vars[pos].first->set_var_score_pos(pos);


/* this is a binary search */
	int bubble_step = _params.decision.bubble_init_step;
	for (pos = orig_pos ; pos >= 0; pos -= bubble_step)
	    if (_ordered_vars[pos].second >= new_score) break;
	pos += bubble_step;
	for ( bubble_step = bubble_step >> 1; bubble_step > 0; bubble_step = bubble_step >> 1) {
	    if ( pos - bubble_step >= 0) {
		if (_ordered_vars[pos - bubble_step].second < new_score)
		    pos -= bubble_step;
	    }
	}
	//now found the position, do a swap
	_ordered_vars[orig_pos] = _ordered_vars[pos];
	_ordered_vars[orig_pos].first->set_var_score_pos(orig_pos);

	_ordered_vars[pos].first = & var;
	_ordered_vars[pos].second = new_score;
	_ordered_vars[pos].first->set_var_score_pos(pos);

	_stats.total_bubble_move += orig_pos - pos;
    }
    CHECK(
	for (unsigned i=0; i< _ordered_vars.size(); ++i) {
	    assert (_ordered_vars[i].second == _ordered_vars[i].first->score());
	    assert (_ordered_vars[i].first->var_score_pos() == i);
	    assert (i==0 || _ordered_vars[i].second <= _ordered_vars[i-1].second );
	    assert (_ordered_vars[i].first->value() != UNKNOWN || _max_score_pos <= i);
	});
}

void CSolver::decay_variable_score(void) 
{
    unsigned int i, sz;
    for(i=1, sz = variables().size(); i<sz; ++i) {
	CVariable & var = variable(i);
	var.score(0) = var.score(0)/2;
	var.score(1) = var.score(1)/2;
    }
    for (i=0, sz = _ordered_vars.size(); i<sz; ++i) {
	_ordered_vars[i].second = _ordered_vars[i].first->score();
/*  	assert (i==0 || _ordered_vars[i].second <= _ordered_vars[i-1].second );
	assert ( _ordered_vars[i].first->value() != UNKNOWN || 
	!_ordered_vars[i].first->is_branchable() || 
	_max_score_pos <= i); */
    }
}

bool CSolver::decide_next_branch(void)
{
  CHECK(verify_integrity());

  // Ashish: Use the correct implication queue based on symmetry.enable
  if (!_params.symmetry.enable) {
    if (!_implication_queue.empty()) {
      //some hook function did a decision, so skip my own decision making.
      //if the front of implication queue is 0, that means it's finished
      //because var index start from 1, so 2 *vid + sign won't be 0. 
      //else it's a valid decision.
      return (_implication_queue.front().lit != 0);
    }
    //    add_outside_clauses();
    if (_outside_constraint_hook != NULL) {
      _outside_constraint_hook(this);
      if (!_implication_queue.empty())
	return (_implication_queue.front().lit != 0);
    }
  }
  else {   // symmetry enabled
    if (!_implication_pqueue.empty())
      return (_implication_pqueue.top().lit != 0);
    //    add_outside_clauses();
    if (_outside_constraint_hook != NULL) {
      _outside_constraint_hook(this);
      if (!_implication_pqueue.empty())
	return (_implication_pqueue.top().lit != 0);
    }
  }
    
  ++_stats.num_decisions;
  int s_var = 0;
  if (num_free_variables() == 0) //no more free vars
    return false;
  DBG2(
       cout << "Ordered Vars ";
       for (unsigned i=0; i< _ordered_vars.size(); ++i)
       cout << (_ordered_vars[i].first - & variables()[0]) << " ";
       cout << endl;
       );

  CHECK(
	int free = 0;
	for (unsigned i=0; i< _ordered_vars.size(); ++i) {
	  assert (_ordered_vars[i].second == _ordered_vars[i].first->score());
	  assert (_ordered_vars[i].first->var_score_pos() == i);
	  assert (i==0 || _ordered_vars[i].second <= _ordered_vars[i-1].second );
	  assert (_ordered_vars[i].first->value() != UNKNOWN || _max_score_pos <= i);
	  if (_ordered_vars[i].first->value() == UNKNOWN) ++free;
	}
	assert (free == num_free_variables());
	);

  for (unsigned i=_max_score_pos; i<_ordered_vars.size(); ++i) {
    CVariable & var = *_ordered_vars[i].first;
    if (var.value()==UNKNOWN && var.is_branchable()) {
      //move the max score position pointer
      _max_score_pos = i;
      //make some randomness happen
      if (--_stats.current_randomness < _params.decision.base_randomness)
	_stats.current_randomness = _params.decision.base_randomness;

      int randomness = _stats.current_randomness;
      if (randomness >= num_free_variables())
	randomness = num_free_variables() - 1;
      int skip =rand()%(1+randomness);
      int index = i;
      while (skip > 0) {
	++index;
	if (_ordered_vars[index].first->value()==UNKNOWN && 
	    _ordered_vars[index].first->is_branchable())
	  --skip;
      }
      CVariable * ptr = _ordered_vars[index].first;
      assert (ptr->value() == UNKNOWN && ptr->is_branchable());
      int sign = ptr->score(0) > ptr->score(1) ? 0 : 1;
      //	    int sign = (random() > (RAND_MAX >> 1))? 0: 1;
      int var_idx = ptr - &(*variables().begin());
      s_var = var_idx + var_idx + sign;

      //////////////////////////////////////////
      // In progress: testing sign flipping!  //
//       if (_params.symmetry.enable && !_branchlevel_stack.empty() && _branchinfo_vector[_branchlevel_stack.top()].branch_idx == 0) {
//         static bool firsttime = true;
//         if (firsttime) {
//           firsttime = false;
//           cout << "NOTE:: Flipping sign on ad-hoc basis for now!" << endl;
//         }
//         sign = 1 - sign;
//         s_var = var_idx + var_idx + sign;
//       }
      //////////////////////////////////////////

      break;
    }
  }

  assert (s_var >= 2); //there must be a free var somewhere
  ++dlevel();
  if (dlevel() > _stats.max_dlevel) 
    _stats.max_dlevel = dlevel();

  // Ashish
  int vid = s_var >> 1;

  DBG0 (cout << "**Decision " << _stats.num_decisions << " at level " << dlevel() ;
	cout <<": " << s_var << "\ti.e. " << (s_var&0x1?"-":" ") ;
	cout << vid << endl;
	// Ashish
	if (_params.symmetry.enable) {
	  CSymindexMapping* sm = var_findsymindex(vid);
	  if (!sm)
	    cout << "\t[Var " << vid << " does not have any symmetry]" << endl;
	  else if (symmetry_exhausted())
	    cout << "\t[Var " << vid << " was originally symmetric but is not anymore]" << endl;
          else {
	    cout << "\t[Var " << vid << " indexed by (";
	    copy(sm->indices.begin(), sm->indices.end(), ostream_iterator<const int>(cout, " "));
	    cout << ")]" << endl;
	    cout << "\t[Symmetry set for var " << vid << " begins with var indexed by (";
	    for (unsigned ii=0; ii<sm->indices.size(); ii++)
	      cout << idx_symindex_set(sm->indices[ii]) << ' ';
	    cout << ")]" << endl;
	  }
	});
  _implication_id = 0;
  queue_implication(s_var, NULL_CONS, dlevel());
  return true;
}

int CSolver::preprocess(void) 
{
  unsigned int i, sz;
  assert(dlevel() == 0);

  //1. detect all the unused variables
  vector<int> un_used;
  for (i=1, sz=variables().size(); i<sz; ++i) {
    CVariable & v = variable(i);
    if (v.lits_count(0) == 0 && v.lits_count(1) == 0) {
      un_used.push_back(i);
      queue_implication(i+i, NULL_CONS, 0);
      int r;
      if (_params.symmetry.enable)
        r = deduce_sym();
      else
        r = deduce();
      assert (r == NO_CONFLICT);
    }
  }
  if (_params.verbosity>1 && un_used.size() > 0) {
    cout << un_used.size()<< " Variables are defined but not used " << endl;
    if (_params.verbosity > 2) {
      for (unsigned i=0; i< un_used.size(); ++i)
        cout << un_used[i] << " ";
      cout << endl;
    }
  }

  //2. detect all variables with only one phase occuring (i.e. pure literals)
  vector<int> uni_phased;
  for (i=1, sz=variables().size(); i<sz; ++i) {
    CVariable & v = variable(i);
    if (v.value() != UNKNOWN)
      continue;
    if (v.lits_count(0) == 0){ //no positive phased lits.
      queue_implication( i+i+1, NULL_CONS, 0);
      uni_phased.push_back(-i);
    }
    else if (v.lits_count(1) == 0){ //no negative phased lits.
      queue_implication( i+i, NULL_CONS, 0);
      uni_phased.push_back(i);
    }
  }
  if (_params.verbosity>1 && uni_phased.size() > 0) {
    cout << uni_phased.size()<< " Variables only appear in one phase." << endl;
    if (_params.verbosity > 2) {
      for (i=0; i< uni_phased.size(); ++i)
        cout << uni_phased[i] << " ";
      cout <<endl;
    }
  }
  //3. Unit clauses
  for (i=0, sz=clauses().size(); i<(sz<<1); i+=2) {
    CClause & cl = clause(i);
    if (cl.status()!=DELETED_CONS)
      if (cl.num_lits() == 1)  //unit clause
        if (variable(cl.literal(0).var_index()).value() == UNKNOWN)
          queue_implication(cl.literal(0).s_var(), i, 0);
  }

  // 4. Unit pbcons
  for (i=1, sz=pbcons().size(); i<((sz<<1) | 0x01); i+=2) {
    CPBcons & pb = pbcons(i);
    if (pb.status()!=DELETED_CONS)
      if (pb.watch_slack() < 0) {
        // Any lit with coeff more than real slack is unit-implied
        int real_slack = pb.watch_slack() + pb.watch_max();
        CLitPoolElement * ptr = pb.literals();
        while (ptr->is_literal()) {
          if (ptr->is_watched() && pb.coeff(ptr) > real_slack) {
            if (variable(ptr->var_index()).value() == UNKNOWN)
              queue_implication(ptr->s_var(), i, 0);
          }
          ++ptr;
        }
      }
  }

  if((!_params.symmetry.enable && deduce()==CONFLICT) || (_params.symmetry.enable && deduce_sym()==CONFLICT)) {
#ifdef VERIFY_ON
    for (unsigned i=1; i< variables().size(); ++i) {
      if (variable(i).value() != UNKNOWN) {
        assert (variable(i).dlevel() <= 0); 
        int ante = variable(i).antecedent();
        int ante_id = 0;
        if (ante >= 0) {
          ante_id = clause(ante).id();
          verify_out << "VAR: " << i 
                     << " L: " << variable(i).assgn_stack_pos()
                     << " V: " << variable(i).value() 
                     << " A: " << ante_id 
                     << " Lits:";
          for (unsigned j=0; j< clause(ante).num_lits(); ++j)
            verify_out <<" " <<  clause(ante).literal(j).s_var();
          verify_out << endl;
        }
      }
    }
    verify_out << "CONF: " << clause(_conflicts[0]).id() << " ==";
    for (unsigned i=0; i< clause(_conflicts[0]).num_lits(); ++i) {
      int svar = clause(_conflicts[0]).literal(i).s_var();
      verify_out << " " << svar;
    }
#endif
    return CONFLICT;
  }
  //  if (simplify() == CONFLICT) return CONFLICT;
  return NO_CONFLICT;    
}

void CSolver::mark_var_unbranchable(int vid)
{
    
    if (variable(vid).is_branchable()==true) {
	variable(vid).disable_branch();
	if (variable(vid).value()==UNKNOWN)
	    --num_free_variables();
    }
}

void CSolver::mark_var_branchable(int vid)
{
    CVariable & var = variable(vid);
    if (var.is_branchable()==false) {
	var.enable_branch();
	if (var.value() == UNKNOWN) {
	    ++ num_free_variables();
	    if (var.var_score_pos() < _max_score_pos)
		_max_score_pos = var.var_score_pos();
	}
    }
}

ConsIdx CSolver::add_orig_clause (vector<int> & lits, int gid)
{
    ConsIdx cid = add_clause_with_gid(lits, gid);
    if (cid >= 0)
	clause(cid).set_status(ORIGINAL_CONS);
    return cid;
}

// Ashish:
ConsIdx CSolver::add_orig_pbcons (vector<int> & lits, vector<int> & coeffs, int rhs)
{
    ConsIdx pbid = add_pbcons(lits, coeffs, rhs);
    if (pbid >= 0) {
	pbcons(pbid).set_status(ORIGINAL_CONS);
    }
    else {
	_stats.is_mem_out = true;
	_stats.outcome = MEM_OUT;
    }
    return pbid;
}

ConsIdx CSolver::add_clause_with_gid (vector<int> & lits, int gid)
{
    assert (dlevel() >= 0);
    unsigned gflag;
    if (gid == PERMANENT_GID ) 
	gflag = 0;
    else if (gid == VOLATILE_GID)
	gflag = (~0x0);
    else {
	assert (gid <= WORD_WIDTH && gid > 0);
	gflag = (1 << (gid- 1));
    }
    ConsIdx cid = add_clause(lits, gflag);
    if (cid < 0) {
	_stats.is_mem_out = true;
	_stats.outcome = MEM_OUT;
	return cid;
    }
    return cid;
}

ConsIdx CSolver::add_conflict_clause (vector<int> & lits, int gflag)
{
    ConsIdx cid = add_clause(lits, gflag);
    if (cid >= 0) {
	clause(cid).set_status(CONFLICT_CONS);
    }
    else {
	_stats.is_mem_out = true;
	_stats.outcome = MEM_OUT;
    }
    return cid;
}

ConsIdx CSolver::add_symconflict_clause (vector<int> & lits)
{
    ConsIdx cid = add_symclause(lits);
    if (cid >= 0) {
	clause(cid).set_status(CONFLICT_CONS);
    }
    else {
	_stats.is_mem_out = true;
	_stats.outcome = MEM_OUT;
    }
    return cid;
}

void CSolver::real_solve(void)
{
  while(_stats.outcome == UNDETERMINED) {
    run_periodic_functions();
    if (decide_next_branch()) {
      while (deduce()==CONFLICT) {
	int blevel = analyze_conflicts();
	if (blevel < 0) {
	  _stats.outcome = UNSATISFIABLE;
	  return;
	}
      }
    }
    else {
      if(_sat_hook != NULL && _sat_hook(this))
	continue;
      _stats.outcome = SATISFIABLE;
      return;
    }
    if (time_out()) { 
      _stats.outcome = TIME_OUT;
      return; 
    }
    if (_force_terminate) { 
      _stats.outcome = ABORTED;
      return; 
    }
    if (_stats.is_mem_out) {
      _stats.outcome = MEM_OUT;
      return; 
    }	    
  }
}

// Ashish
void CSolver::real_solve_sym(void)
{
  while(_stats.outcome == UNDETERMINED) {
    run_periodic_functions();

    // take care of initial unit props
    if (deduce_sym()==CONFLICT) { 
      _stats.outcome = UNSATISFIABLE;
      return;
    }

    switch (branch_sym()) {
    case SUCCESS:
    case SYMBRANCH_OVER_WITH_IMPLICATION:
      switch (deduce_sym()) {
      case TRIV_CONFLICT:
	assert(_conflicts.back() < 0);
	_conflicts.back() = - _conflicts.back() - 1;    // remove tag
	DBG0(cout << "Trivial conflict at level "
             << _triv_conflict_dl << ": attempting to branch on already set variable"
	     << endl << "Conflict constraint " << _conflicts.back() << " : " 
	     << constraint(_conflicts.back()) << endl);
        analyze_trivial_conflict();
	DBG1( dump_assignment_stack(););
	break;

      case CONFLICT:
	if (!_params.learning.enable) {
	  // no learning; look for only one conflict and explicitly backtrack
	  DBG0(cout << "Conflict in constraint " << _conflicts[0] << " : " << constraint(_conflicts[0]) << endl;);
	  _conflicts.clear();
	  back_track_sym();
	}
	else {
	  // learning enabled; backtracking will happen automatically
	  do {
	    int blevel = analyze_conflicts();
	    if (blevel < 0) {
	      _stats.outcome = UNSATISFIABLE;
	      return;
	    }
	  } while (deduce_sym() == CONFLICT);
	}
	break;
      }
      break;

    case SYMBRANCH_OVER_WITHOUT_IMPLICATION:
      break;

    case FAIL_UNSAT:
      _stats.outcome = UNSATISFIABLE;
      return;

    case FAIL_SAT:
      DBG1(dump_assignment_stack();)
        _stats.outcome = SATISFIABLE;
      return;

    case FAIL_MEM_OUT:
      _stats.outcome = MEM_OUT;
      return;
    }

    if (time_out()) { 
      _stats.outcome = TIME_OUT;
      return; 
    }
    if (_force_terminate) { 
      _stats.outcome = ABORTED;
      return; 
    }
    if (_stats.is_mem_out) {
      _stats.outcome = MEM_OUT;
      return; 
    }	    
  }
}

int CSolver::solve(void)
{

  DBG1(dump());
  if (_stats.outcome == UNDETERMINED) {
    init_solve();
    //preprocess 
    DBG1(dump_assignment_stack(););
    if(preprocess()==CONFLICT) 
      _stats.outcome = UNSATISFIABLE;
    else { //the real search
      if (_params.symmetry.enable)
	real_solve_sym();
      else
	real_solve();
    }
    _stats.finish_cpu_time = get_cpu_time();
  }
  int result = _stats.outcome;
  //    _stats.outcome = UNDETERMINED;
  return result;
}

// Ashish: This method is the heart of symchaff!
SAT_BranchT CSolver::branch_sym(void) {
  DBG1(cout << "Beginning branch_sym at dlevel = " << dlevel() << endl;)
    int i;
  int skip_decision = false;
  CBranchInfo & bi = _branchinfo_vector[branchlevel_next()];

  if (bi.redo_symbranch) {
    assert(_implication_pqueue.empty());
    skip_decision = true;
    int s_var = (bi.vids[0] << 1) + bi.sign;
    bi.clear();
    int dl = branchlevel_next();
    if (dl <= dlevel())
      back_track(dl);
    ++dlevel();
    _implication_id = 0;
    queue_implication(s_var, NULL_CONS, dl);
  }

  if (bi.inprogress() && !(bi.nosymmetry() && _params.learning.enable)) {
    if (bi.branch_point == 0) {
      // finished with this level; need to backtrack
      DBG1(cout << "Explored all branches of SymDecision at level " 
           << branchlevel_next() << endl;);
      if (_branchlevel_stack.empty()) {    // search complete - unsat!
	bi.clear();
	return FAIL_UNSAT;
      }

      // unsplit every inactive symindex set for this symbranch
      // that is different from the branch_idx
      if (!bi.nosymmetry()) {
        vector<int> & var_indices = var_findsymindex(bi.vids[0])->indices;
        for (i=0; i<var_indices.size(); i++)
          if (var_indices[i] != bi.branch_idx_start)    // inactive symindex
            del_symindex_high(var_indices[i]);
      }

      bool implication_queued = false;

      // learn symconflict clause if learning is enabled
      if (_params.learning.enable) {
	if (bi.symconflict_lits.empty() || bi.max_symconflict_dl == 0) {
          // search complete! unsat!!
	  bi.clear();
	  return FAIL_UNSAT;
	}
	vector<int> temp;
	set<int> symconflict_vars;
        int max_symconflict_dl = bi.max_symconflict_dl;
	for (set<int>::iterator itr = bi.symconflict_lits.begin(); itr != bi.symconflict_lits.end(); ++ itr) {
	  temp.push_back(*itr);
	  symconflict_vars.insert(*itr >> 1);
	}

	// if symconflict clause contains a literal and its negation
	// (which could happen if learned clauses imply a conflict
	//  at a previous level), skip symlearning. Otherwise...
	if (temp.size() == symconflict_vars.size()) {
	  // Note: not sure what gflag/gid are; using 0 at the moment!
	  ConsIdx added_cl = add_symconflict_clause(temp);
	  if (added_cl < 0 ) { //memory out.
	    _stats.is_mem_out = true;
	    _conflicts.clear();
	    assert (_implication_pqueue.empty());
	    return FAIL_MEM_OUT;
	  }
	  DBG0 (cout << "*Added symconflict clause " << (added_cl>>1) << " : " << clause(added_cl) << endl;);

	  // Explicit "conflict-directed" backtracking
          // Remove all but the last symbranch which will be removed
          //   shortly
          assert(max_symconflict_dl > 0);
          DBG0(cout << "Conflict-directed backjumping to (roughly) dlevel "
               << max_symconflict_dl - 1 << endl;)
            while (_branchlevel_stack.top() > max_symconflict_dl)
              back_track_sym(true, true, false);
          int sym_dl = _branchlevel_stack.top();

          // find out which variable needs to be flipped
	  CBranchInfo & new_bi = _branchinfo_vector[sym_dl];
	  int flip_dl = sym_dl + new_bi.branch_size();   // initialize
	  for (i=0; i<temp.size(); ++i) {
	    int var_dl = variable(temp[i] >> 1).dlevel();
	    if (var_dl < flip_dl && var_dl > sym_dl+new_bi.branch_point-1)
	      flip_dl = var_dl;
	  }
	  // update branch_info so that redundant branches are skipped
          if (flip_dl - sym_dl + 1 > new_bi.branch_point + 1) {
            new_bi.branch_idx_to_flip = flip_dl - sym_dl + 1;
            DBG1(cout << "Setting branch_idx_to_flip = " 
                 << new_bi.branch_idx_to_flip << endl;);
          }

          // remove the remaining symbranch
          back_track_sym();
          if (dlevel() >= max_symconflict_dl)
            back_track(max_symconflict_dl);

	  // if there was a unique literal with dlevel >= last sym_dl,
	  // which should now be unassigned, queue it for implication
	  int next_max_symconflict_dl = 0;
	  int num_lastblock_symdecision_lits = 0;
	  int unique_lastblock_symdecision_lit = -1;
	  for (i=0; i<temp.size(); i++) {
	    if (variable(temp[i]>>1).value() == UNKNOWN) {
	      num_lastblock_symdecision_lits ++;
	      unique_lastblock_symdecision_lit = temp[i];
	    }
	    else {
	      int dl = variable(temp[i]>>1).dlevel();
	      if (dl > next_max_symconflict_dl)
		next_max_symconflict_dl = dl;
	    }
	  }
	  if (num_lastblock_symdecision_lits == 1) {
	    implication_queued = true;
	    queue_implication(unique_lastblock_symdecision_lit, added_cl, next_max_symconflict_dl);
	  }
	}
	else {
	  DBG1(cout << "Proposed symconflict clause contains a literal and its negation." << endl << "Skipping symlearning." << endl;);
	  back_track_sym();
	}
      }
      else
	back_track_sym();

      return implication_queued ? SYMBRANCH_OVER_WITH_IMPLICATION : SYMBRANCH_OVER_WITHOUT_IMPLICATION;
    }

    // otherwise proceed with the level in progress
    if (bi.branch_idx_to_flip != -1) {
      if (bi.branch_idx_to_flip < bi.branch_point + 1)
        bi.branch_idx_to_flip = bi.branch_size() + 1;

      DBG0(if (bi.branch_idx_to_flip > bi.branch_point + 1)
           cout << "Skipping symbranch_points " << bi.branch_point+1
           << " to " << bi.branch_idx_to_flip-1 << endl;)

        bi.branch_point = bi.branch_idx_to_flip;
      bi.branch_idx_to_flip = -1;      // reset

      if (bi.branch_point > bi.branch_size()) {
	assert(bi.branch_point == bi.branch_size()+1);
	// finish off this branch quickly using recursion by pretending
        // that all branch_points up to bi.branch_size() have already
        // been handled
	bi.branch_point = bi.branch_size();
        if (dlevel() >= branchlevel_next())
          back_track(branchlevel_next());
	return branch_sym();
      }
    }
    else
      bi.branch_point = (bi.branch_point + 1) % (bi.branch_size() + 1);

    _branchlevel_stack.push(branchlevel_next());
    _implication_id = 0;

    if (bi.nosymmetry()) {
      int vid = bi.vids[0];
      int s_var = (vid << 1) + (1 - bi.sign);
      ++ dlevel();
      if (dlevel() > _stats.max_dlevel)
	_stats.max_dlevel = dlevel();
      queue_implication(s_var, NULL_CONS, dlevel());
      ++ _stats.num_decisions;
      DBG0 (cout << "**Decision " << _stats.num_decisions << " at level " << dlevel() ;
	    cout <<": " << s_var << "\ti.e. " << (s_var&0x1?"-":" ") ;
	    cout << vid << endl;);
      return SUCCESS;
    }

    ++ _stats.num_symdecisions;
    DBG0 (cout << "**SymDecision " << _stats.num_symdecisions 
          << " at level " << _branchlevel_stack.top() 
          << " with new branches starting at level " << dlevel()+1 << endl;);

    int num_old_branches = dlevel()+1 - _branchlevel_stack.top();
    if (num_old_branches >= bi.branch_point) {
      // extra decisions, need to backtrack a bit
      if (bi.branch_point == 0) {
        if (num_old_branches > 0) {
          back_track(_branchlevel_stack.top());
          num_old_branches = 0;
        }
      }
      else {
        back_track(_branchlevel_stack.top() + bi.branch_point - 1);
        num_old_branches = bi.branch_point - 1;
      }
    }

    for (i=0; i<num_old_branches; i++) {
      int vid   = bi.vids[i];
      int s_var = (vid << 1) + (1 - bi.sign);
      DBG0 (
            cout <<"\t: " << s_var << "\ti.e. " << (s_var&0x1?"-":" ")
            << vid << endl;
            );
    }
    for (; i<bi.branch_point; i++) {
      int vid   = bi.vids[i];
      int s_var = (vid << 1) + (1 - bi.sign);
      ++ dlevel();
      queue_implication(s_var, NULL_CONS, dlevel());
      DBG0 (
            cout <<"\t: " << s_var << "\ti.e. " << (s_var&0x1?"-":" ")
            << vid << endl;
            );
    }
    for (; i<bi.branch_size(); i++) {
      int vid   = bi.vids[i];
      int s_var = (vid << 1) + bi.sign;
      ++ dlevel();
      queue_implication(s_var, NULL_CONS, dlevel());
      DBG0 (
	    cout <<"\t: " << s_var << "\ti.e. " << (s_var&0x1?"-":" ")
	    << vid << endl;
	    );
    }
    if (dlevel() > _stats.max_dlevel)    // update stats
      _stats.max_dlevel = dlevel();

    // update active symindex set
    if (bi.branch_point > 0 && bi.branch_point < bi.branch_size()) {     
      // create a new symindex split
      // no splits if b_pt is 0 or bi.branch_size -> all vars are symmetric
      DBG0(cout << "Splitting symindex sets at branch point " 
	   << bi.branch_point
	   << ", i.e. symindex " << (bi.branch_idx_start + bi.branch_point - 1) 
	   << ", for symmetry branch over vars [" 
	   << bi.vids.front() << ',' << bi.vids.back() << ']' << endl;);
      add_symindex_high(bi.branch_idx_start + bi.branch_point - 1);
    }
    DBG0(cout << endl;);

    return SUCCESS;
  }
  else {
    // No decision in progress or {no symmetry and nolearning disabled}
    // If no decision in progress, must of course create a new one
    // If no symmetry (and nolearning disabled), do what zChaff normally
    //   does, i.e. do not take the right branch, instead just pick
    //   another variable to branch upon and rely on 1UIP scheme to
    //   take care of efficiency. The branch with NOT be complete, so
    //   can't do this when nolearning is enabled.

    if (!skip_decision) {
      // Choose a new variable to branch on
      // Use zchaff's default branching scheme augmented with symmetry
      bool success = decide_next_branch();
      if (!success)     // no more free variables; solution found!
        return FAIL_SAT;
    }

    assert(branchlevel_next() == dlevel());
    _branchlevel_stack.push(dlevel());   // store dl for backtrack

    int s_var = _implication_pqueue.top().lit;
    CSymindexMapping* sm = var_findsymindex(s_var >> 1);
    bi.clear();
    bi.sign = s_var & 0x1;
    if (!sm) {   // no symmetry involved
      bi.vids.push_back(s_var >> 1);
      DBG0(cout << endl;);
      return SUCCESS;
    }

    int init_branch_idx;
    if (_params.order.enable)
      init_branch_idx = varclass_idx_order(sm->varclass_idx).front() - 1;
    else
      init_branch_idx  = 0;   // choose an index, say first, for symmetry branching
    //init_branch_idx  = sm->indices.size() - 1;   // choose an index, say last, for symmetry branching

    bi.branch_idx = init_branch_idx;
    int branch_size = size_symindex_set(sm->indices[bi.branch_idx]);
    if (!_params.order.enable && !symmetry_exhausted() 
        && branch_size == 1 && sm->indices.size() > 1) {
      do {
        // try to choose a better branch_idx
        bi.branch_idx = (bi.branch_idx + 1) % sm->indices.size();
        branch_size = size_symindex_set(sm->indices[bi.branch_idx]);
      } while (branch_size == 1 && bi.branch_idx != init_branch_idx);
    }

    if (branch_size == 1) {   // no symmetry left
      bi.branch_idx = -1;
      // store branch_idx_start in case the symindex set expands later
      bi.branch_idx_start = sm->indices[init_branch_idx];
      bi.vids.push_back(s_var >> 1);
      return SUCCESS;
    }

    ///////////////////////////////////////////////////////////////
    // The following code snippet used for develomental testing. //
    // Please don't uncomment it!                                //
//     bi.sign = 1 - bi.sign;
//     static bool firsttime = true;
//     if (firsttime) {
//       firsttime = false;
//       cerr << "WARNING: flipping sign for symbranches" << endl;
//     }
    ///////////////////////////////////////////////////////////////

    DBG0(cout << "        [Using branch index " << bi.branch_idx+1 
         << " for symmetry]" << endl;)

    vector<int> var_indices(sm->indices);

    // First move every inactive symindex of this var to the first
    //   index in that symindex set
    // Then split every inactive symindex of this variable
    //   once at the start of this symbranch
    int first_of_branch_idx = idx_symindex_set(var_indices[bi.branch_idx]);
    for (i=0; i<sm->indices.size(); i++) {
      if (i != bi.branch_idx) {            // inactive symindex
        int this_idx = var_indices[i];
        int first_of_this_idx = idx_symindex_set(this_idx);
        if (first_of_this_idx != first_of_branch_idx) {
          // a different symindex set than branch_idx
          var_indices[i] = first_of_this_idx;
          DBG0(cout 
               << "Splitting symindex set at symindex "
               << first_of_this_idx << endl;)
            add_symindex_high(first_of_this_idx);
        }
        else if (this_idx != var_indices[bi.branch_idx]) {
          // the same symindex set as branch_idx but not the same index
          ++first_of_branch_idx;
          var_indices[i] = first_of_this_idx;
          DBG0(cout
               << "Splitting symindex set at symindex "
               << first_of_this_idx << endl;)
            add_symindex_high(first_of_this_idx);
        }
        else {
          // the same symindex set as branch_idx and in fact the same index
          //   -> mark this index so that it can later be set as 
          //      first_of_branch_idx
          var_indices[i] = -1;    // -1 is used as a marker
        }
      }
    }
    // Point var_indices[branch_idx] as well as all other marked
    // indices to first_of_branch_idx
    for (i=0; i<var_indices.size(); i++)
      if (var_indices[i] == -1)
        var_indices[i] = first_of_branch_idx;
    var_indices[bi.branch_idx] = first_of_branch_idx;
    // Recompute branch_size in case there is a split
    branch_size = size_symindex_set(first_of_branch_idx);

    // Extend the assignment to other vars in the symmetry set
    // First remove the assigned var to maintain good order;
    // it will be inserted back soon in the correct position
    _implication_pqueue.pop();
    --dlevel();
    CVarclass & vc = varclass(sm->varclass_idx);
    bi.branch_idx_start = first_of_branch_idx;

    _implication_id = 0;

    -- _stats.num_decisions;
    ++ _stats.num_symdecisions;
    DBG0 (cout << "Suggested decision replaced with symmetry branching -" << endl;
	  cout << "**SymDecision " << _stats.num_symdecisions 
          << " at level " << dlevel()+1 << endl;);

    for (i=0; i<branch_size; i++) {
      int next_idx = first_of_branch_idx + i;
      if (i > 0)
        for (int j=0; j<var_indices.size(); j++)
          if (var_indices[j] == next_idx - 1)
            var_indices[j] = next_idx;
      int vid   = vc.locate_var(var_indices);
      int s_var = (vid << 1) + bi.sign;
      if (i == 0) {
        s_var = (vid << 1) + (1 - bi.sign);
        DBG0(cout << "Splitting symindex set at symindex " 
             << first_of_branch_idx << endl;)
          add_symindex_high(first_of_branch_idx);
      }
      bi.vids.push_back(vid);    // store branch variables in branchinfo
      ++dlevel();
      queue_implication(s_var, NULL_CONS, dlevel());
      DBG0 (
	    cout <<"\t: " << s_var << "\ti.e. " << (s_var&0x1?"-":" ")
	    << vid << endl;
	    );
    }

    if (dlevel() > _stats.max_dlevel)    // update stats
      _stats.max_dlevel = dlevel();
    DBG0(cout << endl;);

    return SUCCESS;
  }
}

void CSolver::back_track(int blevel)
{
  //  DBG1(cout << "Back track to " << blevel <<" ,currently in "<< dlevel() << " level" << endl;);
  // Ashish: I believe they mean the following:
  DBG1(cout << "Back track level " << blevel <<", currently in level "<< dlevel() << endl;);
  DBG2 (dump());
  CHECK(verify_integrity());
  assert(blevel <= dlevel());
  for (int i=dlevel(); i>= blevel; --i) {
    vector<int> & assignments = *_assignment_stack[i];
    for (int j=assignments.size()-1 ; j>=0; --j)
      unset_var_value(assignments[j]>>1);
    assignments.clear();
  }
  dlevel() = blevel - 1;
  if (dlevel() < 0 ) 
    dlevel() = 0;
  ++_stats.num_backtracks;
  DBG2 (dump());
  CHECK(verify_integrity());
}

void CSolver::back_track_sym(bool complete_backtrack, 
                             bool discard_own_symconf_lits,
                             bool discard_last_symconf_lits)
{
  int sym_dl = _branchlevel_stack.top();
  DBG1(cout << "Symbacktrack: sym_dl = " << sym_dl << endl;
       cout << "Beginning symbacktrack at dlevel = " << dlevel() << endl;)
  _branchlevel_stack.pop();
  CBranchInfo & bi = _branchinfo_vector[sym_dl];

  // delete symindex splits of bi
  if (bi.branch_point > 0 && bi.branch_point < bi.branch_size()) { 
    DBG0(cout << "Deleting symindex sets split at branch point "
         << bi.branch_point
         << ", i.e. symindex " << (bi.branch_idx_start + bi.branch_point - 1) 
         << ",\n  for symmetry branch over vars [" 
         << bi.vids.front() << ',' << bi.vids.back() << ']' << endl;);
    del_symindex_high(bi.branch_idx_start + bi.branch_point - 1);
  }

  if (discard_own_symconf_lits)
    bi.symconflict_lits.clear();

  if (_branchlevel_last > sym_dl) {
    // two backtracks happened without a symbranch in between
    // i.e. fast_backjumping is happening
    //  -> transfer symconflict_lits to lower dlevel and clear bi_last
    CBranchInfo & bi_last = _branchinfo_vector[_branchlevel_last];
    if (!discard_last_symconf_lits) {
      for (set<int>::iterator itr = bi_last.symconflict_lits.begin(); 
           itr != bi_last.symconflict_lits.end(); ++ itr) {
        int dl = variable(*itr >> 1).dlevel();
        if (dl < sym_dl)
          bi.add_symconflict_lit(*itr, dl);
      }
    }
    bi_last.clear();
  }
  _branchlevel_last = sym_dl;

  // try to backtrack only as far as needed in the symbranch
  int blevel = sym_dl + (bi.branch_point % bi.branch_size());
  if (complete_backtrack || dlevel() < blevel)
      back_track(sym_dl);
  else
    back_track(blevel);
  DBG1(cout << "Finishing symbacktrack at dlevel = " << dlevel() << endl;)
}

SAT_DeductionT CSolver::deduce(void) 
{
  while (!_implication_queue.empty() && _conflicts.size()==0) {
    DBG2(dump_implication_queue(););
    const CImplication & imp = _implication_queue.front();
    int lit = imp.lit;
    int vid = lit>>1;
    int dl = imp.dlevel;

    ConsIdx cons = imp.antecedent;
    _implication_queue.pop();
    CVariable & var = variable(vid);
    if ( var.value() == UNKNOWN) { // an implication
      set_var_value(vid, !(lit&0x1), cons, dl);
      var.assgn_stack_pos() = _assignment_stack[dl]->size();
      _assignment_stack[dl]->push_back(lit);
      CHECK(verify_integrity());
    }
    else if (var.value() == (unsigned)(lit&0x1) ) { //a conflict
      //note: literal & 0x1 == 1 means the literal is in negative phase
      //when a conflict occure at not current dlevel, we need to backtrack
      //to resolve the problem.
      //conflict analysis will only work if the conflict occure at 
      //the top level (current dlevel)
      if (dl == dlevel()) {
	_conflicts.push_back(cons);
	continue;
      }
      else {
	assert (dl < dlevel());
	int orig_dl = var.dlevel();
	vector<CImplication> still_valid;
	if (orig_dl < dl) //this means other implication will take care of it.
          // Ashish: ignore this implication and don't assign the variable
	  continue;
	if (orig_dl > dl) {
	  //in this case, backtrack to orig_dl level, and do the implication need to be done
          // Ashish: backtrack and assign the variable at the lower dlevel
	  while (!_implication_queue.empty()) {
	    if (_implication_queue.front().dlevel < orig_dl ) 
	      still_valid.push_back(_implication_queue.front());
	    _implication_queue.pop();
	  }
	  for (unsigned i=0; i< still_valid.size(); ++i) 
	    _implication_queue.push(still_valid[i]);
	  back_track(orig_dl);
	  set_var_value(vid, !(lit&0x1), cons, dl);
	  var.assgn_stack_pos() = _assignment_stack[dl]->size();
	  _assignment_stack[dl]->push_back(lit);
	}
	else { 
	  assert (orig_dl == dl);
	  //in this case, backtrack to that dlevel and it's a conflict clause
          // Ashish: got a conflict!
	  back_track(orig_dl + 1);
	  _conflicts.push_back(cons);
	}
      }
    }
    else {
      //so the variable have been assigned before
      if (var.dlevel() <= dl) continue;
      // Ashish: if the variable was assigned at a higher level,
      //         assign it now at this lower level instead
      int old_dl = var.dlevel();
      int old_pos = var.assgn_stack_pos();
      assert ((*_assignment_stack[old_dl])[old_pos] == lit);
      (*_assignment_stack[old_dl])[old_pos] = 0; //insert a bubble
      unset_var_value(vid);
      set_var_value(vid, !(lit& 0x1), cons, dl);
      var.assgn_stack_pos() = _assignment_stack[dl]->size();
      _assignment_stack[dl]->push_back(lit);
    }
  }
  //if loop exited because of a conflict, we need to clean implication queue
  while(!_implication_queue.empty()) 
    _implication_queue.pop();
  return (_conflicts.size()?CONFLICT:NO_CONFLICT);
}

// Ashish
SAT_DeductionT CSolver::deduce_sym(void) 
{
  // Copy of deduce(), with minor modifications for symmetry stuff
  while (!_implication_pqueue.empty() && _conflicts.size()==0) {
    DBG2(dump_implication_pqueue(););
    const CImplication & imp = _implication_pqueue.top();
    int lit = imp.lit;
    int vid = lit>>1;
    int dl = imp.dlevel;

    // Ashish: because of sym-backtracks, dl might be more than dlevel(),
    //         in which case the implication should be ignored
    if (dl > dlevel()) {
      _implication_pqueue.pop();
      continue;
    }

    ConsIdx cons = imp.antecedent;
    _implication_pqueue.pop();
    CVariable & var = variable(vid);
    if ( var.value() == UNKNOWN) { // an implication
      set_var_value(vid, !(lit&0x1), cons, dl);
      var.assgn_stack_pos() = _assignment_stack[dl]->size();
      _assignment_stack[dl]->push_back(lit);
      CHECK(verify_integrity());
    }
    else if (var.value() == (unsigned)(lit&0x1) ) { //a conflict
      //note: literal & 0x1 == 1 means the literal is in negative phase

      if (cons == -1) {                   
        // It's a decision var => trivial conflict!  
        //   Note: such conflicts don't happen in normal
        //   zChaff. However, with multiway branching //
        //   (i.e. branch_sym()), we need to take this into
        //   account. It // appears to be easier to do it here rather
        //   than in // branch_sym.

	// first remove all other conflicts
	_conflicts.clear();

	cons = - var.antecedent() - 1;    // Use the other antecedent for lit
        _conflicts.push_back(cons);       //   with negative sign - 1 as a tag

        if (cons >= 0) {
          // This should never happen!
          cout << "zchaff_solver.cpp: ERROR 15 in CSolver::deduce_sym()\nAborting!\n";
          dump_assignment_stack();
          exit(15);
        }
        assert(cons < 0);

	// clean up implication pqueue
	while(!_implication_pqueue.empty()) 
	  _implication_pqueue.pop();

        _triv_conflict_dl = dl;
	return TRIV_CONFLICT;
      }

      //when a conflict occure at not current dlevel, we need to backtrack
      //to resolve the problem.
      //conflict analysis will only work if the conflict occure at 
      //the top level (current dlevel)
      if (dl == dlevel()) {
	_conflicts.push_back(cons);
      }
      else {
	assert (dl < dlevel());
	int orig_dl = var.dlevel();

	vector<CImplication> still_valid;
	if (orig_dl < dl) //this means other implication will take care of it.
          // Ashish: ignore this implication and don't assign the variable
	  continue;
	if (orig_dl > dl) {
	  //in this case, backtrack to orig_dl level, and do the implication need to be done

          // Ashish: backtrack and assign the variable at the lower dlevel.
          //         I believe they want to backtrack orig_dl, not
          //         backtrack to orig_dl; it's OK to simply
          //         sym_backtrack // that symbranch and then
          //         backtrack orig_dl
	  while (!_implication_pqueue.empty()
                 && _implication_pqueue.top().dlevel < orig_dl ) {
            still_valid.push_back(_implication_pqueue.top());
	    _implication_pqueue.pop();
	  }
          // the queue is now either empty or contains only implications
          // at dlevel >= orig_dl
	  while (!_implication_pqueue.empty())
	    _implication_pqueue.pop();
	  for (unsigned i=0; i< still_valid.size(); ++i) 
	    _implication_pqueue.push(still_valid[i]);

          while (orig_dl < _branchlevel_stack.top())
            back_track_sym(true, true, true);
          // orig_dl is now in the current symbranch or the next
          // the (that has already been backtracked); compute var's
          // original sym_dl
          int orig_sym_dl = _branchlevel_stack.top();
          if (orig_dl >= branchlevel_next())
            orig_sym_dl = branchlevel_next();
          if (dl < orig_sym_dl)
            _branchinfo_vector[orig_sym_dl].add_symconflict_lit(lit, dl);
          if (orig_dl < branchlevel_next())
            back_track_sym();
          if (orig_dl <= dlevel())
            back_track(orig_dl);

          // If dl and orig_dl have the same sym_dl then the above
          // back_track_sym() might reduce dlevel() below dl. If
          // so, don't set vid at dl
          if (dlevel() >= dl) {
            set_var_value(vid, !(lit&0x1), cons, dl);
            var.assgn_stack_pos() = _assignment_stack[dl]->size();
            _assignment_stack[dl]->push_back(lit);
          }
	}
	else { 
	  assert (orig_dl == dl);
	  //in this case, backtrack to that dlevel and it's a conflict clause
          // Ashish: got a conflict! Don't remove orig_dl yet; it will
          //         be handled by conflict analysis
          while (!_branchlevel_stack.empty()
                 && orig_dl < _branchlevel_stack.top())
            back_track_sym(true, true, true);
          if (orig_dl+1 <= dlevel())
            back_track(orig_dl+1);
          else
            assert(orig_dl+1 == dlevel()+1);
	  _conflicts.push_back(cons);
	}
      }
    }
    else {
      //so the variable have been assigned before
      // Ashish: if the variable was assigned at a higher level,
      //         assign it now at this lower level instead
      if (var.dlevel() <= dl) continue;
      int old_dl = var.dlevel();
      int old_pos = var.assgn_stack_pos();
      assert ((*_assignment_stack[old_dl])[old_pos] == lit);
      (*_assignment_stack[old_dl])[old_pos] = 0; //insert a bubble
      unset_var_value(vid);
      set_var_value(vid, !(lit& 0x1), cons, dl);
      var.assgn_stack_pos() = _assignment_stack[dl]->size();
      _assignment_stack[dl]->push_back(lit);
    }
  }
  //if loop exited because of a conflict, we need to clean implication queue
  while(!_implication_pqueue.empty()) 
    _implication_pqueue.pop();
  return (_conflicts.size()?CONFLICT:NO_CONFLICT);
}


void CSolver::verify_integrity(void) 
{	
    for (unsigned i=1; i< variables().size(); ++ i) {
	if (variable(i).value() != UNKNOWN) {
	    int pos = variable(i).assgn_stack_pos();
	    int value = variable(i).value();
	    int dlevel = variable(i).dlevel();
	    assert ((*_assignment_stack[dlevel])[pos] == (int) (i+i+1-value));
	}
    }
    for (vector<CClause>::iterator itr = clauses().begin(); itr != clauses().end(); ++itr) {
	if (itr->status() == DELETED_CONS)
	    continue;
	CClause & cl = *itr;
	int num_0 = 0;
	int num_1 = 0;
	int num_unknown = 0;
	int watched[2]; 
	int watch_index = 0;
	watched[1] = watched[0] = 0;
	for (unsigned j=0; j< cl.num_lits(); ++j) {
	    CLitPoolElement lit = cl.literal(j);
	    int vid = lit.var_index();
	    if (variable(vid).value() == UNKNOWN)
		++ num_unknown;
	    else {
		if (literal_value(lit) == 0)
		    ++ num_0;
		else 
		    ++ num_1;
	    }
	    if (lit.is_watched()) {
		watched[watch_index] = lit.s_var();
		++watch_index;
	    }
	}
	if (watch_index==0) {
	    assert ( cl.num_lits() == 1);
	    continue;
	}
	assert (watch_index == 2);
	for (unsigned j=0; j< cl.num_lits(); ++j) {
	    CLitPoolElement lit = cl.literal(j);
	    int vid1 = (watched[0]>>1);
	    if (variable(vid1).value() == (unsigned)(watched[0] & 0x1)) {
		if (!lit.is_watched()) {
		    assert (literal_value(lit) == 0);
		    assert (variable(lit.var_index()).dlevel() <= variable(vid1).dlevel());
		}
	    }
	    int vid2 = (watched[1]>>1);
	    if (variable(vid2).value() == (unsigned)(watched[1] & 0x1)) {
		if (!lit.is_watched()) {
		    assert (literal_value(lit) == 0);
		    assert (variable(lit.var_index()).dlevel() <= variable(vid1).dlevel());
		}
	    }
	}
    }
}

void CSolver::mark_vars_of_dl(vector<int> & lits, int dl)
//given a vector of lits (all are "in_new_clause", which is generated by
//last round of mark_vars_at_level, mark_vars_of_dl will mark those which dlevel = dl,
//and put the rest into _conflict_lits
{
    assert (_num_marked == 0);
    DBG1(
	cout << "The Lits involved are : " << endl;
	for (unsigned i=0; i< lits.size(); ++i) {
	    cout << "V: " << (lits[i] & 0x1 ? "-":"+") << (lits[i] >> 1) << ": " 
		 << variable(lits[i]>>1);
	}
	cout << endl << endl;
	);
    
    for (vector<int>::iterator itr = lits.begin(); itr != lits.end(); ++ itr) {
	int v = (*itr)>>1;
	assert (variable(v).new_cl_phase() != UNKNOWN);
	if (variable(v).dlevel() == dl) {
	    -- _num_in_new_cl;
	    variable(v).set_new_cl_phase(UNKNOWN);
	    assert (!variable(v).is_marked());
	    variable(v).set_marked();
	    ++ _num_marked;
	}
	else {
	    assert (variable(v).dlevel() < dl);
            _conflict_lits.push_back((*itr));
	}
    }
}

void CSolver::mark_vars_at_level(ConsIdx idx, int var_idx, int dl)
{
  assert (_resolvents.empty() || var_idx != -1);
#ifdef VERIFY_ON	
  _resolvents.push_back(constraint(idx).id());
#endif
  for (CLitPoolElement * itr = constraint(idx).literals(); (*itr).val() > 0 ; ++ itr) {
    int v = (*itr).var_index();
    if (v == var_idx) 
      continue;
    else if (_params.pb.enable && (idx & 0x01) && literal_value(*itr) != 0)
      // Ashish: process only those vars that are assigned false
      // For Boolean clauses, this is practically irrelevant
      continue;
    else if (variable(v).dlevel() == dl) {
      if (!variable(v).is_marked()) {
	variable(v).set_marked();
	++ _num_marked;
      }
    }
    else {
      assert (variable(v).dlevel() < dl);
      if (variable(v).new_cl_phase() == UNKNOWN){ //it's not in the new cl
        ++ _num_in_new_cl;
        variable(v).set_new_cl_phase((*itr).var_sign());
        _conflict_lits.push_back((*itr).s_var());
      }
      else //if this variable is already in the new clause, it must have the same phase 
        assert(variable(v).new_cl_phase() == (*itr).var_sign());
    }
  }
}

int CSolver::analyze_conflicts(void) {
  assert (_conflicts.size() > 0);
  assert(_conflict_lits.size() == 0); 
  assert (_implication_queue.empty());
  assert (_implication_pqueue.empty());    // Ashish
  assert (_num_marked == 0);
  DBG1(dump_assignment_stack());
  CHECK(verify_integrity());

  if (dlevel() == 0){ //already at level 0. Conflict means unsat.
#ifdef VERIFY_ON
    for (unsigned i=1; i< variables().size(); ++i) {
      if (variable(i).value() != UNKNOWN) {
	assert (variable(i).dlevel() <= 0); 
        int ante = variable(i).antecedent();
	ConsIdx ante_id = 0;
	if (ante >= 0) {
	  ante_id = cons(ante).id();
	  verify_out << "VAR: " << i 
		     << " L: " << variable(i).assgn_stack_pos()
		     << " V: " << variable(i).value() 
		     << " A: " << ((ante_id & 0x01)?"PB ":"CL ") << (ante_id>>1)
		     << " Lits:";
	  for (unsigned j=0; j< cons(ante).num_lits(); ++j)
	    verify_out << " " << cons(ante).literal(j).s_var();
	  verify_out << endl;
	}
      }
    }
    verify_out << "CONF: " << constraint(_conflicts[0]).id() << " ==";
    for (unsigned i=0; i< constraint(_conflicts[0]).num_lits(); ++i) {
      int svar = constraint(_conflicts[0]).literal(i).s_var();
      verify_out << " " << svar;
    }
#endif
    _conflicts.clear();
    back_track(0);
    return -1;
  }

  int result = conflict_analysis_firstUIP();
  //int result = conflict_analysis_decisions_only();
  DBG1( dump_assignment_stack(););
  return result;
}

// Ashish
void CSolver::analyze_trivial_conflict(void) {
  // For symmetry learning, store literals involved in conflict
  //   that have dlevel less than the last symbranch level
  // Also set branch_idx_to_flip for efficient symbranching

  int sym_dl = _branchlevel_stack.top();
  CBranchInfo & bi = _branchinfo_vector[sym_dl];

  int flip_dl = sym_dl + bi.branch_size();   // initialize

  CConstraint & cons = constraint(_conflicts.back());
  unsigned i;
  vector<int> involved_lits;
  for (i=0; i < cons.num_lits(); ++i) {
    int svar = cons.literal(i).s_var();
    ++ _num_marked;
    variable(svar>>1).set_marked();
    involved_lits.push_back(svar);
  }

  // Compute decision literals that need to go in the symconflict clause
  set<ConsIdx> visited_ante;  // Pbcons may otherwise be visited more
                              // than once if it implies multiple variables
  for (i=0; i<involved_lits.size(); ++i) {
    int vid = involved_lits[i] >> 1;
    int vid_dl = variable(vid).dlevel();
    ConsIdx ante_cons = variable(vid).get_antecedent();
    if (ante_cons != NULL_CONS) {
      // it's not a decision variable -> process antecedent
      if (_params.pb.enable) {
        if (visited_ante.find(ante_cons) != visited_ante.end())
          continue; // already processed; 
        // this check is unnecessary for Boolean clauses
        else
          visited_ante.insert(ante_cons);
      }
      CConstraint & c = constraint(variable(vid).get_antecedent());
      for (unsigned j=0; j< c.num_lits(); ++j) {
        int lit = c.literal(j).s_var();
        if (variable(lit>>1).is_marked()==false) {
          if (_params.pb.enable && literal_value(c.literal(j)) != 0)
            // Store only those vars that are assigned false
            // For Boolean clauses, all lits are false
            continue;
          ++ _num_marked;
          variable(lit>>1).set_marked();
          involved_lits.push_back(lit);
        }
      }
    }
    else if (vid_dl < sym_dl) {
      // it's a decision variable of a previous symbranch
      if (_params.learning.enable)
        // for now, storing the "decision clause" as a symconflict clause
        bi.add_symconflict_lit(involved_lits[i], vid_dl);
    }
    else {
      // it's a decision variable of the current symbranch
      // Note: >= is used here in the second condition as opposed 
      //       to > elsewhere because of the later check to see if
      //       flip_dl should be set to _triv_conflict_dl
      if (vid_dl < flip_dl && vid_dl >= sym_dl+bi.branch_point-1)
        flip_dl = vid_dl; 
    }
  }

  visited_ante.clear();

  // update branch_info so that redundant branches are skipped
  if (flip_dl == sym_dl + bi.branch_size())
    // No previous variable found for flipping
    // Must flip the variable at _triv_conflict_dl
    if (_triv_conflict_dl > sym_dl + bi.branch_point - 1)
      // otherwise must skip branches till the end
      flip_dl = _triv_conflict_dl;
  _triv_conflict_dl = -1;   // reset
  if (flip_dl - sym_dl + 1 > bi.branch_point + 1) {
    bi.branch_idx_to_flip = flip_dl - sym_dl + 1;
    DBG1(cout << "Setting branch_idx_to_flip = " 
         << bi.branch_idx_to_flip << endl;);
  }

  // clear all marks
  for (vector<int>::iterator itr = involved_lits.begin(); itr != involved_lits.end(); ++itr) {
    int svar = *itr;
    assert (variable(svar>>1).is_marked());
    -- _num_marked;
    variable(svar>>1).clear_marked();
  }
  assert (_num_marked == 0);

  _conflicts.clear();
  back_track_sym();
}


//when all the literals involved are in _conflict_lits
//call this function to finish the adding clause and backtrack
int CSolver::finish_add_conf_clause(int gflag)
{
  unsigned int i;
  int back_dl = 0;
  int max_dl = 0; // Ashish
  int unit_lit = -1;
  vector<int> involved_lits;  // Ashish

  ConsIdx added_cl = add_conflict_clause(_conflict_lits, gflag);
  if (added_cl < 0 ) { //memory out.
    _stats.is_mem_out = true;
    _conflicts.clear();
    assert (_implication_queue.empty());
    assert (_implication_pqueue.empty());   // Ashish
    return 1; 
  }
    
#ifdef VERIFY_ON
  verify_out << "CL: " <<  (clause(added_cl).id()>>1) << " <=";
  for (unsigned i=0; i< _resolvents.size(); ++i)
    verify_out << " " <<  _resolvents[i];
  verify_out << endl;
  _resolvents.clear();
#endif

  adjust_variable_order (_conflict_lits);
    
  DBG0( cout << "**Add Clause " << (added_cl>>1) << " : "<<clause(added_cl) << endl; );

  for (i=0; i< clause(added_cl).num_lits(); ++i) {
    int vid = clause(added_cl).literal(i).var_index();
    int sign =clause(added_cl).literal(i).var_sign();
    assert (variable(vid).value() != UNKNOWN);
    assert (literal_value(clause(added_cl).literal(i)) == 0);
    int dl = variable(vid).dlevel();

    // Ashish: need to change the following comparison with dlevel()
    // for symbranching
    if (!_params.symmetry.enable) {
      if ( dl < dlevel()) {
	if (dl > back_dl) 
	  back_dl = dl;
      } else {
	assert (unit_lit == -1);
	unit_lit = vid + vid + sign; 
      }
    } else {
      // symmetry enabled; don't rely on dlevel()
      // note, however, that zchaff sets back_dl to 0 when the learnt
      //   clause is a unit clause; maintain that
      if (dl > max_dl) {
	back_dl = max_dl;
	max_dl = dl;
	unit_lit = vid + vid + sign;
      }
      else if (dl > back_dl)
	back_dl = dl;
      if (_params.learning.enable) {
	++ _num_marked;
	variable(vid).set_marked();
	involved_lits.push_back(vid+vid+sign);
      }
    }
  }

  if (_params.symmetry.enable && _params.learning.enable) {
    // Compute decision literals that need to go in the symconflict clause
    // Also find the lowest dlevel var in this symdecision that can be flipped
    int sym_dl;
    if (dlevel() >= branchlevel_next()) {
      DBG0(cout << "This symbranch has already been backtracked" << endl;)
      sym_dl = branchlevel_next();
    }
    else
      sym_dl = _branchlevel_stack.top();
    CBranchInfo & bi = _branchinfo_vector[sym_dl];
    int flip_dl = sym_dl + bi.branch_size();   // initialize
    set<ConsIdx> visited_ante;  // Pbcons may otherwise be visited more
                                // than once if it implies multiple variables
    for (i=0; i<involved_lits.size(); ++i) {
      int vid = involved_lits[i] >> 1;
      int vid_dl = variable(vid).dlevel();
      ConsIdx ante_cons = variable(vid).get_antecedent();
      if (ante_cons != NULL_CONS) {
	// it's not a decision variable -> process antecedent
	if (_params.pb.enable) {
	  if (visited_ante.find(ante_cons) != visited_ante.end())
	    continue; // already processed; 
	              // this check is unnecessary for Boolean clauses
	  else
	    visited_ante.insert(ante_cons);
	}
	CConstraint & c = constraint(ante_cons);
	for (unsigned j=0; j< c.num_lits(); ++j) {
	  int lit = c.literal(j).s_var();
	  if (variable(lit>>1).is_marked()==false) {
	    if (_params.pb.enable && literal_value(c.literal(j)) != 0)
	      // Store only those vars that are assigned false
	      // For Boolean clauses, all lits are false
	      continue;
	    ++ _num_marked;
	    variable(lit>>1).set_marked();
	    involved_lits.push_back(lit);
	  }
	}
      }
      else if (vid_dl < sym_dl)
	// it's a decision variable of a previous symbranch
	bi.add_symconflict_lit(involved_lits[i], vid_dl);
      else
	// it's a decision variable of the current symbranch
	if (vid_dl < flip_dl && vid_dl > sym_dl+bi.branch_point-1)
	  flip_dl = vid_dl;
    }

    visited_ante.clear();

    // update branch_info so that redundant branches are skipped
    if (flip_dl - sym_dl + 1 > bi.branch_point + 1) {
      bi.branch_idx_to_flip = flip_dl - sym_dl + 1;
      DBG1(cout << "Setting branch_idx_to_flip = " 
           << bi.branch_idx_to_flip << endl;);
    }

    // clear all marks
    for (vector<int>::iterator itr = involved_lits.begin(); itr != involved_lits.end(); ++itr) {
      int svar = *itr;
      assert (variable(svar>>1).is_marked());
      -- _num_marked;
      variable(svar>>1).clear_marked();
    }
    assert (_num_marked == 0);
  }

  // backtrack
  if (!_params.symmetry.enable)
    back_track(back_dl+1);
  else {
    if (back_dl+1 >= branchlevel_next()) {
      DBG1(cout << "The corresponding symbranch has already been backtracked\n";)
      back_track(back_dl+1);
    }
    else {
      int sym_dl = _branchlevel_stack.top();
      CBranchInfo & bi = _branchinfo_vector[sym_dl];
      if (!_params.fastbackjumping.enable || bi.fast_backjumped)
        back_track_sym();
      else {
        // perform fast backjumping here if called for
        bi.fast_backjumped = true;
        back_track_sym();
          
        // Jump to the max of back_dl and max_symconflict_dl (called
        // fast_back_dl). Note that bi.symconflict_lits must be
        // updated before computing fast_back_dl because it may be
        // affected by the previous backtrack. This is done by the
        // above call to back_track_sym

        int fast_back_dl = back_dl;
        if (bi.max_symconflict_dl > fast_back_dl)
            fast_back_dl = bi.max_symconflict_dl;

        if (!_branchlevel_stack.empty() &&
            fast_back_dl+1 < _branchlevel_stack.top()) {    // fast backjumping is non-trivial
          // keep the symbranch containing fast_back_dl
          // replace the next symbranch with the current one
          CBranchInfo bi_save = bi;
          DBG0(cout << "Fast backjumping to (roughly) level " << fast_back_dl << endl;);

          while (!_branchlevel_stack.empty() &&    // will empty if fast_back_dl=0
                 _branchlevel_stack.top() > fast_back_dl)
            back_track_sym(true, true, true);

          CBranchInfo & bi_new = _branchinfo_vector[branchlevel_next()];
          bi_new = bi_save;
          // make sure the symindex set splits at the boundaries of
          // bi_new still exist; if not, re-do the symbranch with larger
          // branch_size
          if (bi_new.branch_idx_start > 0) {  
            // there is potential of symmetry expansion
            int lo_split = bi_new.branch_idx_start - 1;
            int hi_split = bi_new.branch_idx_start + bi_new.branch_size() - 1;
            if ((lo_split > 0 && _symindex_sets.count(-(lo_split+1)) == 0)
                || (_symindex_sets.count(-(hi_split+1)) == 0)) {
              DBG0(cout << "Fastbackjumping: Branch size increased from the original " << bi_new.branch_size()
                   << " for symbranch" << endl
                   << "\toriginally at sym_dl " << sym_dl
                   << " and now at sym_dl " << branchlevel_next() << "."<< endl
                   << "\tWill re-do the symbranch with larger branch size." 
                   << endl << endl;)
                bi_new.redo_symbranch = true;
            }
          }
        }
      }
    }
  }

  queue_implication(unit_lit, added_cl, back_dl);
	    
  for (i=1; i< _conflicts.size(); ++i) //after resolve the first conflict, others must also be resolved
    assert(!is_conflicting(_conflicts[i]));
  _conflicts.clear();

  while (_conflict_lits.size()) {
    int svar = _conflict_lits.back();
    _conflict_lits.pop_back();
    CVariable & var = variable(svar >> 1);
    assert (var.new_cl_phase() == (unsigned)(svar & 0x1));
    -- _num_in_new_cl;
    var.set_new_cl_phase(UNKNOWN);
  }
  assert (_num_in_new_cl == 0);
  DBG0(cout << "Conflict Analasis: conflict at level: " << back_dl + 1;
       cout << "  Assertion Clause is " << (added_cl>>1) << endl; );

  if (!_params.symmetry.enable)
    return back_dl;
  else
    return _branchlevel_last - 1;
}

int CSolver::conflict_analysis_grasp (void)
{
    fatal(_POSITION_, "Not implemented in this version");
    return 0;
}

int CSolver::conflict_analysis_decisions_only (void)
{
  //For experiment purpose only, don't use it 
  unsigned int i;
  ConsIdx cl = _conflicts[0];
  if (cl & 0x1) {
    cerr << "ERROR 14: Can't handle PB constraint in conflict_analysis_decisions_only" << endl;
    exit(14);
  }

  DBG0(cout <<"Conflict clause: " << cl << " : " << clause(cl) << endl;);
  int gflag = clause(cl).gflag();

  vector<int> involved_lits;
  for (i=0; i < clause(cl).num_lits(); ++i) {
    int svar = clause(cl).literal(i).s_var();
    ++ _num_marked;
    variable(svar>>1).set_marked();
    involved_lits.push_back(svar);
  }
  for (i=0; i<involved_lits.size(); ++i) {
    int vid = involved_lits[i] >> 1;
    if (variable(vid).get_antecedent() != NULL_CONS) {
      //it's not a decision variable
      CClause & c = clause(variable(vid).get_antecedent());
      gflag |= c.gflag();
      for (unsigned j=0; j< c.num_lits(); ++j) {
	int lit = c.literal(j).s_var();
	if (variable(lit>>1).is_marked()==false) {
	  ++ _num_marked;
	  variable(lit>>1).set_marked();
	  involved_lits.push_back(lit);
	}
      }
    }
  }
  for (vector<int>::iterator itr = involved_lits.begin(); 
       itr != involved_lits.end(); ++itr) {
    int svar = *itr;
    if (variable(svar>>1).get_antecedent() == NULL_CONS) {
      ++ _num_in_new_cl;
      variable(svar>>1).set_new_cl_phase(svar & 0x1);
      _conflict_lits.push_back(svar);
    }
    assert (variable(svar>>1).is_marked());
    -- _num_marked;
    variable(svar>>1).clear_marked();
  }
  assert (_num_marked == 0);

  return finish_add_conf_clause(gflag);
}

// int CSolver::conflict_analysis_allUIP (void)
// {
//     //For experiment purpose only, don't use it 
//     vector<int> real_conflict_lits;

//     ClauseIdx cl = _conflicts[0];
//     DBG0(cout <<"Conflict clause: " << cl << " : " << clause(cl) << endl;);
//     mark_vars_at_current_dlevel (cl, -1 /*var*/);
//     unsigned gflag = clause(cl).gflag();
//     vector <int> & assignments = *_assignment_stack[dlevel()]; //current dl must be the conflict cl.
//     for (int i=assignments.size()-1; i >= 0; --i) { //now add conflict lits, and unassign vars
// 	int assigned = assignments[i];
// 	if (variable(assigned>>1).is_marked()) {  
// 	    //this variable is involved in the conflict clause or it's antecedent
// 	    variable(assigned>>1).clear_marked();
// 	    -- _num_marked; 
// 	    ClauseIdx ante_cl = variables()[assigned>>1].get_antecedent();
// 	    if ( _num_marked == 0 ) { 
// 				//the first UIP encountered
// 		real_conflict_lits.push_back(assigned ^ 0x1); //this is the flipped lit
// 		break;
// 	    }
// 	    else {
// 		assert ( ante_cl != NULL_CONS );
// 		gflag |= clause(ante_cl).gflag();
// 		mark_vars_at_current_dlevel(ante_cl, assigned>>1/*var*/);
// 	    }
// 	}
//     }
//     //at this point, real_conflict_lits has the firstUIP, _conflict_lits has the vars involved in other dlevel
//     vector<int> current;
//     for (int dl = dlevel() - 1; dl > 0; --dl) {
// 	DBG1(cout << "Processing Dlevel " << dl << endl; );
// 	current.swap(_conflict_lits);
// 	_conflict_lits.clear();
// 	assert (_num_marked == 0);
// 	mark_vars_of_dl(current, dl);
// 	if (_num_marked == 0) //conflict does not involve this level
// 	    continue;
// 	vector <int> & assignments = *_assignment_stack[dl]; 
// 	for (int i=assignments.size()-1; i >= 0; --i) { //now add conflict lits, and unassign vars
// 	    int assigned = assignments[i];
// 	    if (variable(assigned>>1).is_marked()) {  
// 				//this variable is involved in the conflict clause or it's antecedent
// 		variable(assigned>>1).clear_marked();
// 		-- _num_marked; 
// 		ClauseIdx ante_cl = variables()[assigned>>1].get_antecedent();
// 		if ( _num_marked == 0 ) { 
// 		    //UIP encountered
// 		    real_conflict_lits.push_back(assigned^0x1);  // add this assignment
// 		    DBG1(cout << "Put " << (assigned&0x1 ? "+":"-") << (assigned >> 1) << "in conflict clause" << endl;);
// 		    break; //go on to the next level
// 		}
// 		else {
// 		    assert ( ante_cl != NULL_CONS );
// 		    gflag |= clause(ante_cl).gflag();
// 		    mark_vars_at_level(ante_cl, assigned>>1/*var*/, dl);
// 		}
// 	    }	    
// 	}
//     }
//     assert (_num_marked == 0);
//     for (unsigned j=0; j< real_conflict_lits.size(); ++j ) {
// 	int svar = real_conflict_lits[j];
// 	assert (variable(svar>>1).new_cl_phase() == UNKNOWN);
// 	variable(svar>>1).set_new_cl_phase(svar & 0x1);
// 	_conflict_lits.push_back(svar);
// 	++ _num_in_new_cl;
//     }
//     return finish_add_conf_clause(gflag);
// }

// int CSolver::conflict_analysis_lastUIP(void)
// {
//     //For experiment purpose only, don't use it 
//     ClauseIdx cl = _conflicts[0];
//     DBG0(cout <<"Conflict clause: " << cl << " : " << clause(cl) << endl;);
//     mark_vars_at_current_dlevel (cl, -1 /*var*/);
//     unsigned gflag = clause(cl).gflag();
//     vector <int> & assignments = *_assignment_stack[dlevel()]; //current dl must be the conflict cl.
//     for (int i=assignments.size()-1; i >= 0; --i) { //now add conflict lits, and unassign vars
// 	int assigned = assignments[i];
// 	if (variable(assigned>>1).is_marked()) {  
// 	    //this variable is involved in the conflict clause or it's antecedent
// 	    variable(assigned>>1).clear_marked();
// 	    -- _num_marked; 
// 	    ClauseIdx ante_cl = variables()[assigned>>1].get_antecedent();
// 	    if ( ante_cl == NULL_CONS ) { //last UIP is the decision varialbe
// 		assert (i == 0);	     
// 		assert (_num_marked == 0); //last UIP is still a UIP
// 		assert (variable(assigned>>1).new_cl_phase() == UNKNOWN);
// 		_conflict_lits.push_back(assigned^0x1);  // add this assignment's reverse, e.g. UIP
// 		++ _num_in_new_cl;
// 		variable(assigned>>1).set_new_cl_phase((assigned^0x1)&0x1);
// 	    }
// 	    else {
// 		assert ( ante_cl != NULL_CONS );
// 		gflag |= clause(ante_cl).gflag();
// 		mark_vars_at_current_dlevel(ante_cl, assigned>>1/*var*/);
// 	    }
// 	}
//     }
//     return finish_add_conf_clause(gflag);
// }

int CSolver::conflict_analysis_firstUIP (void) 
{
  assert (dlevel() > 0);
  set<ConsIdx> visited_ante;  // Ashish: pbcons may otherwise be visited more
                              // than once if it implies multiple variables
  ConsIdx idx = _conflicts[0];
  DBG0(cout <<"Conflict constraint: " << ((idx & 0x01)?"PB ":"CL ")
       << (idx>>1) << " : " << constraint(idx) << endl;);
  mark_vars_at_current_dlevel (idx, -1 /*var*/);
  unsigned gflag = constraint(idx).gflag();
  vector <int> & assignments = *_assignment_stack[dlevel()]; //current dl must be the conflict cl.
  for (int i=assignments.size()-1; i >= 0; --i) { //now add conflict lits, and unassign vars
    int assigned = assignments[i];
    if (variable(assigned>>1).is_marked()) {  
      //this variable is involved in the conflict clause or its antecedent
      variable(assigned>>1).clear_marked();
      -- _num_marked; 
      ConsIdx ante_cons = variable(assigned>>1).get_antecedent();
      if ( _num_marked == 0 ) {
        //the first UIP encountered, conclude add clause
        assert (variable(assigned>>1).new_cl_phase() == UNKNOWN);
        _conflict_lits.push_back(assigned^0x1);  // add this assignment's reverse, e.g. UIP
        ++ _num_in_new_cl;
        variable(assigned>>1).set_new_cl_phase((assigned^0x1)&0x1);
        break; //if do this, only one clause will be added.
      }
      else {
        assert ( ante_cons != NULL_CONS );
        if (!_params.pb.enable || visited_ante.find(ante_cons) == visited_ante.end()) {
          gflag |= constraint(ante_cons).gflag();
          mark_vars_at_current_dlevel(ante_cons, assigned>>1/*var*/);
          if (_params.pb.enable)
	    visited_ante.insert(ante_cons);
        }
      }
    }
  }
  visited_ante.clear();
  return finish_add_conf_clause(gflag);
}		

// int CSolver::conflict_analysis_mincut (void) 
// {
//     assert (1 && "Not available, need a mincut library PLED");
//     return 0;
//      ClauseIdx cl = _conflicts[0];
//      int asrt = -1;
//      DBG0(cout <<"Conflict clause: " << cl << " : " << clause(cl) << endl;);
//      mark_vars_at_current_dlevel (cl, -1 /*var*/);
//      unsigned gflag = clause(cl).gflag();
//      ClauseIdx added_cl;
//      vector <int> & assignments = *_assignment_stack[dlevel()]; //current dl must be the conflict cl.
//      for (int i=assignments.size()-1; i >= 0; --i) { //now add conflict lits, and unassign vars
//  	int assigned = assignments[i];
//  	if (variable(assigned>>1).is_marked()) {  
//  	    //this variable is involved in the conflict clause or it's antecedent
//  	    variable(assigned>>1).clear_marked();
//  	    -- _num_marked; 
//  	    ClauseIdx ante_cl = variables()[assigned>>1].get_antecedent();
//  	    if ( _num_marked == 0 ) { 
//  		//the first UIP encountered, conclude add clause
//  		assert (variable(assigned>>1).new_cl_phase() == UNKNOWN);
//  		asrt = assigned^0x1; //this is the assert literal
//  		break; //if do this, only one clause will be added.
//  	    }
//  	    else {
//  		assert ( ante_cl != NULL_CONS );
//  		gflag |= clause(ante_cl).gflag();
//  		mark_vars_at_current_dlevel(ante_cl, assigned>>1/*var*/);
//  	    }
//  	}
//      }
//      assert (asrt != -1);
//      if (_conflict_lits.size() == 0) {
//  	_conflict_lits.push_back(asrt);
//  	variable(asrt>>1).set_new_cl_phase( asrt & 0x1);
//  	++ _num_in_new_cl;
//  	return finish_add_conf_clause(gflag);
//      }
//      int nvtxs ;
//      int nedges;
//      vector<int> edge_from;
//      vector<int> edge_to;
//      vector<DOUBLE> ewts;
//      vector<int> part;
//      vector<int> st;
//      vector<int> var_map;
//      var_map.resize(variables().size());
//      int source_size = _conflict_lits.size();
//      int sink_size = 0;
//      for (int i=0; i< _conflict_lits.size(); ++i) {
//  	int svar = _conflict_lits[i];
//  	st.push_back(svar  );
//  	variable(svar>>1).set_marked();
//  	variable(svar>>1).set_new_cl_phase(UNKNOWN);
//  	var_map[svar>>1] = i;
//      }
//      _num_in_new_cl -= _conflict_lits.size();
//      assert (_num_in_new_cl == 0);
//      assert (_num_marked == 0);
//      _num_marked += source_size;
//      _conflict_lits.clear();
//      for (int i=0; i< st.size(); ++i) {
//  	int vid = st[i]>>1;
//  	int sign = st[i] & 0x1;
//  	part.resize(part.size() + 2); //add two vetex
//  	if (i < source_size) 
//  	    part[i+i] = 0;
//  	else 
//  	    part[i+i] = 3;
//  	//split the node
//  	edge_from.push_back(i+i);
//  	edge_to.push_back(i+i+1);
//  	ewts.push_back(1);
//  	//add other edges
//  	ClauseIdx ante = variable(vid).get_antecedent();
//  	if (ante != NULL_CONS && clause(ante).num_lits() > 1) {
//  	    part[i+i+1] = 3;
//  	    CClause & cl = clause(ante);
//  	    for (int j=0; j< cl.num_lits(); ++j) {
//  		int svar = cl.literal(j).s_var();
//  		if (variable(svar>>1).is_marked() == false) {
//  		    variable(svar>>1).set_marked();
//  		    ++ _num_marked;
//  		    var_map[svar>>1] = st.size();
//  		    st.push_back(svar);
//  		}
//  		if (svar != st[i]) {
//  		    edge_from.push_back(i+i+1);
//  		    edge_to.push_back(var_map[svar>>1] << 1);
//  		    ewts.push_back(variables().size());
//  		}
//  	    }
//  	}
//  	else { //decision variable
//  	    part[i+i+1] = 1; //it's a sink
//  	    ++ sink_size;
//  	}
//      }
//      assert (st.size() == part.size()/2);
//      assert (edge_from.size() == edge_to.size());
//      int flow = (int) PLED_FindMaxFlow(part.size(), edge_from.size(),
//  				      edge_from.begin(), edge_to.begin(), 
//  				      ewts.begin(), part.begin());
//      assert (flow <= source_size);
//      for (int i=0; i< edge_from.size(); ++i) {
//  	if (part[edge_from[i]] == 0 && part[edge_to[i]] == 1) {
//  	    assert (edge_from[i] == edge_to[i] -1 );
//  	    int svar = st[edge_from[i]>>1];
//  	    _conflict_lits.push_back(svar);
//  	    variable(svar>>1).set_new_cl_phase(svar & 0x1);
//  	    ++ _num_in_new_cl;
//  	    -- flow;
//  	}
//      }
//      assert (flow == 0);
//      _conflict_lits.push_back(asrt);
//      variable(asrt>>1).set_new_cl_phase( asrt & 0x1);
//      ++ _num_in_new_cl;
//      for (int i=0; i< st.size(); ++i) {
//  	assert (variable(st[i]>>1).is_marked());
//  	variable(st[i]>>1).clear_marked();
//      -- _num_marked;
//      }
//      assert (_num_marked == 0);
//      return finish_add_conf_clause(gflag); 
// }		
 

void CSolver::print_cls(ostream & os)
{
    for (vector<CClause>::iterator itr = clauses().begin(); itr != clauses().end(); ++itr) {
	CClause & cl = *itr;
	if (cl.status() == DELETED_CONS)
	    continue;
	if (cl.status() == ORIGINAL_CONS)
	    os <<"0 ";
	else {
	    assert (cl.status() == CONFLICT_CONS);
	    os << "A ";
	}
	for (int i=1; i< 33; ++i) 
	    os <<( cl.gid(i) ? 1:0);
	os << "\t";
	for (unsigned j=0; j< cl.num_lits(); ++j)
	    os << (cl.literal(j).var_sign() ? "-":"") << cl.literal(j).var_index() << " ";
	os <<"0" <<  endl;
	    
    }
}


int CSolver::mem_usage(void) 
{
    int mem_dbase = CDatabase::mem_usage();
    int mem_assignment = 0;
    for (int i=0; i< _stats.max_dlevel; ++i)
	mem_assignment += _assignment_stack[i]->capacity() * sizeof(int);
    mem_assignment += sizeof(vector<int>)* _assignment_stack.size();
    return mem_dbase + mem_assignment;
}

void CSolver::clean_up_dbase(void)
{
    assert (dlevel() == 0);
    assert (_stats.is_solver_started == false);

    int mem_before = mem_usage();
    //1. remove all the learned clauses
    int old_unrelevance = _params.cls_deletion.max_unrelevance;
    int old_minlits = _params.cls_deletion.min_num_lits;
    int old_cls_size = _params.cls_deletion.max_conf_cls_size;
    _params.cls_deletion.max_unrelevance = 0;
    _params.cls_deletion.min_num_lits = 0;
    _params.cls_deletion.max_conf_cls_size = 0;

    delete_unrelevant_clauses();

    _params.cls_deletion.max_unrelevance = old_unrelevance;
    _params.cls_deletion.min_num_lits = old_minlits;
    _params.cls_deletion.max_conf_cls_size = old_cls_size;
    //2. free up the mem for the vectors if possible
    for (unsigned i=0; i< variables().size(); ++i) {
	for (int j=0; j< 2; ++j ) { //both phase
	    vector<CLitPoolElement *> watched;
	    vector<CLitPoolElement *> & old_watched = variable(i).watched(j);
	    watched.reserve(old_watched.size());
	    for (vector<CLitPoolElement *>::iterator itr = old_watched.begin(); 
		 itr != old_watched.end(); ++itr)
		watched.push_back(*itr);
	    //because watched is a temp mem allocation, it will get deleted
	    //out of the scope, but by swap it with the old_watched, the contents are reserved.
	    old_watched.swap(watched); 
#ifdef KEEP_LIT_CLAUSES
	    vector<int> lits_cls;
	    vector<int> & old_lits_cls = variable(i).lit_clause(j);
	    lits_cls.reserve(old_lits_cls.size());
	    for (vector<int>::iterator itr = old_lits_cls.begin(); itr != old_lits_cls().end(); ++itr)
		lits_cls.push_back(*itr);
	    old_lits_cls.swap(lits_cls); 
#endif
	}
    } 
    int mem_after = mem_usage();
    if (_params.verbosity > 0) 
	cout << "Database Cleaned, releasing (approximately) " 
	     << mem_before - mem_after << " Bytes" << endl;
}

void CSolver::update_var_score(void) {
    unsigned int i,sz;
    for (i=1, sz=variables().size(); i<sz; ++i) {
	_ordered_vars[i-1].first = & variable(i);
	_ordered_vars[i-1].second = variable(i).score();
    }
    DBG2(
	cout << "Before Update ";
	for (unsigned i=0; i< _ordered_vars.size(); ++i)
	cout << (_ordered_vars[i].first - & variables()[0]) << "(" << _ordered_vars[i].first->score() << ") ";
	cout << endl;
	);
    ::stable_sort(_ordered_vars.begin(), _ordered_vars.end(), cmp_var_stat);
    DBG2(
	cout << "After Update";
	for (unsigned i=0; i< _ordered_vars.size(); ++i)
	cout << (_ordered_vars[i].first - & variables()[0]) << "(" << _ordered_vars[i].first->score() << ") ";
	cout << endl;
	);
    for (i=0, sz= _ordered_vars.size(); i<sz; ++i) 
	_ordered_vars[i].first->set_var_score_pos(i);
    _max_score_pos = 0;
}
	
void CSolver::restart (void){
  _stats.num_restarts ++;

  if (_params.symmetry.enable) {
    _symindex_sets = _symindex_sets_orig;
    _symindex_sets_reverse = _symindex_sets_reverse_orig;
    _symindex_distinct = _symindex_sets.size() - 1;
    _branchinfo_vector[branchlevel_next()].clear();
    while (!_branchlevel_stack.empty()) {
      _branchinfo_vector[_branchlevel_stack.top()].clear();
      _branchlevel_stack.pop();
    }
    _branchlevel_last = -1;
    _triv_conflict_dl = -1;
  }

  if (_params.verbosity > 1 ) 
    cout << "Restarting ... " << endl;
  if (dlevel() > 1) {
    //clean up the original var_stats.
    /*	for (int i=1; i<variables().size(); ++i) {
	variable(i).score(0) = variable(i).lits_count(0);
	variable(i).score(1) = variable(i).lits_count(1);
	}
	update_var_score(); */
    back_track(1); //backtrack to level 0. restart.	
  }
}

// void CSolver::set_up_resolve(ClauseIdx conf_cl, set<CVariable *, cmp_var_assgn_pos> & conf_lits)
// {
//     assert (_conflict_lits.empty());
//     assert (conf_lits.empty());
//     CClause & cl = clause(conf_cl);
//     for (CLitPoolElement * lit = cl.first_lit(); lit->is_literal(); ++lit) 
// 	conf_lits.insert(& variable(lit->var_index()));
// }

// unsigned CSolver::resolve_one_lit(set<CVariable *, cmp_var_assgn_pos> & conf_lits)
// {
//     assert (!conf_lits.empty());
//     CVariable * vptr = (* conf_lits.begin());
//     unsigned vid = vptr - &variable(0);
//     conf_lits.erase(conf_lits.begin());
//     int cl_idx = vptr->antecedent();
//     assert (cl_idx != NULL_CONS);
//     CClause & cl = clause(cl_idx);
//     for (CLitPoolElement * lit = cl.first_lit(); lit->is_literal(); ++lit) {
// 	if (lit->var_index() != vid) {
// 	    conf_lits.insert(&variable(lit->var_index()));
// 	}
//     }
//     return cl.gflag();
// }

// bool CSolver::is_uip_reached(set<CVariable *, cmp_var_assgn_pos> & conf_lits)
// {
//     assert (!conf_lits.empty());
//     if (conf_lits.size() == 1) return true;
//     set<CVariable *, cmp_var_assgn_pos>::iterator itr = conf_lits.begin();
//     CVariable * v1 = *itr;
//     ++itr;
//     CVariable * v2 = *itr;
//     assert (v1->dlevel() >= v2->dlevel());
//     if (v1->dlevel() != v2->dlevel())
// 	return true;
//     return false;
// }

// int CSolver::conflict_analysis_firstUIP_resolve_based(void)
// {
//     assert (dlevel() > 0);
//     ClauseIdx cl = _conflicts[0];
//     unsigned gflag = clause(cl).gflag();
//     set<CVariable *, cmp_var_assgn_pos> conf_lits;
//     set_up_resolve(cl, conf_lits);
//     while(!is_uip_reached(conf_lits))
// 	gflag |= resolve_one_lit(conf_lits);
//     vector<int> cls_lits;
//     for (set<CVariable *, cmp_var_assgn_pos>::iterator itr = conf_lits.begin();
// 	 itr != conf_lits.end(); ++itr) {
// 	int vid = (*itr) - &variable(0);
// 	int sign = variable(vid).value();
// 	cls_lits.push_back(vid + vid + sign);
//     }
//     ClauseIdx added_cl = add_conflict_clause(&(*cls_lits.begin()), cls_lits.size(), gflag);
//     if (added_cl < 0 ) { //memory out.
// 	_stats.is_mem_out = true;
// 	_conflicts.clear();
// 	assert (_implication_queue.empty());
// 	assert (_implication_pqueue.empty());    // Ashish
// 	return 1; 
//     }
//     adjust_variable_order (cls_lits);
//     int back_dl = 0;
//     int unit_lit = cls_lits[0];
//     if (cls_lits.size() > 1) {
// 	back_dl = variable(cls_lits[1]>>1).dlevel();
//     }
//     DBG0( cout << "**Add Clause " <<added_cl<< " : "<<clause(added_cl) << endl; );
//     back_track(back_dl + 1);
//     queue_implication(unit_lit, added_cl, back_dl);
	    
//     for (unsigned i=1; i< _conflicts.size(); ++i) //after resolve the first conflict, others must also be resolved
// 	assert(!is_conflicting(_conflicts[i]));
//     _conflicts.clear();
//     DBG0(cout << "Conflict Analasis: conflict at level: " << back_dl + 1;
// 	 cout << "  Assertion Clause is " << added_cl<< endl; );
//     return back_dl;
// }

//this function can be called within a solving process. i.e. not after
//solve() terminate

int CSolver::add_clause_incr(vector<int> & lits, int gid)
{
    unsigned gflag;
    if (gid == PERMANENT_GID ) 
	gflag = 0;
    else if (gid == VOLATILE_GID)
	gflag = ~0x0;
    else {
	assert (gid <= WORD_WIDTH && gid > 0);
	gflag = (1 << (gid- 1));
    }

    int cl = add_clause(lits, gflag);
    if (cl < 0) return -1;
    clause(cl).set_status(ORIGINAL_CONS);

    if(clause(cl).num_lits()==1){
        int var_idx=clause(cl).literal(0).var_index();
        if(literal_value(clause(cl).literal(0))==0&&variable(var_idx).dlevel()==0){
            back_track(0);
            if (preprocess()==CONFLICT)
            	_stats.outcome = UNSATISFIABLE;
        }
        else{
            if(dlevel()>0)
                back_track(1);      
            queue_implication(clause(cl).literal(0).s_var(),cl,0);
        }
        return cl;
    }
    int max_level = 0;
    int max_level2 = 0;
    int unit_lit = 0;
    int unknown_count = 0;
    int num_sat = 0;
    int sat_dlevel,max_lit;
    int i, sz;
    bool already_sat=false;

    for (i=0, sz= clause(cl).num_lits(); i<sz; ++i) {
        int var_idx=lits[i]>>1;
        int value=variable(var_idx).value();
        if (value == UNKNOWN) continue;
        if (variable(var_idx).dlevel() == 0 
                && variable(var_idx).antecedent() == -1 
                && literal_value(clause(cl).literal(i)) == 0) {
            back_track(0);
            if (preprocess()==CONFLICT)
            	_stats.outcome = UNSATISFIABLE;
            return cl;
        }
    }
    for (i=0, sz= clause(cl).num_lits(); unknown_count<2&& i<sz; ++i) {
        int var_idx=lits[i]>>1;
        int value=variable(var_idx).value();
        if(value==UNKNOWN){
            unit_lit=clause(cl).literal(i).s_var();
            unknown_count++;
        }
        else{
            int dl=variable(var_idx).dlevel();
            if(dl >= max_level){
                max_level2=max_level;
                max_level=dl;
                max_lit=clause(cl).literal(i).s_var();
            }
            else if(dl > max_level2)
                max_level2=dl;
            int effect=literal_value(clause(cl).literal(i));
            if(effect==1){
                already_sat=true;
                num_sat++;
                sat_dlevel=dl;
            }
        }
    }
    if(unknown_count==0){
        if(already_sat){
            if(num_sat==1&&sat_dlevel==max_level&&max_level>max_level2){
                back_track(max_level2+1);
                queue_implication(max_lit,cl,max_level2);
            }
        }
        else{
            assert(is_conflicting(cl));
            if(max_level>max_level2){
                back_track(max_level2+1);
                queue_implication(max_lit,cl,max_level2);
            }
            else{
                back_track(max_level);
                if (max_level==0 && preprocess()==CONFLICT)
                    _stats.outcome = UNSATISFIABLE;
            }
        }
    }
    else if(unknown_count==1){
        if(!already_sat){
            if(max_level<dlevel())
                back_track(max_level+1);
            queue_implication(unit_lit, cl, max_level);
        }
    }
    return cl;
}
