/*********************************************************************
 Copyright 2000-2003, Princeton University.  All rights reserved. 
 By using this software the USER indicates that he or she has read, 
 understood and will comply with the following:

 --- Princeton University hereby grants USER nonexclusive permission 
 to use, copy and/or modify this software for internal, noncommercial,
 research purposes only. Any distribution, including commercial sale 
 or license, of this software, copies of the software, its associated 
 documentation and/or modifications of either is strictly prohibited 
 without the prior consent of Princeton University.  Title to copyright
 to this software and its associated documentation shall at all times 
 remain with Princeton University.  Appropriate copyright notice shall 
 be placed on all software copies, and a complete copy of this notice 
 shall be included in all copies of the associated documentation.  
 No right is  granted to use in advertising, publicity or otherwise 
 any trademark,  service mark, or the name of Princeton University. 


 --- This software and any associated documentation is provided "as is" 

 PRINCETON UNIVERSITY MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS 
 OR IMPLIED, INCLUDING THOSE OF MERCHANTABILITY OR FITNESS FOR A 
 PARTICULAR PURPOSE, OR THAT  USE OF THE SOFTWARE, MODIFICATIONS, OR 
 ASSOCIATED DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, 
 TRADEMARKS OR OTHER INTELLECTUAL PROPERTY RIGHTS OF A THIRD PARTY.  

 Princeton University shall not be liable under any circumstances for 
 any direct, indirect, special, incidental, or consequential damages 
 with respect to any claim by USER or any third party on account of 
 or arising from the use, or inability to use, this software or its 
 associated documentation, even if Princeton University has been advised
 of the possibility of those damages.
*********************************************************************/

/*********************************************************************
 ADDENDUM: Modifications and additions to the original software,
 zChaff version 2003.11.04, have been made by Ashish Sabharwal (the
 "author") to produce this new software, SymChaff version 2005.03.

 --- This new software and any associated documentation is also
     provided "as is"

 THE AUTHOR OR THE UNIVERSITY OF WASHINGTON MAKE NO REPRESENTATIONS OR
 WARRANTIES, EXPRESS OR IMPLIED, INCLUDING THOSE OF MERCHANTABILITY OR
 FITNESS FOR A PARTICULAR PURPOSE, OR THAT USE OF THE SOFTWARE,
 MODIFICATIONS, OR ASSOCIATED DOCUMENTATION WILL NOT INFRINGE ANY
 PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER INTELLECTUAL PROPERTY RIGHTS
 OF A THIRD PARTY.

 The author or the University of Washington shall not be liable under
 any circumstances for any direct, indirect, special, incidental, or
 consequential damages with respect to any claim by USER or any third
 party on account of or arising from the use of, or inability to use,
 this software or its associated documentation, even if the author or
 the University of Washington have been advised of the possibility of
 those damages.
*********************************************************************/

#ifndef __ZCHAFF_DATABASE__
#define __ZCHAFF_DATABASE__

#include <set>
#include <ext/hash_map>
#include <queue>
#include "zchaff_base.h"

using __gnu_cxx::hash_map;

#define STARTUP_LIT_POOL_SIZE 0x8000

/**Struct**********************************************************************

  Synopsis    [Definition of the statistics of clause database]

  Description []

  SeeAlso     [CDatabase]

******************************************************************************/

struct CDatabaseStats {
  unsigned 	mem_used_up_counts;
  bool 	mem_used_up;
  unsigned 	init_num_clauses;
  unsigned 	init_num_literals;
  unsigned 	num_added_clauses;
  long64 	num_added_literals;

  // Ashish: pb related
  unsigned 	init_num_pbcons;
  unsigned 	num_added_pbcons;
  unsigned      num_deleted_pbcons;

  // Ashish: symmetry related
  unsigned 	num_added_symclauses;
  long64 	num_added_symliterals;

  unsigned 	num_deleted_clauses;
  long64 	num_deleted_literals;
  unsigned 	num_compact;
  unsigned	num_enlarge;
};

/**Struct**********************************************************************

  Synopsis    [Definition of the parameters of clause database]

  Description []

  SeeAlso     [CDatabase]

******************************************************************************/


struct CDatabaseParams {
    int 	mem_limit;
};


/**Class**********************************************************************

  Synopsis    [Definition of clause database ]

  Description [Clause Database is the place where the information of the 
               SAT problem are stored. it is a parent class of CSolver ]

  SeeAlso     [CSolver]

******************************************************************************/
class CDatabase
{
 protected:
  CDatabaseStats 	_stats;

  CDatabaseParams	_params;

  unsigned 		_allocated_gid;		//the gids that have already been allocated

  //for efficiency, the memeory management of lit pool is done by the solver
  CLitPoolElement * _lit_pool_start;		//the begin of the lit vector
  CLitPoolElement * _lit_pool_finish;		//the tail of the used lit vector
  CLitPoolElement * _lit_pool_end_storage;   	//the storage end of lit vector


  vector<CVariable> 	_variables; 		//note: first element is not used

  vector<CClause> 	_clauses;

  queue<ConsIdx> 	_unused_clause_idx;	

  // Ashish: pb related
  vector<CPBcons> 	_pbcons;
  queue<ConsIdx> 	_unused_pbcons_idx;

  // Note: cons_idx follows the scheme of the id() in CClause and 
  // CPBcons in that the first bit determines what kind of constraint
  // it is. However, as constraints are deleted and new ones created,
  // _clauses[5] may not be the same as the clause with _id=5.


  //////////////////////////////
  // Ashish: symmetry related //

  // vector of variable classes
  vector<CVarclass>               _varclasses;

  // symindex sets are maintained as a multi-set. When splitting, their endpoints
  //   are added to the multiset. When unsplitting, these endpoints are deleted.

  /*
   A symindex set is represented by its starting point. Note that in
     the .sym input file, the symindex sets are represented by their
     high ending point instead, which is somewhat more natural
     (e.g. 20 packages, 30 trucks, etc. would be represented as
     (20,30)). For internal computation with the STL library, it's a
     little easier to manipulate instead (the negative of) the
     starting points (e.g. 20 packages, 30 trucks would be represented
     as (-1,-21,-51), where 51 is the implicit starting point of the
     hypothetical next symindex set). All this is taken care of by
     add_symindex_high() and related methods. _symindex_sets_reverse
     actually stores the other format as well, namely, (0,20,50).
  */
  multiset<CSymindexSet>          _symindex_sets, _symindex_sets_reverse;

  // stores the original symindex sets; recall that these sets change over time
  multiset<CSymindexSet>          _symindex_sets_orig, _symindex_sets_reverse_orig;

  // _highest and _distinct are used to check whether symmetry has been exhausted
  CSymindexSet                    _symindex_highest;
  int                             _symindex_distinct;

  // stores a map from each variable to the varclass it belongs to and
  // the symindices it is indexed by
  hash_map<int,CSymindexMapping>  _symindex_map;

  // for specified ordering (.ord file) of the indexes of varclasses and
  // the priority groups of varclasses; guides the branching process
  vector<vector<int> >            _varclass_idx_order;
  vector<vector<int> >            _varclass_priority_groups;

  // done with symmetry related declarations //
  /////////////////////////////////////////////

 protected:
  //constructors & destructors
  CDatabase() ;

  ~CDatabase();

  void init_stats(void) {
    _stats.mem_used_up_counts 	= 0;
    _stats.mem_used_up 		= false;
    _stats.init_num_clauses 	= num_clauses();
    _stats.init_num_pbcons 	= num_pbcons();  // Ashish
    _stats.init_num_literals 	= num_literals();
    _stats.num_deleted_clauses 	= 0;
    _stats.num_deleted_pbcons 	= 0;             // Ashish
    _stats.num_deleted_literals = 0;
    _stats.num_enlarge		= 0;
    _stats.num_compact		= 0;
  }
  //lit pool naming convention follows STL Vector
  CLitPoolElement * lit_pool_begin(void) ;

  CLitPoolElement * lit_pool_end(void) ;

  void lit_pool_incr_size( int size) ;

  void lit_pool_push_back(int value) ; 

  int lit_pool_size(void) ;

  int lit_pool_free_space(void) ;

  double lit_pool_utilization(void);

  CLitPoolElement & lit_pool(int i) ;

  //functions on lit_pool
  void output_lit_pool_stats(void);

  bool enlarge_lit_pool(void);	//when allocated memeory runs out, do a reallocation

  void compact_lit_pool(void);	//garbage collection

  unsigned literal_value (CLitPoolElement l) { //note: it will return 0 or 1 or other , 
    //here "other" may not equal UNKNOWN
    return (variable(l.var_index()).value() ^ l.var_sign()); 
  }

  unsigned svar_value (int svar) {		//note: it will return 0 or 1 or other , 
    //here "other" may not equal UNKNOWN
    return (variable(svar>>1).value() ^ (svar&0x1)); 
  }
  // constraint properties
  void mark_cons_deleted(CConstraint & cons);

  // clause properties
  int find_unit_literal(ConsIdx cl);	//if not unit clause, return 0.

  bool is_conflicting(ConsIdx cl);

  bool is_unit (ConsIdx cl);

  bool is_satisfied(ConsIdx cl);

  //others
  ConsIdx get_free_clause_idx(void);
  ConsIdx get_free_pbcons_idx(void);  // Ashish

  ConsIdx add_clause (vector<int> & lits, int gflag = 0, 
                      bool is_symclause = false);
  ConsIdx add_symclause (vector<int> & lits);   // Ashish
  ConsIdx add_pbcons (vector<int> & lits, vector<int> & coeffs, int rhs); // Ashish


 public:
  //member access function
  vector<CVariable>& variables(void) { 
    return _variables; 
  }

  CVariable & variable(int idx) { 
    return _variables[idx]; 
  }

  CClause & clause(ConsIdx idx) {
    assert ((idx & 0x01) == 0);
    return _clauses[idx >> 1]; 
  }

  vector<CClause> & clauses(void) {
    return _clauses;
  }

  // Ashish: pb related
  CPBcons & pbcons(ConsIdx idx) { 
    assert ((idx & 0x01) == 1);
    return _pbcons[idx >> 1]; 
  }

  vector<CPBcons> & pbcons(void) {
    return _pbcons;
  }

  // Ashish: CConstraint related
  CConstraint & constraint(ConsIdx idx) {
    return (idx & 0x01) 
      ? static_cast<CConstraint &> (_pbcons[idx>>1])
      : static_cast<CConstraint &> (_clauses[idx>>1]);
  }

  // Ashish: symmetry related
  vector<CVarclass> & varclasses(void) {
    return _varclasses;
  }
  CVarclass & varclass(int idx) {
    return _varclasses[idx];
  }
  multiset<CSymindexSet> & symindex_sets(void) {
    return _symindex_sets;
  }
  CSymindexSet idx_symindex_set(int idx) {
    return - * _symindex_sets.lower_bound(-idx);
  }
  CSymindexSet size_symindex_set(int idx) {
    return * _symindex_sets_reverse.lower_bound(idx) - idx_symindex_set(idx) + 1;
  }
  CSymindexSet var_symindex_set(int var, int i) {
    CSymindexMapping* sm = var_findsymindex(var);
    if (!sm) return 0;
    return idx_symindex_set(sm->indices[i]);
  }
  hash_map<int,CSymindexMapping> & symindex_map(void) {
    return _symindex_map;
  }
  CSymindexMapping* var_findsymindex(int var) {
    hash_map<int,CSymindexMapping>::iterator itr = _symindex_map.find(var);
    return (itr == _symindex_map.end()) ? NULL : & itr->second;
  }
  vector<int> & varclass_idx_order(int varclass_idx) {
    return _varclass_idx_order[varclass_idx];
  }
  int num_varclass_priority_groups() {
    return _varclass_priority_groups.size();
  }
  vector<int> & varclass_priority_group(int i) {
    return _varclass_priority_groups[i];
  }

  CDatabaseStats & stats(void) {
    return _stats;
  }

  void set_mem_limit(int n) {
    _params.mem_limit = n;
  }

  //clause group management
  int alloc_gid (void); 

  void free_gid (int gid);

  int get_volatile_gid(void) {return -1;}

  int	get_permanent_gid(void) { return 0; }

  bool is_gid_allocated(int gid);

  int merge_cons_group(int g1, int g2);

  //some stats
  unsigned & init_num_clauses()	        { return _stats.init_num_clauses; }

  unsigned & init_num_literals ()	{ return _stats.init_num_literals; }

  unsigned & num_added_clauses ()	{ return _stats.num_added_clauses; }

  long64  & num_added_literals() 	{ return _stats.num_added_literals; }

  // Ashish
  unsigned & init_num_pbcons()	        { return _stats.init_num_pbcons; }
  unsigned & num_added_pbcons ()	{ return _stats.num_added_pbcons; }
  unsigned & num_deleted_pbcons() 	{ return _stats.num_deleted_pbcons; }

  // Ashish
  unsigned & num_added_symclauses ()	{ return _stats.num_added_symclauses; }
  long64  & num_added_symliterals() 	{ return _stats.num_added_symliterals; }

  unsigned & num_deleted_clauses() 	{ return _stats.num_deleted_clauses; }

  long64 & num_deleted_literals() 	{ return _stats.num_deleted_literals; }

  unsigned num_variables(void) 	{ return variables().size() - 1; }

  unsigned num_clauses(void) 		{ return _clauses.size() - _unused_clause_idx.size();}

  unsigned num_literals(void)		{ return _stats.num_added_literals - _stats.num_deleted_literals;}

  // Ashish
  unsigned num_pbcons(void) 		{ return _pbcons.size() - _unused_pbcons_idx.size();}

  // Ashish: symmetry related
  unsigned num_sym()                    { return symindex_sets().size(); }
  unsigned num_varclasses()             { return varclasses().size() - 1; }

  unsigned num_mem_compacts(void)	{ return _stats.num_compact; }

  unsigned num_mem_enlarges(void) 	{ return _stats.num_enlarge; }

  //functions 
  unsigned estimate_mem_usage (void);

  unsigned mem_usage(void) ;

  void set_variable_number(int n) { 
    variables().resize(n + 1) ; 
  }
  int add_variable(void) {
    variables().resize( variables().size() + 1);
    return variables().size() - 1;
  }

  // Ashish: symmetry related
  void set_sym_number(int a) {
    // Don't really need to set anything here; not a vector
    // Insert the first element into the set, namely -1
    _symindex_sets.insert((CSymindexSet) -1);
  }
  void set_varclass_number(int n_varclass) {
    _varclasses.resize(n_varclass+1);   // varclasses start at 1, not 0
  }

  void add_symindex_high(int sym_high) {
    _symindex_sets.insert((CSymindexSet) -(sym_high+1));
    _symindex_sets_reverse.insert((CSymindexSet) sym_high);

    if (_symindex_sets.count((CSymindexSet) -(sym_high+1)) == 1)
      ++ _symindex_distinct;
    DBG0(if (symmetry_exhausted()) cout << "\n===> Symmetry exhausted! <===\n\n")
  }

  void set_symindex_highest(int sym_highest) {
    _symindex_highest = sym_highest;
    _symindex_distinct = _symindex_sets.size() - 1;
    _symindex_sets_orig = _symindex_sets;
    _symindex_sets_reverse_orig = _symindex_sets_reverse;
  }

  void del_symindex_high(int sym_high);

  void add_varclass(int vc_idx, int* index, int num_indices) {
    _varclasses[vc_idx] = CVarclass(index, num_indices);
  }

  bool symmetry_exhausted() {
    return (_symindex_highest == _symindex_distinct);
  }

  bool symmetry_enabled() {
    return (_symindex_sets.size() > 0);
  }

  void store_varclass_idx_order(int vc_idx, int* index, int num_indices) {
    static bool first_time = true;
    if (first_time) {
      // set the size of _varclass_idx_order
      _varclass_idx_order.resize(_varclasses.size());
      first_time = false;
    }
    _varclass_idx_order[vc_idx] = vector<int>(index, index+num_indices);
  }


  void store_varclass_priority_group(vector<int> & priority_group) {
    _varclass_priority_groups.push_back(vector<int>(priority_group));
  }

  void compute_varclass_details();

  int locate_var(CSymindexMapping & cm) {
    return _varclasses[cm.varclass_idx].locate_var(cm.indices);
  }

  void add_symindex_mapping(int var_idx, int varclass_idx, int* index, unsigned num_indices);


  //dump functions
  void detail_dump_cl(ConsIdx cl_idx, ostream & os = cout);

  void detail_dump_pb(ConsIdx pb_idx, ostream & os = cout);

  void dump(ostream & os = cout);

  void dump_symmetry_info(ostream & os = cout);

  void dump_varclasses(ostream & os = cout);

  void dump_symindex_sets(ostream & os = cout);

  void dump_symindex_map(ostream & os = cout);

  friend ostream & operator << ( ostream & os, CDatabase & db) { 
    db.dump(os); 
    return os;
  }
};

#endif











